import {FileTransport} from "./filetransport";
import {API, isElectron, Log, Logger, Utils} from "@solid/libs";
import { TransportInitParameters } from "./msemediaplayer";
import { StreamParams } from "@solid/types";

const isUseFileTransport = false && isElectron();

enum WebSocketMediaTransportClose {
  MANUAL = 4000
}

type WebSocketMediaTransportParameters = {
  logger?: Logger | Console,
  debug?: boolean,
  debugData?: boolean
}

export class WebSocketMediaTransport {
  _fileTransport: FileTransport = new FileTransport("ws");

  _logger: Logger | Console = console;

  _api: API = new API();

  /**
   * web socket URI
   */
  _wsURI: string = "";
  _websocket: WebSocket | null = null;

  _requestLog: string[] = [];
  _requestLogMaxSize = 50;

  _debug = false;
  _debugData = false;

  _parameters: TransportInitParameters = {
    isUseFileTransport: false,
    isLocal: false
  };

  _stream: StreamParams = {
    mime: "",
    width: 0,
    height: 0,
  };

  _heartBeatInterval: number = 0;
  _lastReceivedTime: number = 0;
  _currentChunk: Uint8Array | null = null;
  _endEvent: {time: number | null, event: string} | null = null
  _endKeyTime: number = 0;
  _doPlay: boolean = false;
  _initMIME: boolean = false;
  _isChunkSplitted: boolean = false;
  _skipChunks: boolean = false;
  _chunkPosition: number = 0;
  _chunkNumber?: number;
  _chunkNumber2?: number;
  _chunkSize?: number;
  _chunkMD5?: string;

  _onClose: (e: CloseEvent) => void = () => {};

  constructor(parameters: WebSocketMediaTransportParameters = {}) {
    if (typeof parameters.logger !== "undefined") {
      this._logger = parameters.logger;
    }

    if (typeof parameters.debug !== "undefined") {
      this._debug = parameters.debug;
    }
  }

  async init(parameters: TransportInitParameters): Promise<void> {
    this.setParameters(parameters);

    if (this.isOpen()) {
      return;
    }

    isUseFileTransport && await this._fileTransport.init();

    const {url, streamingOverages: {type, message}} = await this._getURL(!!parameters.isLocal);

    if (type === "warning") {
      Log.warning(message);
    } else
    if (type === "alert") {
      Log.error(message);
    } else
    if (type === "block") {
      throw {message};
    }

    let isResolve = false;
    let isReject = false;

    this._wsURI = url;

    return new Promise((resolve, reject) => {
      const protocols = parameters.isLocal ? "media.videonext.com" : "stream";
      this._websocket = new WebSocket(this._wsURI, protocols);

      this._websocket.binaryType = "arraybuffer";
      this._websocket.addEventListener("open", () => {
        this._logger.log("onwsopen");

        this._parameters.open && this._parameters.open();

        // this._heartbeat();

        if (!isResolve) {
          isResolve = true;
          resolve();
        }
      });
      this._onClose = (e) => {
        let message = Utils.webSocketCloseCodeMessage(e.code);
        this._logger.log(`onwsclose: [${e.code}] ${message}`);

        if (!isResolve && !isReject) {
          isReject = true;
          reject({code: e.code, message: message});
        }

        this.close(e.code !== WebSocketMediaTransportClose.MANUAL);
      };
      this._websocket.addEventListener("close", this._onClose);
      this._websocket.addEventListener("message", (message) => {
        isUseFileTransport && this._fileTransport.message(message);

        if (message.data instanceof ArrayBuffer) {
          this._debugData && this._logger.log("chunk> " + message.data.byteLength + " bytes");
        } else {
          this._debugData && this._logger.log("onmessage> ", message.data);
        }

        this._parameters.message && this._parameters.message(message);
      });
      this._websocket.addEventListener("error", () => {
        this._logger.log('onwserror');

        this._parameters.error && this._parameters.error();
      });
    });
  };

  setParameters(parameters: TransportInitParameters): void {
    this._parameters = Object.assign(this._parameters, parameters);
  };

  async _getURL(isLocal: boolean): Promise<{url: string, streamingOverages: {type: string, message: string}}> {
    try {
      const {url, streamingOverages} = await this._api.getMediaURL({isLocal: isLocal, cameraid: this._parameters.obj});
      return {url: url, streamingOverages};
    }
    catch (e: any) {
      throw new Error(e.message);
    }
  };

  send(data: object) {
    const dataJSON = JSON.stringify(data);

    this._logger.log('send cloud>', dataJSON);

    isUseFileTransport && this._fileTransport.send(data);

    this._requestLog.push(dataJSON);
    if (this._requestLog.length > this._requestLogMaxSize) {
      this._requestLog.shift();
    }

    if (!this._websocket) {
      this._logger.warn("this._websocket is undefined");
      return;
    }

    // TODO: check for websocket still in connecting state
    this._websocket.send(dataJSON);
  };

  getLog(): string[] {
    return this._requestLog;
  };

  isOpen(): boolean {
    return !!(this._websocket && this._websocket.readyState === WebSocket.OPEN);
  };

  close(isCallback = true): void {
    this._logger.log("WebSocketMediaTransport close");

    clearInterval(this._heartBeatInterval);

    isUseFileTransport && this._fileTransport.close(isCallback);

    if (this._websocket
        && this._websocket.readyState !== WebSocket.CLOSING && this._websocket.readyState !== WebSocket.CLOSED) {
      this._websocket.close(WebSocketMediaTransportClose.MANUAL, "manual close");
      this._websocket && this._websocket.removeEventListener("close", this._onClose);
    }

    if (this._websocket
        && (this._websocket.readyState === WebSocket.CLOSING || this._websocket.readyState === WebSocket.CLOSED)) {
      isCallback && this._parameters.close && this._parameters.close(0);
      this._websocket = null;
    }
  };

  _heartbeat(): void {
    this._heartBeatInterval = window.setInterval(() => this.send({}), 5000);
  };
}
