import {Logger, Utils} from "@solid/libs";
import {Chunk, Media, MSEMediaPlayer} from "./msemediaplayer";
import { MSEMediaPlayerError } from "@solid/types";
import "rvfc-polyfill";

type Stat = {
  // corruptedVideoFrames?: number,
  creationTime?: number,
  droppedVideoFrames: number,
  droppedVideoFramesPercent: number,
  totalVideoFrames: number,
  live_latency: number,
  buffer_latency: number | undefined,
  buffer_media_queue_length: number,
  buffer_player: string,
  currentTime: number,
  // duration: this._player.duration,
  playbackRate: number,
  paused: boolean,
  // ended: this._player.ended,
  // seeking : this._player.seeking,
  readyState: number,
  chunkList: Chunk[],
  timeBetweenKeyChunks: number
}

type MSEPlayerParameters = {
  logger?: Logger | Console
};

type MSEPlayerInitParameters = {
  node?: HTMLVideoElement,
  mediaplayer?: MSEMediaPlayer,
  // video: HTMLVideoElement | null,
  frame?: ((time: number, isLast?: boolean) => void) | null,
  fps?: ((fps: number) => void) | null,
  error?: ((code: number, message: string) => void) | null
};

export class MSEPlayer {
  _logger: Logger | Console = console;

  _player: HTMLVideoElement | null = null;

  _mediaSource: MediaSource | null = null;
  _sourceBuffer: SourceBuffer | null = null;

  _mediaQueue: Media[] = [];

  _mediaplayer: MSEMediaPlayer | null = null;

  _videoFrame: VideoFrame | null = null;

  _callPlay = false;
  _callPause = false;

  _canAppend = false;

  _lastRenderedTime = 0;

  _forceRedraw = false;

  _lastTimestampList: number[] = [];

  _parameters: MSEPlayerInitParameters = {};

  _isMissedChunk = false;
  _isKeyChunk = false;
  _isSearchingBufferForPlay = false;
  _skipUntilNextKeyChunk = false;

  _timeBetweenKeyChunks = 0; // sec
  _reinitLagTimeout = 15.0; // sec

  _lastChunkList: Chunk[] = [];
  _maxLength = 240;
  _lastTime = 0;
  _prevKeyChunkTime = 0;

  _sendOnLagging = false;

  _mime = "";

  _resolve?: () => void;

  constructor(parameters: MSEPlayerParameters = {}) {
    if (!window.MediaSource) {
      throw {message: 'MediaSource API is not available'};
    }

    if (typeof parameters.logger !== "undefined") {
      this._logger = parameters.logger;
    }
  }

  _addStatChunk({key = false, skip = false}) {
    this._lastChunkList.push({key, skip});
    if (this._lastChunkList.length > this._maxLength) {
      this._lastChunkList.shift();
    }
  };

  async init(parameters: MSEPlayerInitParameters) {
    this._player = parameters.node ?? document.createElement("video");
    this._player.volume = parameters.mediaplayer?._muted ? 0 : 1;
    this._mediaplayer = parameters.mediaplayer ?? null;

    this._mediaQueue = [];

    this._parameters = {
      ...this._parameters,
      frame: parameters.frame ?? null,
      fps: parameters.fps ?? null,
      error: parameters.error ?? null
    }

    this._videoFrame = new VideoFrame({
      video: this._player,
      frame: this._parameters.frame ?? null,
      fps: this._parameters.fps ?? null,
      logger: this._logger
    });

    this._initPlayer();
  };

  _initPlayer() {
    if (!this._player) {
      return;
    }

    this._player.addEventListener('waiting', () => {
      if (!this._player) {
        return;
      }

      const bufferedLength = this._mediaSource && this._mediaSource.sourceBuffers.length > 0 ? this._mediaSource.sourceBuffers[0].buffered.length : 0;
      this._logger.log('player waiting', `player.readyState == ${this._player.readyState} sourceBuffer.buffered.length == ${bufferedLength}`);

      // TODO: find more correct way for sending last event
      if (this._player.readyState === 2) {
        // call frame callback for processing EOS, EOA or EOC event when frame buffer is empty
        this._videoFrame && this._videoFrame.parameters.frame && this._videoFrame.parameters.frame(this._videoFrame.time, true);
      }

      return;

      // TODO: correct or remove this logic for processing gaps in sourceBuffer
      /*
      if (this._mediaSource.sourceBuffers.length > 0) {
        const sourceBuffer = this._mediaSource.sourceBuffers[0];

        if (this._player.readyState !== 2 || sourceBuffer.buffered.length <= 1) {
          // this._logger.log(`player.readyState == ${this._player.readyState} sourceBuffer.buffered.length == ${sourceBuffer.buffered.length}`);
          return;
        }

        const findCurrentIndex = (time: number): number => {
          let index = null;
          for (let i = 0; i < sourceBuffer.buffered.length; i++) {
            const start = sourceBuffer.buffered.start(i);
            const end = sourceBuffer.buffered.end(i);

            if (time <= start) {
              index = i - 1;
              break;
            }
            if (start <= time && time <= end) {
              index = i;
              break;
            }

            if (i === sourceBuffer.buffered.length - 1) {
              index = sourceBuffer.buffered.length - 1;
            }
          }
          return index;
        };
        const goToNext = async (index: number): Promise<void> => {
          if (!this._isSearchingBufferForPlay) {
            return;
          }

          const sourceBuffer = this._mediaSource.sourceBuffers[0];

          this._logger.log(`goToNext ${index} ${sourceBuffer.buffered.length}`);

          if (this._player.readyState !== 2 || sourceBuffer.buffered.length <= 1) {
            this._logger.log(`player.readyState == ${this._player.readyState} sourceBuffer.buffered.length == ${sourceBuffer.buffered.length}`);

            this._isSearchingBufferForPlay = false;
            return;
          }

          if (index < sourceBuffer.buffered.length) {
            let start = sourceBuffer.buffered.start(index);
            this._player.currentTime = start;

            this._logger.log(`currentTime => ${start}`);

            this._isSearchingBufferForPlay = false;
            return;
          } else {
            this._logger.warn(`not yet ${index} >= ${sourceBuffer.buffered.length}`);
          }

          await Utils.wait(100);

          goToNext(index);
        };

        const currentTime = this._player.currentTime;
        const currentBufferedSegment = findCurrentIndex(currentTime);
        if (currentBufferedSegment === sourceBuffer.buffered.length - 1) {
          this._logger.log(`${currentBufferedSegment} === ${sourceBuffer.buffered.length} - 1`);
          return;
        }

        this._isSearchingBufferForPlay = true;

        const nextBufferedSegment = currentBufferedSegment + 1;
        goToNext(nextBufferedSegment);
      }
      */
    });

    /*
    this._player.addEventListener('error', (e) => {
      let code = this._player.error.code;
      let message = this._player.error.message;
      this._logger.error(code, message);
      this._mediaplayer.stop(0, message);
    });
    */

    /*
    this._player.addEventListener('seeking', () => {this._logger.log('seeking');});
    this._player.addEventListener('seeked', () => {this._logger.log('seeked');});

    this._player.addEventListener('loadstart', () => {this._logger.log('player loadstart');});
    this._player.addEventListener('durationchange', () => {this._logger.log('player durationchange', this._player.duration);});
    this._player.addEventListener('progress', () => {
      const endVal = this._player.seekable && this._player.seekable.length ? this._player.seekable.end(this._player.seekable.length - 1) : 0;
            const length = (100 / (this._player.duration || 1) * endVal) + '%';

      this._logger.log('player progress', length, this._player.duration, endVal);
    });
    this._player.addEventListener('timeupdate', () => {this._logger.log('player timeupdate');});

    this._player.addEventListener('suspend', () => {this._logger.log('player suspend');});
    this._player.addEventListener('abort', () => {this._logger.log('player abort');});
    this._player.addEventListener('emptied', () => {this._logger.log('player emptied');});
    this._player.addEventListener('stalled', () => {this._logger.log('player stalled');});

    this._player.addEventListener('loadedmetadata', () => {this._logger.log('player loadedmetadata');});
    this._player.addEventListener('loadeddata', () => {this._logger.log('player loadeddata');});
    this._player.addEventListener('playing', () => {this._logger.log('player playing');});
    this._player.addEventListener('canplay', () => {this._logger.log('player canplay');});
    // event is basically useless — because Safari doesn’t fire it at all,
    // while Opera and Chrome fire it immediately after the "canplay" event, even when it’s yet to preload so much as a quarter of a second!
    // Only Firefox and IE10 appear to implement this event correctly.
    this._player.addEventListener('canplaythrough', () => {this._logger.log('player canplaythrough');});
    */

    const isSafari = /^((?!chrome|android).)*safari/i.test(navigator.userAgent);

    // pause\play stream after browser pause\play video component (ex: background\active tab in chrome)
    this._player.addEventListener("play", async (e) => {
      this._logger.log('player play event');

      // if 'play' was called by browser
      if (!this._callPlay) {
        this._logger.log("browser call play");

        // resume stream after Chrome 'play' video from background tab
        if (!isSafari) {
          await this._mediaplayer?.play();
        }
      } else {
        this._callPlay = false;
      }
    });
    this._player.addEventListener("pause", async (e) => {
      this._logger.log('player pause event');

      if (!this._player) {
        return;
      }

      this._isSearchingBufferForPlay = false;

      // if 'pause' was called by browser
      if (!this._callPause) {
        this._logger.log("browser call pause");

        // stop stream after Chrome 'pause' video on background tab
        if (!isSafari) {
          this._mediaplayer?.pause();
        }

        // safari 'pause' video in some cases (empty buffer)
        if (isSafari) {
          await this._player.play();
        }
      } else {
        this._callPause = false;
      }
    });
  };

  initMediaSource(mime: string): Promise<void> {
    if (!this._mediaplayer) {
      this._logger.error("initMediaSource: this._mediaplayer is undefined");
      return new Promise<void>((resolve, reject) => reject());
    }

    if (!this._videoFrame) {
      this._logger.error("initMediaSource: this._videoFramer is undefined");
      return new Promise<void>((resolve, reject) => reject());
    }

    this._sendOnLagging = false;
    this._prevKeyChunkTime = 0;

    this._videoFrame.reinit();

    return new Promise((resolve, reject) => {
      this._resolve = resolve;

      if (!this._player) {
        this._logger.error("initMediaSource: this._player is undefined");
        reject();
        return;
      }

      //if (mime !== this._mime) {
        this._logger.log(`initMediaSource ${mime}`);
        this._mime = mime;
      //if (!this._mediaSource) {
        this._mediaSource = new MediaSource();

        this._mediaSource.addEventListener('sourceopen', async () => {
          this._logger.log('onsourceopen');

          try {
            await this._addSourceBuffer(mime);
            this._resolve && this._resolve();
          }
          catch (e) {
            reject(e);
          }
        }, false);

        /*
        this._mediaSource.addEventListener('sourceended', function () {this._logger.log('onsourceended');}, false);
        this._mediaSource.addEventListener('sourceclose', function () {this._logger.log('onsourceclose');}, false);
        */
        /*
        this._mediaSource.addEventListener('updatestart', function () {this._logger.log('updatestart');}, false);
        this._mediaSource.addEventListener('update', function () {this._logger.log('update');}, false);
        this._mediaSource.addEventListener('updateend', function () {this._logger.log('updateend');}, false);
        this._mediaSource.addEventListener('addsourcebuffer', function () {this._logger.log('addsourcebuffer');}, false);
        this._mediaSource.addEventListener('removesourcebuffer', function () {this._logger.log('removesourcebuffer');}, false);
        */
        this._mediaSource.addEventListener('error', () => {this._logger.error('mediaSource error');}, false);
        this._mediaSource.addEventListener('abort', () => {this._logger.error('mediaSource abort');}, false);
      //}

      this._player.src = URL.createObjectURL(this._mediaSource);
    });
  };

  addMedia(media: Media, isFirst: boolean): void {
    // this._logger.log("addMedia", isFirst);

    if (!this._mediaplayer) {
      this._logger.error("addMedia: this._mediaplayer is undefined");
      return;
    }

    if (!this._videoFrame) {
      this._logger.error("addMedia: this._videoFrame is undefined");
      return;
    }

    this._isMissedChunk = false;

    this._lastTime = this._lastTimestampList.length > 0 ? this._lastTimestampList[this._lastTimestampList.length - 1] : 0;

    // sync Live video by skipping chunks
    if (!this._mediaplayer.isArchive()) {
      /*
      let bufferLatency = 0;
      if (this._mediaSource.sourceBuffers.length > 0) {
        const sourceBuffer = this._mediaSource.sourceBuffers[0];
        const buffered = sourceBuffer.buffered;

        if (buffered.length > 0) {
          const end = buffered.end(buffered.length - 1);

          bufferLatency = end - this._player.currentTime;
        }
      }
      */
      // const isLaggingForSkipChunks = bufferLatency > this._skipChunkLagTimeout;

      const liveLatency = this._lastTime > 0 && this._videoFrame.time > 0 ? (this._lastTime - this._videoFrame.time) / 1000 : 0;

      // reinit player, if playback is stuck
      const isLaggingForReinit = liveLatency > this._reinitLagTimeout;
      if (isLaggingForReinit) {
        if (!this._sendOnLagging) {
          this._sendOnLagging = true;
          this._logger.log(`lagging for ${this._reinitLagTimeout} sec, reinit stream`);
          this._mediaplayer.pause();
          Utils.wait(1000)
            .then(() => this._mediaplayer?.play());
        }
      }

      // skip all chunks from current to next keyframe to prevent creating gaps in sourceBuffers (processing not trivial)
      const isLaggingForSkipChunk = this._timeBetweenKeyChunks !== 0 ? liveLatency > this._timeBetweenKeyChunks * 2 : false;
      if (isLaggingForSkipChunk && !this._skipUntilNextKeyChunk) {
        this._skipUntilNextKeyChunk = true;
      }
      if (this._isKeyChunk) {
        this._skipUntilNextKeyChunk = false;
        this._addStatChunk({key: true});
      }
      if (!isFirst && !this._isKeyChunk && this._skipUntilNextKeyChunk) {
        // TODO: correct skip chunk logic
        this._logger.log("skip", !isFirst, !this._isKeyChunk, this._skipUntilNextKeyChunk, JSON.stringify(this._lastTimestampList));
        this._addStatChunk({skip: true});
        return;
      } else
      if (!this._isKeyChunk) {
        this._addStatChunk({key: false});
      }
    } else {
      this._addStatChunk({key: this._isKeyChunk});
    }

    // add timestamps
    for (let i = 0; i < this._lastTimestampList.length; i++) {
      this._videoFrame.addTime([this._lastTimestampList[i]]);
    }

    // add media chunk
    this._mediaQueue.push(media);

    if (this._canAppend || isFirst) {
      this._canAppend = false;
      this.appendNextMediaSegment();
    }
  };

  addTime(timestampList: number[], isKey: boolean = false): void {
    if (this._isMissedChunk) {
      this._logger.log("missed chunk");
    }
    this._lastTimestampList = timestampList.slice();
    this._isMissedChunk = true;
    this._isKeyChunk = isKey;

    if (isKey) {
      const keyChunkTime = timestampList[timestampList.length - 1];

      this._timeBetweenKeyChunks = this._prevKeyChunkTime !== 0 ? (keyChunkTime - this._prevKeyChunkTime) / 1000 : 0;

      this._prevKeyChunkTime = keyChunkTime;
    }
  };

  appendNextMediaSegment(): boolean {
    /*
    if (!this._mediaSource.sourceBuffers[0].updating && this._mediaSource.readyState === 'open')
    {
      mediaSource.endOfStream();
    }
    */

    if (!this._mediaSource) {
      this._logger.error("appendNextMediaSegment: this._mediaSource is undefined");
      return false;
    }

    if (!this._player) {
      this._logger.error("appendNextMediaSegment: this._player is undefined");
      return false;
    }

    if (!this._mediaplayer) {
      this._logger.error("appendNextMediaSegment: this._mediaplayer is undefined");
      return false;
    }

    if (this._mediaSource.readyState === "closed") {
      // TODO: check
      this._logger.error("appendNextMediaSegment closed");
      return false;
    }

    // Make sure the previous append is not still pending.
    if (this._mediaSource.sourceBuffers.length === 0) {
      // TODO: check
      this._logger.error("appendNextMediaSegment sourceBuffers.length === 0");
      return false;
    }

    let sourceBuffer = this._mediaSource.sourceBuffers[0];

    if (sourceBuffer.updating) {
      // this._logger.warn("appendNextMediaSegment sourceBuffer.updating");
      return false;
    }

    let mediaSegment: Uint8Array | null = null;
    if (this._mediaQueue.length > 0) {
      let media = this._mediaQueue.shift();
      mediaSegment = media!.data;
    }

    if (!mediaSegment) {
      // this._logger.warn("appendNextMediaSegment !mediaSegment");
      this._canAppend = true;
      return false;
    }

    // NOTE: If mediaSource.readyState == “ended”, this appendBuffer() call will
    // cause mediaSource.readyState to transition to "open". The web application
    // should be prepared to handle multiple “sourceopen” events.
    try {
      sourceBuffer.appendBuffer(mediaSegment);
    }
    catch (e: any) {
      this._logger.error(e.message);

      // TODO: maybe be need to move this check to player error event handler
      if (this._player.error) {
        this._logger.error(this._player.error.message);
      }

      if (e.name === "InvalidStateError") {
        this._logger.error("InvalidStateError");
        let message = this._player.error ? this._player.error.message : e.message;
        this._mediaplayer.stop(0, message);
      } else
      if (e.name === 'QuotaExceededError') {
        this._logger.error("QuotaExceededError");
        let removeStart = sourceBuffer.buffered.start(0);
        let removeEnd = this._player.currentTime - 1;
        if (sourceBuffer.buffered.length > 0) {
          removeEnd = sourceBuffer.buffered.end(sourceBuffer.buffered.length - 1);
        }

        if (removeEnd > removeStart) {
          this._logger.log("Remove> ", removeStart, removeEnd);
          sourceBuffer.remove(removeStart, removeEnd);
        }
      } else {
        this._parameters.error && this._parameters.error(MSEMediaPlayerError.MSE_PLAYER_SOURCE_BUFFER_ERROR, e.message);
      }
    }

    return true;
  };

  _addSourceBuffer(mime: string): Promise<void> {
    this._logger.log('Add source buffer ' + mime);

    this._lastChunkList = [];

    return new Promise((resolve, reject) => {
      if (!this._mediaSource) {
        this._logger.error("_addSourceBuffer: this._mediaSource is undefined");
        reject();
        return;
      }

      if (this._mediaSource.sourceBuffers.length > 0) {
        this._logger.error("mediaSource.sourceBuffers not empty");
      }

      this._sourceBuffer = this._mediaSource.addSourceBuffer(mime);
      this._sourceBuffer.mode = mime.includes("mp4a") ? "segments" : "sequence";

      // Append the initialization segment.
      /*var firstAppendHandler(e) {
        this._logger.log("firstAppendHandler");
        // var sourceBuffer = e.target;
        this._sourceBuffer.removeEventListener('updateend', firstAppendHandler);

        // Append some initial media data.
        this.appendNextMediaSegment();
      };*/

      this._sourceBuffer.addEventListener('updatestart', (e) => {
        // this._logger.log("sourceBuffer updatestart");
      }, false);
      this._sourceBuffer.addEventListener('update', (e) => {
        // this._logger.log("sourceBuffer update");
      }, false);
      this._sourceBuffer.addEventListener('updateend', (e) => {
        // this._logger.log("sourceBuffer updateend");
        this.appendNextMediaSegment();
      }, false);
      this._sourceBuffer.addEventListener('error', (e) => {
        // TODO: check this logic
        this._logger.error('sourceBuffer error');
        this._parameters.error && this._parameters.error(MSEMediaPlayerError.MSE_PLAYER_SOURCE_BUFFER_ERROR, 'sourceBuffer error');
        reject({message: 'sourceBuffer error'});
      }, false);
      this._sourceBuffer.addEventListener('abort', (e) => {
        // TODO: check this logic
        this._logger.error('sourceBuffer abort');
        this._parameters.error && this._parameters.error(MSEMediaPlayerError.MSE_PLAYER_SOURCE_BUFFER_ERROR, 'sourceBuffer abort');
        reject({message: 'sourceBuffer abort'});
      }, false);

      resolve();
    });
  }

  muteAudio(muted: boolean): void {
    if (!this._player) return;
    this._player.volume = muted ? 0 : 1;
  }

  async play(playbackRate = 1.0) {
    this._logger.log("play");

    if (!this._videoFrame) {
      const message = "play: this._videoFrame is undefined";
      this._logger.error(message);
      throw new Error(message);
    }

    if (!this._player) {
      const message = "play: this._player is undefined";
      this._logger.error(message);
      throw new Error(message);
    }

    this._callPlay = true;

    this._videoFrame.listen();

    // this._player.playbackRate = playbackRate;
    await this._player.play();
  }

  setSpeed(speed: number) {
    this._logger.log("setSpeed");

    if (!this._player) {
      const message = "forward: this._player is undefined";
      this._logger.error(message);
      throw new Error(message);
    }

    if (!this._videoFrame) {
      const message = "forward: this._videoFrame is undefined";
      this._logger.error(message);
      throw new Error(message);
    }

    this._player.playbackRate = speed;
  }

  pause() {
    if (!this._videoFrame) {
      this._logger.error("pause: this._videoFrame is undefined");
      return;
    }

    if (!this._player) {
      this._logger.error("pause: this._player is undefined");
      return;
    }

    this._logger.log("pause HTMLMediaElement", this._player.readyState);

    this._videoFrame.stopListen();

    this._callPause = true;
    this._player.pause();

    this._mediaQueue = [];
  }

  stop() {
    this._logger.log("stop");

    // TODO: implement stop command
    this.pause();

    this._mediaQueue = [];
  }

  setStartTime(startTime: number) {
    if (!this._videoFrame) {
      this._logger.error("setStartTime: this._videoFrame is undefined");
      return;
    }

    // this.startTime = startTime;
    this._videoFrame.setStartTime(startTime);
  }

  render(source: CanvasImageSource, target: CanvasDrawImage): HTMLVideoElement | null {
    if (!this._player) {
      this._logger.error("render: this._player is undefined");
      return null;
    }

    if (!this._forceRedraw && this._lastRenderedTime === this._player.currentTime) { return this._player; }
    this._forceRedraw = false;
    this._lastRenderedTime = this._player.currentTime;

    if (this._player.readyState > 1) {
      // target.drawImage(this._player, 0, 0);
    } else {
      this._logger.log("not ready");
    }

    return this._player;
  }

  redraw(): void {
    this._forceRedraw = true;
  }

  destroy(): void {
    this._videoFrame?.stopListen();

    // remove video from DOM
    if (this._player) {
      this._player.src = "";
      this._player.load();
    }
  }

  setStatFrameCall(isCall: boolean = false): void {
    if (!this._videoFrame) {
      this._logger.error("setStatFrameCall: this._videoFrame is undefined");
      return;
    }

    // this._isStatFrameCall = isCall;
    this._videoFrame.setStatFrameCall(isCall);
  }

  getStat(): Stat | undefined {
    if (!this._player) {
      return undefined;
    }

    if (!this._mediaSource) {
      return undefined;
    }

    let buffer = [];
    if (this._mediaSource.sourceBuffers.length > 0) {
      const sourceBuffer = this._mediaSource.sourceBuffers[0];
      const buffered = sourceBuffer.buffered;
      for (let i = 0; i < buffered.length; i++) {
        const start = buffered.start(i);
        const end = buffered.end(i);
        // const length = end - start;
        buffer.push({
          start: start,
          end: end,
          // length: (length | 0) + ((length % 1 * 1000 | 0) / 1000)
        });
      }
    }

    let bufferLatency;
    if (buffer.length > 0) {
      const lastBuffer = buffer[buffer.length - 1];
      bufferLatency = lastBuffer.end - this._player.currentTime;
      bufferLatency = (bufferLatency * 10 | 0) / 10;
    }

    let liveLatency = this._videoFrame && this._lastTime > 0 && this._videoFrame.time > 0 ? (this._lastTime - this._videoFrame.time) / 1000 : 0;
    liveLatency = Math.trunc(liveLatency * 10) / 10;
    const stat: Stat = {
      // corruptedVideoFrames: 0,
      // creationTime: 0,
      droppedVideoFrames: 0,
      droppedVideoFramesPercent: 0,
      totalVideoFrames: 0,
      live_latency: liveLatency,
      buffer_latency: bufferLatency,
      buffer_media_queue_length: this._mediaQueue.length,
      buffer_player: JSON.stringify(buffer.reverse()),
      currentTime: this._player.currentTime,
      // duration: this._player.duration,
      playbackRate: this._player.playbackRate,
      paused: this._player.paused,
      // ended: this._player.ended,
      // seeking : this._player.seeking,
      readyState: this._player.readyState,
      chunkList: this._lastChunkList,
      timeBetweenKeyChunks: this._timeBetweenKeyChunks
    };

    if (this._player.getVideoPlaybackQuality) {
      let quality = this._player.getVideoPlaybackQuality();
      // stat.corruptedVideoFrames = quality.corruptedVideoFrames;
      // stat.creationTime = quality.creationTime;
      stat.droppedVideoFrames = quality.droppedVideoFrames;
      stat.totalVideoFrames = quality.totalVideoFrames;
    } else {
      // @ts-ignore
      stat.droppedVideoFrames = this._player.webkitDroppedFrameCount;
      // @ts-ignore
      stat.totalVideoFrames = this._player.webkitDecodedFrameCount;
    }
    stat.droppedVideoFramesPercent = ((stat.droppedVideoFrames / stat.totalVideoFrames) * 100 * 1000 | 0) / 1000;

    return stat;
  };
}

type VideoFrameTimeQueueElement = {
  time: number,
  n: number
};

type VideFrameInitParameters = {
  video: HTMLVideoElement,
  fps?: ((fps: number) => void) | null,
  /**
   * @param time current draw time
   * @param [isLast] will be true with last frame in interval
   */
  frame?: ((time: number, isLast?: boolean) => void) | null,
  logger?: Logger | Console
}

class VideoFrame {
  _logger: Logger | Console = console;

  stop = false;
  time = 0;
  number = 0;
  fps = 0;
  fpsTime = 0;
  fpsNumber = 0;
  startTime = 0;

  video: HTMLVideoElement;

  total = 0;
  totalTimeNumber = 0;
  timeQueue: VideoFrameTimeQueueElement[] = [];

  prevTime = 0;

  parameters: VideFrameInitParameters;

  _isStatFrameCall = false;

  constructor(parameters: VideFrameInitParameters) {
    if (typeof parameters.logger !== "undefined") {
      this._logger = parameters.logger;
    }

    this.parameters = parameters;

    this.video = this.parameters.video;
  }

  reinit() {
    this.total = 0;
    this.totalTimeNumber = 0;
    this.timeQueue = [];

    this.time = 0;
    this.prevTime = 0;
  }

  listen(): void {
    let frame = () => {
      if (!this.video || this.video.paused || this.video.ended) return;

      const totalVideoFrames = this.video.getVideoPlaybackQuality
        ? this.video.getVideoPlaybackQuality().totalVideoFrames
        // @ts-ignore
        : this.video.webkitDecodedFrameCount;

      // call frame callback for every frame
      if (this.total === totalVideoFrames) {
        // prevent several calls for the same frame
        if (this._isStatFrameCall || this.prevTime !== this.time) {
          this.parameters.frame && this.parameters.frame(this.time);
        }
        return;
      }

      this.total = totalVideoFrames;

      // let time = this.startTime + this.video.currentTime * 1000;
      let time = 0;
      // let n = 0;
      let timeInfo: VideoFrameTimeQueueElement | undefined;
      do {
        timeInfo = this.timeQueue.shift();
        if (timeInfo) {
          time = timeInfo.time;
          // n = timeInfo.n;
        }
      } while (timeInfo && timeInfo.n < totalVideoFrames);

      if (time === 0 || this.time === time) {
        return;
      }

      fps();

      this.prevTime = this.time;
      this.time = time;

      this.parameters.frame && this.parameters.frame(this.time);
    };

    let fps = () => {
      let time = Date.now();
      this.fpsNumber++;
      if (time - this.fpsTime >= 1000) {
        this.fpsTime = time;
        this.fps = this.fpsNumber;
        this.fpsNumber = 0;

        this.parameters.fps && this.parameters.fps(this.fps);
      }
    };

    const loop: VideoFrameRequestCallback = (now, metadata) => {
      frame();

      this.video.requestVideoFrameCallback(loop);
    };

    if (this.video) {
      this.video.requestVideoFrameCallback(loop);
    }
  }

  stopListen(): void {
  }

  setStartTime(startTime: number): void {
    this.startTime = startTime;
  }

  addTime(timestampList: number[], isKey: boolean = false): void {
    this.totalTimeNumber++;
    this.timeQueue.push({n: this.totalTimeNumber, time: timestampList[0]});
  }

  setStatFrameCall(isCall: boolean = false): void {
    this._isStatFrameCall = isCall;
  }
}
