import { WebSocketMediaTransport } from "./websocketmediatransport";
import { WebRTCMediaTransport } from "./webrtcmediatransport";
import { FileTransport } from "./filetransport";
import { MSEPlayer } from "./mseplayer";
import { ImagePlayer } from "./imageplayer";
// import  md5 from "md5";
import m from "mithril";
import {
  __,
  AjaxError,
  alignBorder,
  API,
  DeferredPromise,
  getSnapshotURL,
  idle,
  isElectron,
  Log,
  Logger,
  ptsToTimestamp,
  timestampToPts,
  Utils
} from "@solid/libs";
import { MetaDataRender, MetadataRenderVisual } from "./metadatarender";
import {
  StreamParams,
  Metadata,
  MSEMediaPlayerConnection,
  MSEMediaPlayerError,
  MSEMediaPlayerInitParameters,
  MSEMediaPlayerMetadataType,
  MSEMediaPlayerParameters,
  MSEMediaPlayerStatus,
  MSEMediaPlayerStream,
  MSEMediaPlayerTransport,
  UUID
} from "@solid/types";
import TypedEventEmitter from "typed-emitter";
import { EventEmitter } from "events";

export type Message = {
  code?: number,
  error?: string,
  event?: "EOA" | "EOS" | "EOC",
  mime?: string,
  size?: {
    width: number,
    height: number
  }
  c?: number,
  s?: number,
  md5?: string,
  n?: number,
  pts?: string | string[],
  metadata?: Metadata[],
  key?: boolean,
};

export type Chunk = {
  key: boolean,
  skip: boolean
}

export type Media = {
  time: number,
  data: Uint8Array
}

type InitStreamRequest = {
  cmd: "init",
  objid: UUID,
  streamid: string,
  ticket: string,
  params: {
    audio?: boolean,
    fast?: boolean,
    startTime?: number,
    endTime?: number
  }
}

export type TransportInitParameters = {
  obj?: UUID | null,
  open?: () => void,
  close?: ((code: number, error?: string) => void) | null,
  message?: ((message: {data: ArrayBuffer | string}) => void) | null,
  error?: () => void,
  isUseFileTransport?: boolean,
  isLocal?: boolean,
  alwaysConnect?: boolean,
  isArchive?: boolean,
  startTime?: number,
  endTime?: number
}

type TransportInitResult = {
  transport: WebSocketMediaTransport | WebRTCMediaTransport,
  mime: string,
  width: number
  height: number
}

type MSEMediaPlayerEvent = {
  play: (streamParams: StreamParams) => void,
  pause: (byUser: boolean) => void,
  stop: (code?: number, message?: string) => void,
  connecting: (message: string) => void,
  initializing: () => void,
  buffering: () => void,
  frame: (time: number, metadata?: [Metadata]) => void,
  click: (width: number, height: number, x: number, y: number) => void,
  archive: () => void,
  live: () => void,
  resize: () => void,
  record: () => void,
  geoPosition: (geo: [lat: number, lng: number, alt?: number]) => void
};
type MSEMediaPlayerEventName = keyof MSEMediaPlayerEvent;

let getByChunk = false;

export class MSEMediaPlayer {
  _logger: Logger | Console = console;

  _debug = true;
  _debugData = false;

  _node: HTMLElement | null = null;
  _mediaPlayerNode: HTMLElement | null = null;

  _obj: UUID | null = null;

  _player: MSEPlayer | ImagePlayer | null = null;
  _imagePlayer: ImagePlayer | null = null;
  _msePlayer: MSEPlayer | null = null;

  _isArchive = false;

  _status: MSEMediaPlayerStatus = MSEMediaPlayerStatus.STOP;

  _startDate = new Date();
  _endDate = new Date();

  /**
   * list of topics
   *
   * @type {{newimage: {}, imageclick: {}, zoom: {imageclick: {}}, player: {play: {}, pause: {}, stop: {}, connecting: {}, buffering: {}, frame: {}}}}
   * @private
   */
  _topics_old = {
    // old events
    playerready: {},
    bufferchange: {},
    snapshotfinish: {},
    newimage: {},
    imageclick: {},
    statuschange: {},
    // new events
    positionchange: {},
    zoom: {
      imageclick: {}
    },
    player: {
      play: {},
      pause: {},
      stop: {},
      connecting: {},
      buffering: {},
      frame: {},
      setSpeed: {},
      geoPosition: {}
    },
    // not player object events
    archive: {},
    live: {}
  };

  _topics = new Set<MSEMediaPlayerEventName>([
    "play",
    "pause",
    "stop",
    "connecting",
    "initializing",
    "buffering",
    "frame",
    "click",
    "archive",
    "live",
    "resize",
    "record",
    "geoPosition"
  ]);

  _topicsEvents = new EventEmitter() as TypedEventEmitter<MSEMediaPlayerEvent>;

  _stopCodeMessage = {
    "0":       "Stop",
    "default": "Something went wrong",
    [MSEMediaPlayerError.VIDEO_UNAVAILABLE]: __("Video unavailable"),
    [MSEMediaPlayerError.AUDIO_UNAVAILABLE]: __("Audio unavailable"),
    [MSEMediaPlayerError.INVALID_REQUEST]: __("Invalid request"),
    [MSEMediaPlayerError.INTERNAL_ERROR]: __("Internal error")
  };

  _isPrintError = true;

  _archiveIntervalRequest = getByChunk ? 60 * 1000 : 3600 * 1000; // ms

  _transport: WebSocketMediaTransport | WebRTCMediaTransport | null = null;
  _webRTCTransport: WebRTCMediaTransport = new WebRTCMediaTransport({logger: this._logger, debug: this._debug, debugData: this._debugData});
  _webSocketTransport: WebSocketMediaTransport = new WebSocketMediaTransport({logger: this._logger, debug: this._debug, debugData: this._debugData});
  _fileTransport: FileTransport = new FileTransport("reader"/*{logger: this._logger, debug: this._debug, debugData: this._debugData}*/);

  _preferredConnectionType: MSEMediaPlayerConnection = MSEMediaPlayerConnection.AUTO;

  _disableLocalTransport: boolean = false;

  _showControlsOnPause = true;

  _streamid: MSEMediaPlayerStream = MSEMediaPlayerStream.FULL;

  _rendered: StreamParams = {
    mime: "",
    width: 0,
    height: 0
  };

  _isFirstChunk = false;

  _pixelAspectRatio = 1;
  _isStretchAspectRatio = false;

  _currentTime: number | undefined = undefined;

  _metadataType: MSEMediaPlayerMetadataType = MSEMediaPlayerMetadataType.TRIGGERED;
  _isMetadataVisible = true;
  _metadataList: Metadata[] = [];
  _metadataListDraw: MetadataRenderVisual = {};
  _startTime = 0;

  _lastInitParameters: TransportInitParameters | null = null;

  _api: API = new API();

  _inProgress = false;
  _snapshotLoading = false;

  _stoppedAfterIdle = false;
  _stopOnIdleTimeout = 10 * 60 * 1000; // ms

  // TODO: specify type
  _renderList = {};
  _canvas: HTMLCanvasElement | null = null;

  _tryToPlay = false;
  _lastPlayPromise: DeferredPromise | null = null;
  _showPauseButton = false;

  _stopRender = false;

  _context: CanvasRenderingContext2D | null = null;

  _muted: boolean = false;

  _reconnectTimeoutId?: ReturnType<typeof setTimeout>;
  _playTimeoutId?: ReturnType<typeof setTimeout>;

  constructor(parameters: MSEMediaPlayerParameters = {}) {
    parameters = Object.assign({node: null}, parameters);

    if (typeof parameters.logger !== "undefined") {
      this._logger = parameters.logger;
    }

    this._node = parameters.node ?? null;

    if (parameters.connectionType) {
      this._preferredConnectionType = parameters.connectionType;
    } else {
      let storedConnectionType: MSEMediaPlayerConnection | null = null;
      try {
        storedConnectionType = localStorage.getItem("connection") as MSEMediaPlayerConnection | null ?? process.env.PLAYER_CONNECTION as MSEMediaPlayerConnection;
      }
      catch (e) {
      }
      this._preferredConnectionType = storedConnectionType ?? MSEMediaPlayerConnection.AUTO;
    }
  }

  async init(parameters?: MSEMediaPlayerInitParameters, isPrintError?: boolean): Promise<void> {
    this._isPrintError = typeof isPrintError === "undefined" ? this._isPrintError : isPrintError;

    let settings: MSEMediaPlayerInitParameters = {
      ...parameters,
      cacheSize: 4, // size of cache in MB
      streamid: MSEMediaPlayerStream.FULL,
      isMetaDataDraw: true,
      metadataType: MSEMediaPlayerMetadataType.TRIGGERED
    };

    if (typeof settings.idleTimeout !== "undefined") {
      this._stopOnIdleTimeout = settings.idleTimeout * 60 * 1000;
    }

    if (settings.obj) {
      this._obj = settings.obj;
      if (settings.time) {
        this._currentTime = settings.time;
        this._isArchive = true;
      }
    }

    if(typeof settings.showControlsOnPause !== "undefined" && !settings.showControlsOnPause){
      this._showControlsOnPause = false;
    }

    if (typeof settings.isMetaDataDraw == "boolean") {
      this._isMetadataVisible = settings.isMetaDataDraw;
    }

    let storedMetadataType: MSEMediaPlayerMetadataType | null = null;
    try {
      storedMetadataType = localStorage.getItem("metadata") as MSEMediaPlayerMetadataType | null;
    }
    catch (e) {}

    if (storedMetadataType) {
      this._metadataType = storedMetadataType;
    } else
    if (typeof settings.metadataType !== "undefined") {
      this._metadataType = settings.metadataType;
    }

    if (typeof settings.connectionType !== "undefined") {
      this._preferredConnectionType = settings.connectionType;
    }

    if (settings.streamid) {
      this._streamid = settings.streamid;
    }

    if (typeof settings.showPauseButton == "boolean") {
      this._showPauseButton = settings.showPauseButton;
    }

    if (typeof settings.muted == "boolean") {
      this._muted = settings.muted;
    }

    if (typeof settings.disableLocalTransport == "boolean") {
      this._disableLocalTransport = settings.disableLocalTransport;
      if (this._disableLocalTransport) {
        this._preferredConnectionType = MSEMediaPlayerConnection.CLOUD;
      }
    }

    let id = "MediaPlayer_" + Math.round((Math.random() * 10000));

    const MediaPlayerStat = (): m.Component => {
      let canvas: HTMLCanvasElement | null = null;
      let canvasWidth = 240;
      let canvasHeight = 10;
      let lastLength = 0;
      let isShowStat = true;

      const clearFrames = () => {
        if (!canvas) return;

        let context = canvas.getContext("2d");
        context?.clearRect(0, 0, canvas.width, canvas.height);
      };
      const drawFrames = (canvas: HTMLCanvasElement | null, list: Chunk[]) => {
        if (!canvas || !list) return;

        let context = canvas.getContext("2d");
        if (!context) {
          this._logger.error("drawFrames: context is undefined")
          return;
        }

        const lineWidth = 1;
        const leftShift = lineWidth;

        context.lineWidth = lineWidth;

        for (let i = 0; i < list.length; i++) {
          let strokeStyle = "yellow";
          if (list[i].key) {
            strokeStyle = "green";
          }
          if (list[i].skip) {
            strokeStyle = "red";
          }
          context.beginPath();
          context.strokeStyle = strokeStyle;
          context.moveTo(leftShift + i * lineWidth, 0);
          context.lineTo(leftShift + i * lineWidth, 15);
          context.stroke();
        }
      };

      const getStat = (): { strings: {[label: string]: string}, chunkList: Chunk[] }  => {
        const playerStat = this._player ? this._player.getStat() : undefined;
        if (!playerStat) {
          return {
            strings: {},
            chunkList: []
          }
        }

        if (!(this._player instanceof MSEPlayer)) {
          return {
            strings: {},
            chunkList: []
          }
        }
        const drawNode = this.getDrawNode();
        let stat = {
          "Viewport": drawNode ? `${drawNode.offsetWidth}x${drawNode.offsetHeight}` : "",
          "Resolution, mime": this._transport ? `${this._transport._stream.width}x${this._transport._stream.height}, ${this._transport._stream.mime}` : "",
          "Dropped Frames": `${playerStat.droppedVideoFrames}/${playerStat.totalVideoFrames}, ${playerStat.droppedVideoFramesPercent}%`,
          "Live, Buffer Latency": `${playerStat.live_latency}, ${playerStat.buffer_latency}`,
          "Buff Queue Length, Player Buff": `${playerStat.buffer_media_queue_length}, ${playerStat.buffer_player}`,
          "Player Time, paused, readyState": `${playerStat.currentTime}, ${playerStat.paused}, ${playerStat.readyState}`,
          "Time between key chunks": `${playerStat.timeBetweenKeyChunks}`,
          "Playback rate": `${playerStat.playbackRate}`
        };
        return {
          strings: stat,
          chunkList: playerStat.chunkList
        };
      };

      return {
        oncreate: ({dom}) => {
          canvas = dom.querySelector("canvas");
        },
        onremove: () => {
          this._player && this._player.setStatFrameCall(false);
        },
        view: () => {
          this._player && this._player.setStatFrameCall(true);
          const stat = getStat();

          let list = typeof stat.chunkList !== "undefined" ? stat.chunkList : [];
          if (lastLength > list.length) {
            clearFrames();
          }
          lastLength = list.length;
          drawFrames(canvas, list);

          return m(".stat", [
            m("canvas", {
              width: canvasWidth,
              height: canvasHeight,
              style: {
                width: `${canvasWidth}px`,
                height: `${canvasHeight}px`
              }
            }),
            m("controls", [
              m("button", {onclick: () => {
                isShowStat = true;
              }}, "Stat"),
              m("button", {onclick: () => {
                isShowStat = false;
              }}, "Log")
            ]),
            isShowStat && m(".parameters",
              Object.entries(stat.strings).map(([key, value]) => {
                if (key === "chunkList") return;
                return m("div", `${key} = ${value}`);
              })
            ),
            !isShowStat && m(MediaPlayerLog)
          ]);
        }
      }
    };

    type MediaPlayerLogAttrs = {
    }
    type MediaPlayerLogState = {
      isAutoScroll: boolean,
      onupdate: () => void
    }
    const MediaPlayerLog: m.Component<MediaPlayerLogAttrs, MediaPlayerLogState> = {
      oninit: ({state}) => {
        state.isAutoScroll = true;
      },
      view: ({state}) => {
        return m(".log", {
          onmousedown: () => {
            state.isAutoScroll = false;
          },
          onmouseup: (e: MouseEvent) => {
            const target: HTMLElement | null = e.target as HTMLElement;
            if (target && (target.scrollTop === (target.scrollHeight - target.clientHeight))) {
              state.isAutoScroll = true;
            }
          },
        }, this._logger instanceof Logger && this._logger.getLog().map((row) => {
          let date = new Date(row.time);
          let strDate = `${String(date.getUTCHours()).padStart(2, "0")}:${String(date.getUTCMinutes()).padStart(2, "0")}:${String(date.getUTCSeconds()).padStart(2, "0")}:`;
          return m("div", `${strDate} ${row.data}`);
        }));
      },
      onupdate: ({state, dom}) => {
        if (state.isAutoScroll) {
          dom.scrollTop = dom.scrollHeight - dom.clientHeight;
        }
      }
    };

    type MediaPlayerAttrs = {}
    type MediaPlayerState = {
      isShowStat: boolean,
      oncreate: () => void
    }
    const MediaPlayer: m.Component<MediaPlayerAttrs, MediaPlayerState> = {
      oninit: ({state}) => {
        state.isShowStat = false;
      },
      oncreate: ({dom}) => {
        if (this._obj) {
          this._showSnapshot();
        }
        this._canvas = dom.querySelector("canvas.playercanvas");
        this._mediaPlayerNode = document.querySelector("#" + id);

        this._initRender();

        const correctSize = Utils.throttleToDraw(() => {
          this._correctSize();
        });

        const resizeObserver = new ResizeObserver(Utils.throttleToDraw((entries) => {
          for (const entry of entries) {
            if (this._mediaPlayerNode === entry.target) {
              correctSize();
            }
          }
        }));
        resizeObserver.observe(dom);
      },
      view: ({state}: {state: {isShowStat : boolean}}) => {
        const isShowCanvas = this._transport && this._transport._stream && this._transport._stream.mime === "video/x-jpeg";

        return m(".mediaplayer", {
            id: id,
            tabindex: 0,
            onkeydown: (e: KeyboardEvent) => {
              if (e.ctrlKey && e.keyCode === 76) {
                state.isShowStat = !state.isShowStat;
                m.redraw();
              }
            },
            onmousedown : (e: MouseEvent) => {
              // middle button
              if (e.button === 1) {
                state.isShowStat = !state.isShowStat;
                m.redraw();
              }
            }
          }, [
            m(".video", [
              m("video", {"playsInline": ""})
            ]),
            m("canvas.playercanvas", {style: {display: isShowCanvas ? "" : "none"}}),
            this._isMetadataVisible && m(".metadata", {
              onclick: (event: MouseEvent & {target: {clientWidth: number, clientHeight: number}, layerX: number, layerY: number}) => {
                this._publish("click", event.target.clientWidth, event.target.clientHeight, event.layerX, event.layerY);
              }
            }, m(MetaDataRender, {
              list: this._metadataListDraw,
              time: this._currentTime!,
              metadataType: this._metadataType,
              size: this._rendered
            })),
            m(".snapshot", [
              m("img")
            ]),
            (this._inProgress || this._snapshotLoading) && m(".progress", m(".indicator", m("i.fas.fa-circle-notch.fa-spin"))),
            (this._showControlsOnPause && this._status === MSEMediaPlayerStatus.PAUSE
             || this._showPauseButton && this._status === MSEMediaPlayerStatus.PLAY
            ) && m(".mediaplayer-controls", {
              onclick: () => {
                if (this._status === MSEMediaPlayerStatus.PAUSE) {
                  this.play();
                }
                if (this._showPauseButton && this._status === MSEMediaPlayerStatus.PLAY) {
                  this.pause(undefined, true);
                }
              }
            }, [
              this._status === MSEMediaPlayerStatus.PAUSE && m(".play .fas .fa-play-circle"),
              this._showPauseButton && this._status === MSEMediaPlayerStatus.PLAY && m(".pause .fas .fa-pause-circle"),
            ]),
            m(".error"),
            state.isShowStat && m(MediaPlayerStat)
          ]
        );
      }
    };

    if (!this._node) {
      throw new Error("this._node is undefined");
    }

    m.mount(this._node, MediaPlayer);

    this._msePlayer = new MSEPlayer({logger: this._logger});
    this._imagePlayer = new ImagePlayer({logger: this._logger});

    const fps = (fps: number) => {
      // this._logger.log("fps> ", fps);
    };

    const updateWebRTCStatusTime = async (time: number) => {
      if (this.getTransportType() !== MSEMediaPlayerTransport.WEBRTC) {
        return;
      }

      if (!this._transport || !(this._transport instanceof WebRTCMediaTransport)) {
        return;
      }

      const avatarId = await this._transport.getAvatarId(this._obj);
      if (!avatarId) {
        return;
      }

      this._transport.setWebRTCStatus(avatarId, {time: time});
    };

    const updateWebRTCStatusTimeThrottled = Utils.throttle(updateWebRTCStatusTime, 5000);

    /**
     *
     * @param {number} time current draw time
     * @param {boolean} [isLast] will be true with last frame in interval
     */
    const frame = (time: number, isLast?: boolean) => {
      if (!this._transport) {
        this._logger.error("frame: this._transport is undefined")
        return;
      }

      if (!this._player) {
        this._logger.error("frame: this._player is undefined")
        return;
      }

      if (!this._lastInitParameters) {
        this._logger.error("frame: this._lastInitParameters is undefined")
        return;
      }

      if (this._transport._lastReceivedTime === time
        && this._transport._endEvent
      ) {
        this._player.pause();

        let event = this._transport._endEvent.event;
        this._logger.log(`process event ${event}`);

        switch (event) {
          case "EOA":
            if (!this._lastInitParameters.isUseFileTransport) {
              this.stop(MSEMediaPlayerError.END_OF_ARCHIVE, __("End of archive"));
            } else {
              // align to next chunk
              const {start} = alignBorder({startTime: this._transport._endKeyTime + 30000})
              this._currentTime = start; // increase to next chunk to play next interval, after was stopped
              this._play(false);
            }
            break;
          case "EOS":
            // TODO: this._play();
            this.stop(MSEMediaPlayerError.END_OF_STREAM, __("End of stream"));
            break;
          case "EOC":
            this.stop(MSEMediaPlayerError.END_OF_STREAM, __("End of stream"));

            // this.stop(MSEMediaPlayerError.END_OF_CHUNK, __("End of chunk"));
            // align to next chunk
            // const {start} = alignBorder({startTime: this._transport._endKeyTime + 30000})
            // this._currentTime = start; // increase to next chunk to play next interval, after was stopped
            // this._play(true);
            break;
        }
      }

      if (!time) {
        m.redraw();
        return;
      }

      while (this._metadataList.length > 0) {
        let metadata = this._metadataList[0];

        let metadataTime = ptsToTimestamp(metadata.pts);
        // wait for player time
        if (metadataTime > time) {
          break;
        }

        if (metadata.list) {
          // merge new metadata elements with old one
          Object.assign(this._metadataListDraw, metadata.list);

          Object.keys(metadata.list).forEach((id) => {
            // save time for each element of metadata
            this._metadataListDraw[id].time = metadataTime;

            // publish geo data if exist
            if (metadata.list[id].visual?.opt?.geo) {
              const geo = metadata.list[id].visual?.opt?.geo;
              if (geo) {
                this._publish("geoPosition", geo);
              }
            }
          });
        }

        this._metadataList.shift();
      }

      this._currentTime = time;

      updateWebRTCStatusTimeThrottled(time);

      m.redraw();

      this._publish("frame", time);
    };

    const error = (code?: number, message?: string) => {
      this._logger.error(`[${code}] ${message}`);
      this._publish("stop", code, message);
    };

    if (!this._node) {
      throw new Error("this._node is undefined");
    }

    const videoBlock: HTMLVideoElement | null = this._node.querySelector("video");
    if (!videoBlock) {
      throw new Error("video block is undefined");
    }

    await Promise.all([
      this._msePlayer.init({
        node: videoBlock,
        mediaplayer: this,
        fps: fps,
        frame: frame,
        error: error
      }),
      // TODO: correct image player
      /*this._imagePlayer.init({
        mediaplayer: this,
        frame: frame
      })*/
    ]);

    this._addEvents();
  };

  muteAudio(muted: boolean): void {
    if (!this._player) {
        this._logger.error("muteSound: this._player is undefined");
        return;
    }
    this._muted = muted;
    this._player.muteAudio(muted);
  }

  getAudio(): boolean {
    return !this._muted;
  }

  getMetadataDrawType(): MSEMediaPlayerMetadataType {
    return this._metadataType;
  };

  setMetadataDrawType(metadataType: MSEMediaPlayerMetadataType) {
    this._metadataType = metadataType;

    m.redraw();
  };

  showMetaData() {
    this._isMetadataVisible = true;

    m.redraw();
  };

  hideMetaData() {
    this._isMetadataVisible = false;

    m.redraw();
  };

  destroy() {
    this.stop();

    this._stopRender = true;
    this._player && this._player.destroy();

    if (this._node) {
      m.mount(this._node, null);
      this._node = null;
    }

    this._webSocketTransport && this._webSocketTransport.close(false);
    this._webRTCTransport && this._webRTCTransport.close(false);
  };

  getConnectionType(transport: WebSocketMediaTransport | WebRTCMediaTransport | null = this._transport): Exclude<MSEMediaPlayerConnection, MSEMediaPlayerConnection.AUTO> | "" {
    if (!transport) {
      return "";
    }
    return transport instanceof WebRTCMediaTransport ? MSEMediaPlayerConnection.LOCAL : MSEMediaPlayerConnection.CLOUD;
  };

  getTransportType(transport: WebSocketMediaTransport | WebRTCMediaTransport | null = this._transport): MSEMediaPlayerTransport | "" {
    if (!transport) {
      return "";
    }
    return transport instanceof WebRTCMediaTransport ? MSEMediaPlayerTransport.WEBRTC : MSEMediaPlayerTransport.WEBSOCKET;
  };

  getPreferredConnectionType(): MSEMediaPlayerConnection {
    return this._preferredConnectionType;
  };

  async setPreferredConnectionType(type: MSEMediaPlayerConnection = MSEMediaPlayerConnection.AUTO, isPlay: boolean = true): Promise<void> {
    if (![MSEMediaPlayerConnection.LOCAL, MSEMediaPlayerConnection.CLOUD, MSEMediaPlayerConnection.AUTO].includes(type)) {
      throw {message: "incorrect connection type"};
    }

    this._preferredConnectionType = this._disableLocalTransport ? MSEMediaPlayerConnection.CLOUD : type;

    if (this.getConnectionType(this._transport) == this._preferredConnectionType) {
      return;
    }

    if (this._transport?.isOpen()) {
      this._transport.close(isPlay);
    } else {
      // reinit connection if it closed with current parameters
      if (isPlay && this._lastInitParameters && Object.keys(this._lastInitParameters).length > 0) {
        await this._initAllTransportStreams(this._lastInitParameters);
      }
    }
  };

  /**
   *
   * @param transport
   * @returns {string}
   * @private
   */
  _getTransportCaption(transport = this._transport) {
    return this.getTransportType(transport) === MSEMediaPlayerTransport.WEBRTC ? MSEMediaPlayerConnection.LOCAL : MSEMediaPlayerConnection.CLOUD;
  };

  _correctSize() {
    if (!this._node) return;

    /*
    let width = this._transport._stream.width / this._pixelAspectRatio;
    let height = this._transport._stream.height;
    if (this._pixelAspectRatio > 1) {
      width = this._transport._stream.width;
      height = this._transport._stream.height * this._pixelAspectRatio;
    }
    */

    const stream = this._transport && this._transport._stream ? this._transport._stream : undefined;

    let imageAspectRatio = !stream ? 0 : stream.width / stream.height;
    let playerBlock = this._node;

    let playerWidth = playerBlock.offsetWidth;
    let playerHeight = playerBlock.offsetHeight;
    let playerAspectRatio = playerWidth / playerHeight;

    if (this._isStretchAspectRatio) {
      imageAspectRatio = playerAspectRatio;
    } else {
      imageAspectRatio /= this._pixelAspectRatio;
    }

    let cssWidth = "";
    let cssHeight = "";

    let width = 0;
    let height = 0;

    if (imageAspectRatio <= playerAspectRatio) {
      width = Math.floor(playerHeight * imageAspectRatio + 0.5);
      height = playerHeight;

      cssWidth = width + "px";
      cssHeight = 100 + "%";
    } else {
      width = playerWidth;
      height = Math.floor(playerWidth / imageAspectRatio + 0.5);

      cssWidth = 100 + "%";
      cssHeight = height + "px";
    }

    const metadataBlock: HTMLElement | null = this._node.querySelector(".metadata");
    if (metadataBlock) {
      metadataBlock.style.width = cssWidth;
      metadataBlock.style.height = cssHeight;
    }

    // this._logger.log(this._video.videoWidth, this._video.videoHeight);
    if (this._canvas) {
      this._canvas.width = !stream ? 0 : stream.width;
      this._canvas.height = !stream ? 0 : stream.height;
      this._canvas.style.width = cssWidth;
      this._canvas.style.height = cssHeight;
    }

    this._rendered.width = width;
    this._rendered.height = height;

    const videoBlock = this._node.querySelector("video");
    if (videoBlock) {
      if (stream && stream.mime === "video/x-jpeg") {
        videoBlock.style.display = "none";
      } else {
        videoBlock.style.display = "block";
      }
    }

    this._publish("resize");
  };

  /**
   * @private
   */
  _addEvents() {
    this.subscribe("play", () => {
      this._status = MSEMediaPlayerStatus.PLAY;
      this.printErrorMessage("");

      this._correctSize();

      this._hideSnapshot();

      this._render();

      m.redraw();
    });

    this.subscribe("stop", (code?: number, message?: string) => {
      this.clearTimers();
      if (this._status === MSEMediaPlayerStatus.PAUSE) {
        return;
      }

      this._player && this._player.pause();
      this._status = MSEMediaPlayerStatus.STOP;

      let errorMessage;
      if (message) {
        errorMessage = message;
      } else {
        errorMessage = typeof code !== "undefined" && this._stopCodeMessage[code] ? __(this._stopCodeMessage[code]) : __(this._stopCodeMessage["default"]);
      }

      this.printErrorMessage("[" + code + "] " + errorMessage);

      this._showSnapshot();

      this._inProgress = false;
      this._snapshotLoading = false;

      m.redraw();

      /**
       * @type {Array.<number>}
       */
      const errorCodeList = [
        MSEMediaPlayerError.NOT_AUTHORIZED,
        MSEMediaPlayerError.AVATAR_OFFLINE,
        MSEMediaPlayerError.BROKEN_ON_AVATAR_SIDE,
        MSEMediaPlayerError.LIVE_NOT_OPENED_NOT_ENOUGH_AVATAR_VSAAS_BANDWIDTH,
        MSEMediaPlayerError.LIVE_STOPPED_NOT_ENOUGH_AVATAR_VSAAS_BANDWIDTH,
        MSEMediaPlayerError.STREAM_STOPPED_NOT_ENOUGH_AVATAR_LOCAL_BANDWIDTH,
        MSEMediaPlayerError.CRITICAL,
        1006 // websocket connection "Abnormal Closure"
      ];

      if (typeof code !== "undefined" && code > 59000 && code < 60000) {
        // do not restart
      } else
      if (code === MSEMediaPlayerError.STREAM_STOPPED_UNEXPECTEDLY
        || code === MSEMediaPlayerError.INTERNAL_ERROR
        || code === MSEMediaPlayerError.VIDEO_UNAVAILABLE && !this._isArchive) {
        const delay = 2 * 1000; // ms
        this._logger.log(`reconnecting in ${delay/1000} sec`);
        this._reconnectTimeoutId = setTimeout(() => this.play(), delay);
      } else
      if (typeof code !== "undefined" && errorCodeList.includes(code)) {
        const delay = 5 * 1000; // ms // TODO: exponential rise to 1min
        this._logger.log(`reconnecting in ${delay/1000} sec`);
        this._reconnectTimeoutId = setTimeout(() => this.play(), delay);
      } else
      // play is called in aod request callback
      if (code === MSEMediaPlayerError.VIDEO_UNAVAILABLE && this._isArchive && this.getConnectionType() !== MSEMediaPlayerConnection.LOCAL) {
        if (this._lastInitParameters?.isUseFileTransport) {
          if (this._lastInitParameters.startTime) {
            // align chunk
            const {start} = alignBorder({startTime: this._lastInitParameters.startTime})
            this._currentTime = start; // increase to next chunk to play next interval, after was stopped
            this._play(false);
          }
        }
      } else
      if (
        code === MSEMediaPlayerError.END_OF_ARCHIVE ||
        code === MSEMediaPlayerError.END_OF_STREAM ||
        code === MSEMediaPlayerError.END_OF_CHUNK ||
        code === MSEMediaPlayerError.MSE_PLAYER_UNSUPPORTED_CODEC
        ) {
        // do not restart
      } else
      if (code !== 0) {
        const delay = 5 * 1000; // ms // TODO: exponential rise to 1min
        this._logger.log(`reconnecting in ${delay/1000} sec`);
        this._reconnectTimeoutId = setTimeout(() => this.play(), delay);
      }
    });

    this.subscribe("pause", () => {
      this.clearTimers();

      this._status = MSEMediaPlayerStatus.PAUSE;
      this.printErrorMessage(__("Pause"));

      m.redraw();
    });

    this.subscribe("connecting", (message: string) => {
      this.printErrorMessage(__("Opening video") + " / " + message);

      m.redraw();
    });

    this.subscribe("initializing", () => {
      this.printErrorMessage(__("Initializing"));

      m.redraw();
    });

    this.subscribe("buffering", () => {
      // this._status = MSEMediaPlayerStatus.STOP;
      this.printErrorMessage(__("Buffering"));

      m.redraw();
    });

    this._stopOnIdleTimeout !== 0 && idle({
      onIdle: () => {
        // if Player is working with a local stream, DO NOT STOP it
        if (this.getConnectionType() !== MSEMediaPlayerConnection.LOCAL
            && this._status === MSEMediaPlayerStatus.PLAY) {
          this._stoppedAfterIdle = true;
          this._logger.log("idle: stop");
          this.stop();
        }
      },
      onActive: () => {
        if (this._stoppedAfterIdle && this._status === MSEMediaPlayerStatus.STOP) {
          this._stoppedAfterIdle = false;
          this.play();
        }
      },
      idle: this._stopOnIdleTimeout
    }).start();
  }

  printErrorMessage(message: string): boolean {
    if (!this._isPrintError) return false;
    // if (!this._player) return false;
    if (!this._node) return false;

    const errorNode = this._node.querySelector(".error");
    if (!errorNode) {
      return false;
    }

    errorNode.innerHTML = message;

    return true;
  }

  /**
   * change player current archive time when player in pause
   * this call do not start playing at new time, play method should be called
   *
   * @param obj
   * @param time ms
   * @param isPlay
   */
  async setTime(obj: UUID, time: number, isPlay: boolean = false): Promise<void> {
    this._obj = obj;
    this._currentTime = time;
    this._isArchive = true;

    this._showSnapshot();

    if (isPlay) {
      this.stop();
      await this.play();
    }
  }

  async play(obj?: UUID | null, fromTime?: number, toTime?: number, doNotStart: boolean = false, isUseFileTransport: boolean = false): Promise<boolean> {
    if (!this._node) {
      console.assert(this._node, "player node is not created");
      return false;
    }

    this.clearTimers();

    if (this._status === MSEMediaPlayerStatus.INITIALIZING) {
      return false;
    }

    if (!obj && !fromTime && this._currentTime && this._obj && this._isArchive) {
      fromTime = this._currentTime;
    }
    if (!!obj) this._obj = obj;
    if (!this._obj) throw {message: "obj not specified"};

    if (this._obj && !fromTime) {
      this._isArchive = false;
      this._currentTime = undefined;
    } else
    if (this._obj && fromTime) {
      this._isArchive = true;
      this._currentTime = fromTime;

      let date = new Date();
      date.setTime(fromTime);
      this._startDate.setTime(date.getTime());
      let time = toTime ? toTime : this._startDate.getTime() + this._archiveIntervalRequest;
      this._endDate.setTime(time);
    }

    if (doNotStart) {
      return false;
    }

    // pause player to prevent playing old chunks before receiving new one
    if (this._status === MSEMediaPlayerStatus.PLAY) {
      this.pause();
      // this._player && this._player.pause();
    }

    this._status = MSEMediaPlayerStatus.INITIALIZING;
    this._publish("initializing");

    m.redraw();

    let parameters: TransportInitParameters = {};
    if (this._isArchive) {
      parameters.startTime = this._startDate.getTime();
      parameters.endTime = this._endDate.getTime();
      this._publish("archive");
    } else {
      this._publish("live");
    }

    this._inProgress = true;
    m.redraw();

    !this._isArchive && this._showSnapshot();

    parameters.isUseFileTransport = this._isArchive && isElectron() && isUseFileTransport;

    await this._initAllTransportStreams(parameters);

    return true;
  };

  /**
   *
   * @private
   */
  _play(isUseFileTransport = false): void {
    if (this._isArchive) {
      this.play(this._obj, this._currentTime, undefined, false, isUseFileTransport);
    } else {
      this.play(this._obj, undefined, undefined, false, isUseFileTransport);
    }
  };

  setSpeed(speed: number) {
    if (this._player instanceof MSEPlayer) {
      this._player.setSpeed(speed);
    }
  }

  pause(closeTransport = true, byUser = false): boolean {
    if (this._status === MSEMediaPlayerStatus.PAUSE) return true;
    this._publish("pause", byUser);

    // if (!this._player) return false;

    if (!this._transport || !this._transport.isOpen()) return false;
    let request = {
      cmd: "pause",
      params: {}
    };
    this._transport.send(request);

    this._player && this._player.pause();

    this._player instanceof MSEPlayer && this._player.setSpeed(1.0);

    if (closeTransport) {
      this._webSocketTransport && this._webSocketTransport.close(false);
      this._webRTCTransport && this._webRTCTransport.close(false);
    }

    return true;
  };

  stop(code?: number, message?: string): boolean {
    // if (!this._player) return false;
    if (!this._transport || !this._transport.isOpen()) return false;
    if (this._status === MSEMediaPlayerStatus.STOP) return true;

    let request = {
      cmd: "pause",
      params: {}
    };
    this._transport.send(request);

    this._publish("stop", code || 0, message || "");

    return true;
  };

  _showSnapshot(): void {
    if (!this._node) return;
    if (!this._obj) {
      this._logger.error("_showSnapshot: this._obj is undefined");
      return;
    }

    let snapshotBlock: HTMLElement | null = this._node.querySelector(".snapshot");
    if (!snapshotBlock) {
      return;
    }
    snapshotBlock.style.display = "flex";

    this._snapshotLoading = true;
    m.redraw();

    let snapshotImgBlock = snapshotBlock.querySelector("img");
    if (snapshotImgBlock) {
      snapshotImgBlock.onload = snapshotImgBlock.onerror = snapshotImgBlock.onabort = () => {
        this._snapshotLoading = false;
        m.redraw();
      };
      snapshotImgBlock.src = getSnapshotURL(this._obj, this._currentTime ? Math.round(this._currentTime / 1000) : undefined, true, true);
    }

    let videoBlock = this._node.querySelector("video");
    if (videoBlock) {
      videoBlock.style.display = "none"
    }
    // let imageBlock = this._node.querySelector(".image");
    // imageBlock.style.display = "none";
    let metadataBlock: HTMLElement | null = this._node.querySelector(".metadata");
    if (metadataBlock) {
      metadataBlock.style.removeProperty("display");
    }
  };

  _hideSnapshot(): void {
    if (!this._node) return;

    let snapshotBlock: HTMLElement | null = this._node.querySelector(".snapshot");
    if (snapshotBlock) {
      snapshotBlock.style.display = "none";
    }

    let metadataBlock: HTMLElement | null = this._node.querySelector(".metadata");
    if (metadataBlock) {
      metadataBlock.style.removeProperty("display");
    }
  };

  frameBackward() {
    this._logger.warn("frameBackward not implemented");
    return true;
  };

  async _initAllTransportStreams(parameters: TransportInitParameters): Promise<void> {
    let isWebRTC = false;
    let webRTCStreamParameters: StreamParams = {mime: "", width: 0, height: 0};
    let isWebSocket = false;
    // @ts-ignore
    let webRTCInit = false;
    // @ts-ignore
    let webSocketInit = false;

    let isFileTransport = false;

    Object.assign(this._webSocketTransport, {
      /**
       * @type {number}
       * @private
       */
      _lastReceivedTime: null,
      /**
       * @type {Uint8Array}
       * @private
       */
      _currentChunk: null,
      /**
       *
       * @type {{time: null, event: string}}
       * @private
       */
      _endEvent: null,
      /**
       * @type {number} ms
       */
      _endKeyTime: 0
    });

    Object.assign(this._webRTCTransport, {
      _lastReceivedTime: null,
      _currentChunk: null,
      _endEvent: null,
      _endKeyTime: 0
    });

    Object.assign(this._fileTransport, {
      _lastReceivedTime: null,
      _currentChunk: null,
      _endEvent: null,
      _endKeyTime: 0
    });

    const done = async () => {
      this._logger.log(`isWebSocket=${isWebSocket} isWebRTC=${isWebRTC}`);
      if (isFileTransport) {

      } else
      if (isWebSocket && isWebRTC) {
        // close websocket connection when webrtc is available
        this._webSocketTransport.close(false);

        // change websocket connection to webrtc
        if (this.getTransportType(this._transport) === MSEMediaPlayerTransport.WEBSOCKET) {
          const letsPlay = async () => {
            this._logger.log("LET'S TRY TO PLAY");
            // this.stop();

            this._webRTCTransport._doPlay = true;
            await this._initStreamMIME({
              transport: this._webRTCTransport,
              mime: webRTCStreamParameters.mime,
              width: webRTCStreamParameters.width,
              height: webRTCStreamParameters.height
            });
          };

          console.log("3", this._tryToPlay);
          if (this._tryToPlay) {
            this._logger.log("PLAYER IS ALREADY TRY TO PLAY");
            try {
              await this._lastPlayPromise;
            }
            catch (e) {
              console.error(e);
            }
            this._logger.log("DONE: PLAYER IS ALREADY TRY TO PLAY");
          }

          await letsPlay();
        }
      }
    };

    const status = {
      isStopPublished: false,
      webRTC: {
        done: false,
        fail: false,
        isStopPublished: false,
        error: {code: undefined, message: ""}
      },
      webSocket: {
        done: false,
        fail: false,
        isStopPublished: false,
        error: {code: undefined, message: ""}
      }
    };
    const fail = () => {
      if (status.isStopPublished) {
        return;
      }

      // TODO: check logic for sending stop event, do not send second stop event

      if (this._preferredConnectionType === MSEMediaPlayerConnection.AUTO
          && status.webSocket.fail && status.webRTC.fail) {
        this._transport = this._webSocketTransport;
        status.isStopPublished = true;
        this._publish("stop", status.webSocket.error.code, status.webSocket.error.message);
      } else
      if (this._preferredConnectionType === MSEMediaPlayerConnection.CLOUD
          && status.webSocket.fail) {
        status.isStopPublished = true;
        this._publish("stop", status.webSocket.error.code, status.webSocket.error.message);
      } else
      if (this._preferredConnectionType === MSEMediaPlayerConnection.LOCAL
          && status.webRTC.fail) {
        status.isStopPublished = true;
        this._publish("stop", status.webRTC.error.code, status.webRTC.error.message);
      }
    };

    // 1. init async WebRTC and WEbSocket transport
    // 2. start stream from first connected transport
    // 3. if play was started not from WebRTC, and we successfully connected to WebRTC
    // then close connection to WebSocket and start stream from WebRTC

    // TODO: check, when both WebRTC and WebSocket connection fail

    if (parameters.isUseFileTransport) {
      // TODO: correct later
      // this._transport = this._fileTransport;
    } else
    if (this._preferredConnectionType !== MSEMediaPlayerConnection.AUTO) {
      if (this._preferredConnectionType === MSEMediaPlayerConnection.CLOUD) {
        this._transport = this._webSocketTransport;
      } else
      if (this._preferredConnectionType === MSEMediaPlayerConnection.LOCAL) {
        this._transport = this._webRTCTransport;
      }
    }

    const initWebRTCConnection = async () => {
      // init WebRTC connection
      if (this._preferredConnectionType === MSEMediaPlayerConnection.AUTO
          || this._preferredConnectionType === MSEMediaPlayerConnection.LOCAL) {
        // await Utils.wait(1500);
        return this._initTransportStream(this._webRTCTransport, parameters)
          .then(async ({transport, mime, width, height}) => {
            status.webRTC.done = true;

            Object.assign(webRTCStreamParameters, {mime, width, height});

            isWebRTC = true;

            if (isWebSocket) {
              this._logger.warn("websocket defined");
              await done();
              return false;
            }

            await this._initStreamMIME({
              transport: transport,
              mime: mime,
              width: width,
              height: height
            });

            webRTCInit = true;
            await done();

            return true;
          })
          .then((result) => {
            this._logger.warn("webrtc", result);
          })
          .catch(({code, message}) => {
            this._logger.error(`webrtc [${code}] ${message}`);

            status.webRTC.fail = true;
            status.webRTC.error.code = code;
            status.webRTC.error.message = message;

            fail();
          });
      }
    };
    const initWebSocketConnection = async () => {
      // init WebSocket connection
      if (this._preferredConnectionType === MSEMediaPlayerConnection.AUTO
          || this._preferredConnectionType === MSEMediaPlayerConnection.CLOUD) {
        return this._initTransportStream(this._webSocketTransport, parameters)
          .then(async ({transport, mime, width, height}) => {
            status.webSocket.done = true;

            isWebSocket = true;

            if (isWebRTC) {
              this._logger.warn("webrtc defined");
              await done();
              return false;
            }

            await this._initStreamMIME({
              transport: transport,
              mime: mime,
              width: width,
              height: height
            });

            webSocketInit = true;
            await done();

            return true;
          })
          .then((result) => {
            this._logger.warn("websocket", result);
          })
          .catch(({ code, message }) => {
            this._logger.error(`websocket [${code}] ${message}`);

            status.webSocket.fail = true;
            status.webSocket.error.code = code;
            status.webSocket.error.message = message;

            fail();
          });
      }
    };

/*    const initFileTransport = async () => {
      console.log("initFileTransport");
      // init FileTransport connection
      return this._initTransportStream(this._fileTransport, parameters)
        .then(async ({transport, mime, width, height}) => {
          status.webSocket.done = true;

          isFileTransport = true;

          await this._initStreamMIME({
            transport: transport,
            mime: mime,
            width: width,
            height: height
          });

          await done();

          return true;
        })
        .then((result) => {
          this._logger.warn("file", result);
        })
        .catch(({code, message}) => {
          this._logger.error(`file [${code}] ${message}`);
          this._publish("stop", code, message);
        });
    };

    if (parameters.isUseFileTransport) {
      await initFileTransport();

      return;
    }*/

    const avatarId = await this._webRTCTransport.getAvatarId(this._obj);
    const isPreviousWebRTCSuccess = avatarId && this._webRTCTransport.isPreviousWebRTCSuccess(avatarId);

    // we should not check connection to websocket if previous connection to webrtc was successful
    if (this._preferredConnectionType === MSEMediaPlayerConnection.AUTO
        && isPreviousWebRTCSuccess) {
      await initWebRTCConnection();
    } else {
      await Promise.race([
        initWebRTCConnection(),
        initWebSocketConnection()
      ]);
    }
  };

  _initTransportStream(transport: WebSocketMediaTransport | WebRTCMediaTransport, parameters: TransportInitParameters): Promise<TransportInitResult> {
    this._publish("connecting", this.getConnectionType(transport) === MSEMediaPlayerConnection.LOCAL ? __("local access") : __("cloud access"));

    transport._initMIME = false;
    return new Promise((resolve, reject) => {
      const initTransport = async (transport: WebSocketMediaTransport | WebRTCMediaTransport) => {
        const onmessage = (transport: WebSocketMediaTransport | WebRTCMediaTransport, message: {data: ArrayBuffer | string}) => {
          // console.log(transport, message);
          if (!(message.data instanceof ArrayBuffer)) {
            let messageData: Message = {};
            try {
              messageData = JSON.parse(message.data)
            }
            catch (e) {
              this._logger.error(this._getTransportCaption(transport) + " invalid json");
            }

            if (messageData.hasOwnProperty("code")) {
              this._logger.error(`code ${this._getTransportCaption(transport)}< [${messageData.code}] ${messageData.error}`);
              // if (transport._initMIME) {
                this._publish("stop", messageData.code, messageData.error);
              // }

              if (!transport._initMIME) {
                reject({code: messageData.code, message: messageData.error});
              }
            } else
            if (messageData.hasOwnProperty("event")) {
              // TODO: add check when event came after last frame
              let event = messageData.event!;
              this._logger.log(`event ${this._getTransportCaption(transport)}< ${event}`);
              transport._endEvent = {
                time: transport._lastReceivedTime,
                event: event
              };

              // call for checking endEvent logic
              if (this._player
                  && this._player instanceof MSEPlayer
                  && this._player._videoFrame
                  && this._player._videoFrame.parameters.frame
              ) {
                this._player._videoFrame.parameters.frame(this._player._videoFrame.time);
              }
            } else
            if (messageData.hasOwnProperty("mime")) {
              this._logger.log(`mime ${this._getTransportCaption(transport)}<`, message.data);

              if (!messageData.size) {
                reject("stream size is undefined");
                return;
              }

              resolve({
                transport: transport,
                mime: messageData.mime!,
                width: messageData.size.width,
                height: messageData.size.height
              });
            }
          }

          if (transport._initMIME) {
            this.onmessage(transport, message);
          }
        };

        let parameters: TransportInitParameters = {
          obj: this._obj,
          isArchive: this._isArchive,
          alwaysConnect: this._preferredConnectionType !== MSEMediaPlayerConnection.AUTO,
          isLocal: this.getConnectionType(transport) === MSEMediaPlayerConnection.LOCAL,
          message: (message) => onmessage(transport, message)
        };

        if (transport && transport.isOpen()) {
          transport.setParameters(parameters);
          return;
        }

        await transport.init(parameters);
      };

      initTransport(transport)
        .then(async () => {
          await this._initStream(transport, parameters);

          const transportClose = async (code: number, error?: string) => {
            transport.setParameters({
              message: null,
              close: null
            });

            this._player && this._player.pause();

            // TODO: check for saving from not chunk start
            const parameters = Object.assign(this._lastInitParameters!, {startTime: this._currentTime});

            this.printErrorMessage(__("Connection closed, reconnecting"));

            // wait some time before restart connection to give remote service time to restart if it fails
            await Utils.wait(1000);

            await this._initAllTransportStreams(parameters);
          };

          transport.setParameters({
            close: transportClose
          });
        })
        .catch((e) => reject(e));
    });
  };

  async _initStream(transport: WebSocketMediaTransport | WebRTCMediaTransport, parameters: TransportInitParameters): Promise<void> {
    transport._skipChunks = true;

    if (!this._obj) {
      this._logger.error("_initStream: this._obj is undefined")
      throw new Error("_initStream: this._obj is undefined");
    }
    // send request
    let request: InitStreamRequest = {
      cmd: "init",
      objid: this._obj,
      streamid: this._streamid,
      ticket: transport instanceof FileTransport ? "" : await this._getTicket(transport),
      params: {
        // audio: true
        // fast: parameters.isUseFileTransport
      }
    };
    if (parameters.startTime && parameters.endTime) {
      request.params.startTime = timestampToPts(parameters.startTime);
      request.params.endTime = timestampToPts(parameters.endTime);
    }
    this._lastInitParameters = parameters;
    transport.send(request);

    transport._doPlay = true;
  };

  async _initStreamMIME({transport, mime, width, height}: {transport: WebSocketMediaTransport | WebRTCMediaTransport, mime: string, width: number, height: number}): Promise<void> {
    this._publish("buffering");

    transport._skipChunks = false;
    transport._isChunkSplitted = false;
    this._isFirstChunk = true;
    this._startTime = 0;

    transport._stream.mime = mime;
    transport._stream.width = width;
    transport._stream.height = height;

    this._transport = transport;

    this._metadataList = [];
    this._metadataListDraw = {};
    m.redraw();

    /**
     * @returns {Promise<void>}
     */
    const init = async () => {
      if (MediaSource.isTypeSupported(transport._stream.mime)) {
        if (!this._msePlayer) {
          throw new Error("init: this._player is undefined");
        }

        this._player = this._msePlayer;

        this.addRenderer("msePlayer", this._player.render.bind(this._player));

        await this._msePlayer.initMediaSource(transport._stream.mime);

        transport._initMIME = true;
      } else
      if (transport._stream.mime === "video/x-jpeg") {
        if (!this._imagePlayer) {
          throw new Error("init: this._player is undefined");
        }

        this._player = this._imagePlayer;

        this.addRenderer("imagePlayer", this._player.render.bind(this._player));

        transport._initMIME = true;
      } else
      if (transport._stream.mime.includes("hvc1") ) {
        const message = "Your browser or hardware does not support the H.265 codec. You can use our standalone application.";
        this._publish("stop", MSEMediaPlayerError.MSE_PLAYER_UNSUPPORTED_CODEC, message);
      } 
      else {
        this._publish("stop", 0, "Unsupported MIME type or codec: " + transport._stream.mime);
      }
    };

    await init();

    this._logger.log("doPlay", transport._doPlay);
    if (transport._doPlay) {
      transport._doPlay = false;

      // send play
      let request = {
        cmd: "play",
        params: {}
      };
      transport.send(request);

      this._inProgress = false;
      m.redraw();
    }
  }

  async onmessage(transport: WebSocketMediaTransport | WebRTCMediaTransport, message: {data: ArrayBuffer | string}) {
    if (!this._player) {
      return;
    }

    if (message.data instanceof ArrayBuffer) {
      if (transport._skipChunks) {
        return;
      }

      if (transport._isChunkSplitted && transport._currentChunk) {
        transport._currentChunk.set(new Uint8Array(message.data), transport._chunkPosition);
        transport._chunkPosition += message.data.byteLength;

        if (transport._chunkNumber === 0) {
          if (transport._chunkSize !== transport._chunkPosition) {
            // TODO: check this logic
            this._logger.error(`size ${transport._chunkSize} !== ${transport._chunkPosition}`);
          }
          transport._isChunkSplitted = false;
        } else {
          return;
        }
      } else {
        transport._currentChunk = new Uint8Array(message.data);
      }

      // if (!this._chunkList) this._chunkList = [];
      // if (this._chunkList.length <= 20) { this._chunkList.push(transport._currentChunk); m.redraw();}

      /*
      let chunkmd5 = md5(transport._currentChunk);
      // this._logger.log("now " + chunkmd5);
      // this._logger.log("was " + transport._chunkMD5);
      if (chunkmd5 !== transport._chunkMD5) {
          this._logger.log(`${chunkmd5} != ${transport._chunkMD5}`);
      }
      */

      let media: Media = {
        time: transport._lastReceivedTime,
        data: transport._currentChunk
      };
      this._player.addMedia(media, this._isFirstChunk);

      if (this._isFirstChunk) {
        this._isFirstChunk = false;
        // this._status = MSEMediaPlayerStatus.PLAY_START_2;
        console.log("1. try to play", this._tryToPlay);
        this._tryToPlay = true;
        if (this._lastPlayPromise) {
          this._logger.log("RESOLVE PREVIOUS PLAYER PLAY PROMISE");
          try {
            await this._lastPlayPromise;
          }
          catch (e) {
            console.error(e);
          }
          this._logger.log("DONE: RESOLVE PREVIOUS PLAYER PLAY PROMISE");
        }
        this._lastPlayPromise = new DeferredPromise();
        let isPlayEnded = false;
        this._lastPlayPromise
          .then(() => {
            console.log("then 2");
            this._publish("play", transport._stream);
          })
          .catch((e) => {
            console.log("catch 2");
            // Chrome throw exception for autoplay videos
            if (e) {
              this._logger.error(e.message);
            } else {
              console.error("ERROR ON PLAYER PLAY");
              console.trace();
            }
            this.pause();
          })
          .finally(async () => {
            console.log("2. try to play", this._tryToPlay);
            isPlayEnded = true;
            // await Utils.wait(1000);
            this._tryToPlay = false;
          });
        // if play not resolved, resolve related logic
        // TODO: remove this...
        const waitPlayInterval = 10000;
        this._playTimeoutId = setTimeout(() => {
          if (!isPlayEnded) {
            isPlayEnded = true;
            const message = `play not ended after ${waitPlayInterval/1000}sec`;
            this._lastPlayPromise?.reject({message: message});
            // TODO: restart stream from this point
          }
        }, waitPlayInterval);
        this._player.play(Number(process.env.PLAYER_DEFAULT_SPEED))
          .then(() => {
            console.log("then");
            this._lastPlayPromise?.resolve();
          })
          .catch((e) => {
            console.log("catch");
            this._lastPlayPromise?.reject(e);
          });
      }

      return;
    }

    let messageData: Message | null = null;
    try {
      messageData = JSON.parse(message.data)
    }
    catch (e) {
      this._logger.error(this._getTransportCaption(transport) + " invalid json");
    }

    if (!messageData) {
      return;
    }

    if (!transport._skipChunks && messageData.hasOwnProperty("c")) {
      transport._isChunkSplitted = true;
      if (typeof transport._chunkNumber !== "undefined" && transport._chunkNumber !== 0 && transport._chunkNumber - 1 !== messageData.c) {
        this._logger.error(`${transport._chunkNumber} - 1 !== ${messageData.c}`);
      }
      transport._chunkNumber = messageData.c;
      if (messageData.hasOwnProperty("s")) {
        transport._chunkPosition = 0;
        transport._chunkSize = messageData.s as number;
        transport._currentChunk = new Uint8Array(transport._chunkSize);
      }
      if (messageData.hasOwnProperty("md5")) {
        transport._chunkMD5 = messageData.md5;
      }
      if (messageData.hasOwnProperty("n")) {
        // this._logger.log(messageData.n);
        if (transport._chunkNumber2 && ((messageData.n as number) - transport._chunkNumber2 !== 1)) {
          this._logger.log(`${messageData.n} - ${transport._chunkNumber2} !== 1`);
        }
        transport._chunkNumber2 = messageData.n;
      }
    } else
    if (!transport._skipChunks && messageData.hasOwnProperty("pts")) {
      let pts: string = "";
      if (messageData.pts instanceof Array) {
        pts = messageData.pts[0];
        const isKey = messageData.key ?? false;
        this._player.addTime(messageData.pts.map(someDate => ptsToTimestamp(someDate)), isKey);
      } else {
        pts = messageData.pts!;
        this._player.addTime([ptsToTimestamp(pts)]);
      }

      let timestamp = ptsToTimestamp(pts);
      transport._lastReceivedTime = timestamp;

      if (messageData.key) {
        transport._endKeyTime  = timestamp;
      }

      if (this._startTime === 0) {
        this._startTime = timestamp;
        this._player.setStartTime(this._startTime);
      }

      if (messageData.hasOwnProperty("metadata")) {
        for (let i = 0; i < messageData.metadata!.length; i++) {
          this._metadataList.push(messageData.metadata![i]);
        }
      }
    }
  }

  /**
   * subscribe for event
   *
   * @param {string} eventName event name
   * @param {Function} callback function
   */
  subscribe(eventName: MSEMediaPlayerEventName, callback: MSEMediaPlayerEvent[MSEMediaPlayerEventName]) {
    // if (!this._player) return false;
    if (!eventName) return false;

    if (!this._topics.has(eventName)) {
      this._logger.error(`Unknown topic '${eventName}'`);
      return false;
    }

    this._topicsEvents.on(eventName, callback.bind(this));

    return true;
  }

  /**
   * subscribe for single event
   *
   * @param {string} eventName event name
   * @param {Function} callback function
   */
  once(eventName: MSEMediaPlayerEventName, callback: MSEMediaPlayerEvent[MSEMediaPlayerEventName]) {
    if (!eventName) return false;

    if (!this._topics.has(eventName)) {
      this._logger.error(`Unknown topic '${eventName}'`);
      return false;
    }

    this._topicsEvents.once(eventName, callback.bind(this));

    return true;
  }

  /**
   * unsubscribe from event
   *
   * @param {string} eventName event name
   */
  unsubscribe(eventName: MSEMediaPlayerEventName): boolean {
    if (!eventName) return false;

    if (!this._topics.has(eventName)) {
      this._logger.error(`Unknown topic '${eventName}'`);
      return false;
    }

    this._topicsEvents.removeAllListeners(eventName);

    return true;
  }

  _unsubscribeAll(eventName: MSEMediaPlayerEventName) {
    if (!eventName) return false;

    if (!this._topics.has(eventName)) {
      this._logger.error(`Unknown topic '${eventName}'`);
      return false;
    }

    this._topicsEvents.removeAllListeners(eventName);

    return true;
  }

  /**
   * publish event from media player object
   *
   * @param {string} eventName
   * @param {...*} data
   */
  _publish(eventName: MSEMediaPlayerEventName, ...data: Parameters<MSEMediaPlayerEvent[MSEMediaPlayerEventName]>): boolean {
    // if (!this._player) return false;
    if (!eventName) return false;

    if (!this._topics.has(eventName)) {
      this._logger.error(`Unknown topic '${eventName}'`);
      return false;
    }

    this._topicsEvents.emit(eventName, ...data);

    return true;
  }

  isArchive(): boolean {
    return this._isArchive;
  }

  isPlay(): boolean {
    return this._status === MSEMediaPlayerStatus.PLAY;
  }

  getObj(): string | null {
    return this._obj;
  }

  async stepForward() {
    this.pause();

    let boundary = await this._boundary("next");
    this.play(this._obj, boundary);
  }

  async stepBackward() {
    this.pause();

    let boundary = await this._boundary("prev");
    this.play(this._obj, boundary);
  }

  async _boundary(find: "next" | "prev"): Promise<number> {
    if (!this._currentTime) {
      return 0;
    }

    /*
    const origin = getOrigin();
    let settings = {
      url: origin + "/api/cm/coverage/boundary",
      type: "GET",
      cache: false,
      dataType: "json",
      data: {
        obj: this._obj,
        streamNumber: this._streamid,
        startTime: Math.round(this._currentTime / 1000) || Math.round(Date.now() / 1000),
        find: find
      }
    };*/
    try {
      // let response = await Utils.promisifyAjax($.ajax(settings));
      // return response.boundary * 1000;
    }
    catch (e) {
      if (e instanceof AjaxError) {
        Log.error(e.message);
      } else {
        throw e;
      }
    }

    return 0;
  }

  async setStreamId(streamid: MSEMediaPlayerStream) {
    this._streamid = streamid;

    if (this._status === MSEMediaPlayerStatus.PLAY) {
      this.pause();

      this.printErrorMessage(__("Init stream"));

      // wait for connection close
      // TODO: add once subscription for close event
      await Utils.wait(2000);
      this.play();
    }
  }

  showProgressIndicator() {
    this._inProgress = true;

    m.redraw();
  }

  hideProgressIndicator() {
    this._inProgress = false;

    m.redraw();
  }

  // TODO: specify types
  addRenderer(name: string, renderer: (source: CanvasImageSource, target: CanvasRenderingContext2D) => any | null) {
     this._renderList[name] = renderer;
  }

  _initRender() {
    this._context = this._canvas ? this._canvas.getContext("2d", {alpha: false}) : null;
  }

  _render() {
    if (this._stopRender) {
      return;
    }

    let source = null;
    let target = this._context;
    if (this._status === MSEMediaPlayerStatus.PLAY) {
      for (let name in this._renderList) {
        source = this._renderList[name](source, target);
      }
    }
    requestAnimationFrame(() => this._render());
  }

  redraw() {
    this._player?.redraw();
  }

  getCanvas(): HTMLCanvasElement | null {
    return this._canvas;
  }

  getDrawNode(): HTMLVideoElement | HTMLCanvasElement | null {
    if (!this._node) {
      return null;
    }

    let node: HTMLVideoElement | HTMLCanvasElement | null = this._node.querySelector("video");
    if (this._transport && this._transport._stream && this._transport._stream.mime === "video/x-jpeg") {
      node = this._node.querySelector("canvas");
    }

    return node;
  }

  getMetaDataNode(): HTMLDivElement | null {
    if (!this._node) {
      return null;
    }
    return this._node.querySelector<HTMLDivElement>(".metadata");
  }

  async _getTicket(transport: WebSocketMediaTransport | WebRTCMediaTransport): Promise<string> {
    const {ticket} = await this._api.getMediaTicket({cameraid: this._obj, isLocal: this.getConnectionType(transport) === MSEMediaPlayerConnection.LOCAL});
    return ticket;
  }

  clearTimers(): void {
    if (this._reconnectTimeoutId) {
      clearTimeout(this._reconnectTimeoutId);
      this._reconnectTimeoutId = undefined;
    }
    if (this._playTimeoutId) {
      clearTimeout(this._playTimeoutId);
      this._playTimeoutId = undefined;
    }
  }

  getTime(): number | undefined {
    return this._currentTime;
  }
}
