import m from "mithril";
import {MetadataVisual, MSEMediaPlayerMetadataType, VisualPrimitives} from "@solid/types";

const conf = {
  "visual": {
    "ttl": 2
  },
  "trail": {
    "points_max": 50,
    "time_max": 5,
    "ttl": 2,
    "rgba": "#00FF0077",
    "width": 2
  },
  "box": {
    "ttl": 3, // sec
    "rgba_neutral": "#00ff007d",  // trig=false
    "rgba_triggered": "#ff00007d", // trig=true
    "rgba_background_neutral": "#00ff0020",
    "rgba_background_triggered": "#ff000020"
  },
  "text": {
    "ttl": 3,
    "font-size": "13px",
    "rgba": "#000000ff",
    "rgba_background_neutral": "#00ff00e0",
    "rgba_background_triggered": "#ff0000e0"
  }
};

/*
const a: Message = {
  "pts": ["20210812001513.883127"],
  "metadata": [
    {
      "kind": "md",
      "list": {
        "3.1": {
          "trig": true,
          "visual": {
            "Bounding box": [
              {"polygon": [3071, 52905, 1582, 54726, 93, 54726, 93, 56546, 93, 58367, 93, 60187, 1582, 60187, 3071, 60187, 4561, 62007, 6050, 62007, 7540, 63828, 9029, 63828, 10519, 63828, 12008, 63828, 13497, 63828, 14987, 63828, 16476, 63828, 17966, 63828, 19455, 63828, 20945, 63828, 22434, 63828, 23923, 63828, 25413, 63828, 26902, 63828, 28392, 62007, 29881, 63828, 31371, 63828, 32860, 63828, 34350, 63828, 35839, 63828, 37328, 63828, 38818, 63828, 40307, 62007, 41797, 62007, 43286, 62007, 44776, 62007, 46265, 62007, 47754, 60187, 49244, 58367, 49244, 56546, 49244, 54726, 47754, 52905, 46265, 54726, 44776, 54726, 43286, 54726, 41797, 54726, 40307, 54726, 38818, 52905, 37328, 54726, 35839, 54726, 34350, 54726, 32860, 54726, 31371, 54726, 29881, 54726, 28392, 52905, 26902, 52905, 25413, 52905, 23923, 54726, 23923, 56546, 22434, 58367, 20945, 58367, 19455, 56546, 17966, 56546, 16476, 54726, 16476, 56546, 14987, 58367, 13497, 58367, 12008, 60187, 10519, 60187, 9029, 60187, 7540, 58367, 7540, 56546, 6050, 54726, 4561, 52905]},
              {"polygon": [32860, 9215, 32860, 11036, 34350, 11036, 35839, 11036, 37328, 11036, 35839, 9215, 34350, 11036]}
            ]
          }
        }
      },
      "pts": "20210812001513.883127",
      "type": "result",
      "vae": "md",
      "ver": 6
    }
  ]
};
*/

export type MetadataRenderVisual = MetadataVisual & {
  [id: string]: {
    time: number
  }
}

export type MetaDataRenderAttrs = {
  list: MetadataRenderVisual,
  time: number,
  metadataType: MSEMediaPlayerMetadataType,
  size: {width: number, height: number}
}

const MetaDataRender = (): m.Component<MetaDataRenderAttrs> => {
  const format = function (template: string, params: Record<string, any>): string {
    let tpl = template.replace(/\${(?!this\.)/g, "${this.");
    let tpl_func = new Function(`return \`${tpl}\``);

    return tpl_func.call(params);
  };

  let canvas: HTMLCanvasElement | null = null;
  const textWidth = function (text: string, font: string): number {
    if (!canvas) {
      canvas = document.createElement('canvas');
      let docFragment = document.createDocumentFragment();
      docFragment.appendChild(canvas);
    }
    let context = canvas.getContext("2d");
    if (!context) return 0;
    if (font) context.font = font;
    return context.measureText(text).width;
  };

  const step = 1.1; // em
  const convertText = (textList: string[], id: string, opacity: number, triggered: boolean, x: number, y: number, width: number, height: number) => {
    let maxWidth = 0;
    let index = 0;

    const resultText = m("text", {
      x: 0,
      y: 0,
      opacity: opacity
    }, textList.map((text) => {
      let width = textWidth(text, "13px Verdana");
      if (width > maxWidth) {
        maxWidth = width;
      }

      index += step;
      return m("tspan", {
        "data-id": id,
        x: 5,
        y: `${index}em`
      }, text);
    }));

    return m("svg", {
      viewBox: `0 0 ${width} ${height}`,
      preserveAspectRatio: "none",
      x,
      y,
      width: 65535,
      height: 65535,
    }, [
      m(triggered ? "rect.filled-triggered" : "rect.filled-neutral", {
        "data-id": id,
        x: 1,
        y: 1,
        width: maxWidth + 10,
        height: `${index}em`,
        opacity: opacity
      }),
      resultText
    ])
  };

  return {
    view: ({attrs}) => {
      return attrs.metadataType !== MSEMediaPlayerMetadataType.NONE && Object.keys(attrs.list).length > 0 && m("svg", {
        viewBox: "0 0 65535 65535",
        preserveAspectRatio: "none",
        // chrome z-order fix: add position: "relative" to metadata block
        style: {width: "100%", height: "100%", display: "block", position: "relative"}
      }, [
        m("style", {type: "text/css"},
          "rect.neutral {stroke: " + conf.box.rgba_neutral + "; stroke-width: 200; fill: none;}" +
          "rect.triggered {stroke: " + conf.box.rgba_triggered + "; stroke-width: 200; fill: none;}" +
          "rect.filled-neutral {stroke: #00AA00; stroke-width: 1; fill: " + conf.text.rgba_background_neutral + ";}" +
          "rect.filled-triggered {stroke: #00AA00; stroke-width: 1; fill: " + conf.text.rgba_background_triggered + ";}" +
          "text {fill: " + conf.text.rgba + "; font-size: 13px; font-family: Verdana, FontAwesome;}" +
          "polygon.neutral {stroke: " + conf.box.rgba_neutral + "; stroke-width: 200; fill: " + conf.box.rgba_background_neutral + ";}" +
          "polygon.triggered {stroke: " + conf.box.rgba_triggered + "; stroke-width: 200; fill: " + conf.box.rgba_background_triggered + ";}"
        ),
        Object.entries(attrs.list).map(([id, element]) => {
          // show for (ttl - 1) sec time and then fadeout
          const diff = attrs.time - (element.time + (conf.box.ttl - 1) * 1000);
          const opacity = 1 - (diff < 0 ? 0 : diff) / (conf.box.ttl * 1000);
          if (opacity <= 0) {
            delete attrs.list[id];
            return;
          }

          const visual = element.visual ?? null;
          const opt = visual?.opt ?? null;
          const triggered = element.trig ?? false;

          if (!triggered && attrs.metadataType === MSEMediaPlayerMetadataType.TRIGGERED) {
            return;
          }

          /*
          let elementDate = new Date(element.time);
          let elementTime = String(elementDate.getHours()).padStart(2, "0")
                            + ":" + String(elementDate.getMinutes()).padStart(2, "0")
                            + ":" + String(elementDate.getSeconds()).padStart(2, "0")
                            + "." + String(elementDate.getMilliseconds()).padStart(3, "0");
          let textList = [elementTime];
          */

          let textList: string[] = [];

          let resultVisual = visual && Object.entries(visual).map(([label, list]) => {
            if (label === "opt") {
              return;
            }

            return list && (list as VisualPrimitives[]).map((element) => {
              let box = element.box ?? null;
              let polygon = element.polygon ?? null;
              // let emotions = element.emotions ?? null;

              if (element.text) {
                const list = Array.isArray(element.text) ? element.text : [element.text];
                for (let i = 0; i < list.length; i++) {
                  const text = format(list[i], element);
                  if (text) {
                    textList.push(text);
                  }
                }
              }

              return [
                box && m(triggered ? "rect.triggered" : "rect.neutral", {
                  "data-id": id,
                  x: box[0],
                  y: box[1],
                  width: box[2],
                  height: box[3],
                  opacity: opacity
                }),
                polygon && m(triggered ? "polygon.triggered" : "polygon.neutral", {
                  "data-id": id,
                  opacity: opacity,
                  points: polygon.reduce((result, current, index) => result + (index % 2 === 0 ? " " : ",") + current, "")
                }),
              ];
            });
          });

          let resultText = null;
          if (textList.length > 0 && opt?.box) {
            resultText = convertText(textList, id, opacity, triggered, opt.box[0] + opt.box[2], opt.box[1], attrs.size.width, attrs.size.height);
          }

          return [
            resultVisual,
            resultText
          ];
        })
      ]);
    }
  }
};

export {MetaDataRender};
