const {bufferToBase64, base64ToBuffer, ptsToTimestamp, ptsToDate, timestampToPts} = require("@solid/libs");
const {interpret, Machine} = require("xstate");
const fs = require("fs");
const readline = require("readline");
const path = require("path");

const initEvent = {
	type: "CMD_INIT",
	objid: "",
	streamid: 1,
	ticket: "",
	params: {
		startTime: undefined, // pts
		endTime: undefined, // pts
		fast: false
	}
};

const playEvent = {
	type: "CMD_PLAY",
	cmd: "play",
	params: {}
};

const pauseEvent = {
	type: "CMD_PAUSE",
	cmd: "pause"
};

const idleEvent = {
	type: "IDLE"
};

const mimeEvent = {
	type: "MIME",
	mime: "",
	size: {}
};

const codeEvent = {
	type: "CODE",
	code: undefined,
	error: ""
};

const waitEvent = {
	type: "WAIT"
};

const ptsEvent = {
	type: "PTS",
	pts: [],
	key: false,
	metadata: {},
	init_segment_size: 0
};

const eventEvent = {
	type: "EVENT",
	event: "" // EOA | EOS | EOC
}

const binaryEvent = {
	type: "BINARY",
	data: ""
};

const DATA_DIR = process.env.DATA_DIR || path.join(__dirname, "data");

export class FileTransport {
	constructor(name = "") {
		const instanceName = name;

		this.instanceName = instanceName;

		this._isOpen = false;

		this._isDownload = false;

		/**
		 *
		 * @type {null|{width: number, height: number, mime: string}}
		 * @private
		 */
		this._stream = null;

		/**
		 *
		 * @type {{isArchive: boolean, obj: null|UUID, message: null|function, error: null|function, close: null|function, open: null|function}}
		 * @private
		 */
		this._parameters = {
			obj: null,
			isArchive: true,
			open: null,
			close: null,
			message: null,
			error: null
		};

		this._stream = {
			mime: "",
			width: 0,
			height: 0
		};

		this._dataList = [];
		this._isLoop = false;

		this._readDataList = {};

		const config = {
			id: "streamer",
			context: {
				mime: "",
				size: {},
				transport: this,
				currentPts: 0,
				lastKeyPTS: undefined,

				isInitSegmentSent: false,
				isFirstPTS: false,
				isFirstBinary: false,
				initSegment: undefined,

				initSegmentSize: 0,

				isWriter: false
			},
			on: {
				CODE: "code",
				CMD_PAUSE: "idle"
			},
			initial: "idle",
			states: {
				idle: {
					entry: ["idle"],
					on: {
						CMD_INIT: "init"
					}
				},
				code: {
					entry: ["code"],
					always: "idle"
				},
				init: {
					entry: ["init"],
					on: {
						MIME: "mime"
					}
				},
				mime: {
					entry: ["addMime"],
					on: {
						CMD_PLAY: "data"
					}
				},
				event: {
					entry: ["event"],
					always: "idle"
				},
				data: {
					initial: "wait",
					on: {
						EVENT: "event"
					},
					states: {
						wait: {
							entry: ["wait"],
							on: {
								PTS: {
									target: "pts"
								}
							}
						},
						pts: {
							entry: ["addPts"],
							on: {
								PTS: {
									target: "pts"
								},
								BINARY: [
									{
										target: "binary",
										cond: "isWriter"
									},
									{
										target: "binaryWait"
									}
								]
							}
						},
						binary: {
							entry: ["addBinary"],
							always: "wait"
						},
						binaryWait: {
							entry: ["addBinary"],
							after: {
								1: "wait"
							}
						}
					}
				}
			}
		};

		const writerOptions = {
			guards: {
				isWriter: () => true
			},
			actions: {
				idle: (context, event) => {
					// console.log(instanceName, "idle");
				},
				init: (context, event) => {
					// console.log(instanceName, "init");

					context.obj = event.objid;
					context.streamid = event.streamid;
					context.currentPts = 0;

					context.isFirstPTS = false;
					context.isFirstBinary = false;
					context.initSegment = undefined;

					const dir = context.transport.getStreamDir(event.objid);

					context.transport.mkdir(dir, true);
				},
				code: (context, event) => {
					console.error("code", event.code, event.error);
				},
				addMime: (context, event) => {
					// console.log(instanceName, "addMime");

					context.mime = event.mime;
					context.size = event.size;
				},
				wait: (context, event) => {
					// console.log(instanceName, "wait");
				},
				addPts: (context, event) => {
					// console.log(instanceName, "addPts", context);

					const {type, init_segment_size, ...eventInfo} = event;

					if (!context.isFirstPTS) {
						context.initSegmentSize = init_segment_size;
						context.isFirstPTS = true;
					}

					context.transport._dataList.push({info: {...eventInfo}, canBeStored: false, data: null});
				},
				addBinary: (context, event) => {
					// console.log(instanceName, "addBinary");

					let data = event.data;
					if (!context.isFirstBinary) {
						context.isFirstBinary = true;
						context.initSegment = event.data.slice(0, context.initSegmentSize);

						data = event.data.slice(context.initSegmentSize);
					}

					const lastDataElement = context.transport._dataList[context.transport._dataList.length - 1];
					lastDataElement.data = data;
					lastDataElement.canBeStored = true;

					context.transport.store(context);
				},
				event: (context, event) => {
					console.log(instanceName, "event", event.event);

					// remove not full received elements
					if (context.transport._dataList.length > 0) {
						const lastDataElement = context.transport._dataList[context.transport._dataList.length - 1];
						if (!lastDataElement.canBeStored) {
							context.transport._dataList.pop();
						}
					}

					context.transport.store(context);
				}
			}
		};

		const readerOptions = {
			guards: {
				isWriter: () => false
			},
			actions: {
				idle: (context, event) => {
					console.log(instanceName, "idle");
				},
				init: async (context, event) => {
					console.log(instanceName, "init");

					context.obj = event.objid;
					context.streamid = event.streamid;
					context.transport._readDataList = {};
					context.isInitSegmentSent = false;
					context.initSegment = undefined;

					console.log("parameters", context.transport._parameters);

					if (!event.params.startTime) {
						this._streamerService.send({...codeEvent, code: 0, error: "startTime not specified"});
						return;
					}

					context.currentPts = event.params.startTime;

					try {
						const {initSegment, initSegmentSize, ...chunkInfo} = await context.transport.getChunkInfo(context.currentPts);

						context.initSegment = base64ToBuffer(initSegment);

						this._streamerService.send({...mimeEvent, ...chunkInfo});
					}
					catch (e) {
						this._streamerService.send({...codeEvent, code: 10100/*MSEMediaPlayerError.VIDEO_UNAVAILABLE*/, error: "Requested video fragment is not available in videoarchive (time gap)"});
					}
				},
				code: (context, event) => {
					console.log(instanceName, "code", event.code, event.error);

					context.transport._parameters.message && context.transport._parameters.message({data: JSON.stringify(event)});
				},
				addMime: (context, event) => {
					console.log(instanceName, "addMime", context);
					context.mime = event.mime;
					context.size = event.size;

					context.transport._parameters.message && context.transport._parameters.message({data: JSON.stringify(event)});
				},
				wait: async (context, event) => {
					// console.log(instanceName, "wait", event);

					if (!context.currentPts) {
						console.error("context.currentPts not defined");
					}

					let start = context.transport.getChunkStart(context.currentPts);
					/*if (context.transport._readDataList[start] && context.transport._readDataList[start].list.length === 0) {
						const next = timestampToPts(ptsToTimestamp(start) + 30000);
						if (!await this.isChunkExist(next)) {
							this._streamerService.send({...eventEvent, event: "EOA"});
							return;
						}

						start = next;
					}*/

					if (!context.transport._readDataList[start]) {
						const data = await context.transport.getChunkData(context.currentPts);

						context.transport._readDataList[data.start] = data;
					}

					if (context.transport._readDataList[start] && context.transport._readDataList[start].list.length > 0) {
						const {info, data} = context.transport._readDataList[start].list.shift();
						let buffer = base64ToBuffer(data);

						this._streamerService.send({...ptsEvent, ...info});

						if (!context.isInitSegmentSent) {
							context.isInitSegmentSent = true;

							// combine init segment and first key segment before sending
							const tmp = new Uint8Array(context.initSegment.byteLength + buffer.byteLength);
							tmp.set(new Uint8Array(context.initSegment), 0);
							tmp.set(new Uint8Array(buffer), context.initSegment.byteLength);

							buffer = tmp.buffer;
						}

						this._streamerService.send({...binaryEvent, data: buffer});
					} else {
						// checkNext
						const next = timestampToPts(ptsToTimestamp(start) + 30000);
						if (await this.isChunkExist(next)) {
							this._streamerService.send({...eventEvent, event: "EOC"});
						} else {
							this._streamerService.send({...eventEvent, event: "EOA"});
						}
					}
				},
				addPts: (context, event) => {
					// console.log(instanceName, "addPts");

					context.transport._parameters.message && context.transport._parameters.message({data: JSON.stringify(event)});
				},
				addBinary: (context, event) => {
					// console.log(instanceName, "addBinary");

					context.transport._parameters.message && context.transport._parameters.message({data: event.data});
				},
				event: (context, event) => {
					console.log(instanceName, "event", event.event);

					context.transport._parameters.message && context.transport._parameters.message({data: JSON.stringify(event)});
				}
			}
		};

		const options = name === "reader" ? readerOptions : writerOptions;
		const streamerMachine = Machine(config, options);

		this._parameters.open && this._parameters.open();

		this._context = {
			obj: null,
			streamid: null,
			transport: this
		};
		this._streamerService = interpret(streamerMachine.withContext(this._context))
			.onEvent((event) => {
				// process.env.PLAYER_LOG === "DEBUG" && console.log(instanceName, "streamer+", event);
			})
			.onTransition((state) => {
				// process.env.PLAYER_LOG === "DEBUG" && console.log(instanceName, "streamer>", state.value);
			})
			.start();
	}

	async init(parameters) {
		this.setParameters(parameters);

		// console.log("getCoverage");
		// const coverage = await this.getCoverage(this._parameters.obj);
		// console.log("coverage", coverage);

		if (this.isOpen()) {
			return;
		}

		console.log("check dir", DATA_DIR);
		await this.mkdir(DATA_DIR);
		console.log("end check dir");

		this._isOpen = true;
	}

	setParameters(parameters) {
		this._parameters = Object.assign(this._parameters, parameters);
	}

	/**
	 *
	 * @param {{}} data
	 */
	send(data) {
		console.log(this.instanceName, "send", data);

		if ("cmd" in data) {
			if (data.cmd === "init") {
				this._streamerService.send({...initEvent, ...data});
			} else
			if (data.cmd === "play") {
				this._streamerService.send({...playEvent, ...data});
			} else
			if (data.cmd === "pause") {
				this._streamerService.send({...pauseEvent, ...data});
			}
		}
	}

	/**
	 *
	 * @returns {boolean}
	 */
	isOpen() {
		return this._isOpen;
	}

	/**
	 *
	 * @param {boolean} [isCallback]
	 */
	close(isCallback = true) {
		this._parameters.close && this._parameters.close();
	}

	list() {
		return this._dataList;
	}

	monitor() {
		if (!this._isDownload) {
			return;
		}

		setTimeout(this.monitor.bind(this), 1000);
	}

	async store(context) {
		if (this._isLoop) {
			return;
		}

		this._isLoop = true;

		while (this._dataList.length > 0) {
			if (this._dataList.length === 1 && !this._dataList[0].canBeStored) {
				break;
			}

			const data = this._dataList.shift();
			const pts = data.info.pts[0];
			const isKey = data.info.key;
			if (isKey) {
				context.lastKeyPts = pts;
			}

			const chunkDir = this.getChunkDir(context.lastKeyPts);
			const chunkPath = path.join(chunkDir, `chunk.json`);
			const isExists = await this.isDirExists(chunkDir);
			if (!isExists) {
				console.warn("isChunkExist", isExists, chunkDir);
				await this.mkdir(chunkDir, true);
				const info = {
					mime: context.mime,
					size: context.size,
					initSegment: bufferToBase64(context.initSegment)
				};
				await fs.promises.writeFile(path.join(chunkDir, "info.json"), JSON.stringify(info));
			}

			// skip empty frame
			if (!data.data) {
				continue;
			}

			const info = {
				info: data.info,
				data: bufferToBase64(data.data)
			};
			await fs.promises.appendFile(chunkPath, JSON.stringify(info) + "\n");
		}

		this._isLoop = false;

		// console.log(this.instanceName, "datalist", this._dataList.length);
	}

	async stat(path) {
		try {
			return await fs.promises.stat(path);
		}
		catch (e) {
			// console.warn("stat", path);
			throw e;
		}
		/*
		return new Promise((resolve, reject) => {
			fs.stat(path, (error, stats) => {
				if (error) {
					console.error("stat", path);
					reject(error);
				} else {
					resolve(stats);
				}
			});
		});
		*/
	}

	async isDirExists(dir) {
		let isExists = false;

		try {
			await this.stat(dir);
			isExists = true;
		}
		catch (e) {
		}

		return isExists;
	}

	async mkdir(dir, recursive = false) {
		if (!await this.isDirExists(dir)) {
			console.log("mkdir", dir);
			await fs.promises.mkdir(dir, {recursive: recursive});
		}
	}

	getStreamDir(obj) {
		return path.join(DATA_DIR, obj);
	}

	getChunkDir(pts) {
		const {obj, streamid} = this._context;
		const {year, month, date, hour, minute, second} = this.parsePts(pts);
		const chunkStart = second < 30 ? 0 : 30;

		const dir = this.getStreamDir(obj);
		// const chunkDir = path.join(dir, String(year), `${String(month).padStart(2, "0")}${String(date).padStart(2, "0")}`, `${String(hour).padStart(2, "0")}${String(minute).padStart(2, "0")}`, String(chunkStart));
		// <storage_path>/<objid>/YYMMDD/hh-<stream_id>/YYMMDDhhmmss.<format>
		const chunkDir = path.join(dir, `${String(year)}${String(month).padStart(2, "0")}${String(date).padStart(2, "0")}`, `${String(hour).padStart(2, "0")}-${String(streamid).padStart(2, "0")}`, `${String(year)}${String(month).padStart(2, "0")}${String(date).padStart(2, "0")}${String(hour).padStart(2, "0")}${String(minute).padStart(2, "0")}${String(chunkStart).padStart(2, "0")}`);
		return chunkDir;
	}

	getChunkPath(pts) {
		const chunkDir = this.getChunkDir(pts);
		const chunkPath = path.join(chunkDir, `${pts}.json`);
		return chunkPath;
	}

	async isChunkExist(pts) {
		const chunkDir = this.getChunkDir(pts);
		const isExists = await this.isDirExists(chunkDir);
		console.warn("isChunkExist", isExists, chunkDir);
		return isExists;
	}

	async getChunkInfo(pts) {
		const chunkDir = this.getChunkDir(pts);
		if (!await this.isDirExists(chunkDir)) {
			throw new Error(`${chunkDir} does not exists`);
		}

		const infoString = await fs.promises.readFile(path.join(chunkDir, "info.json"), {encoding: "utf-8"});
		const info = JSON.parse(infoString);
		return info;
	}

	getChunkStart(pts) {
		const date = ptsToDate(pts);
		date.millisecond = 0;
		const chunkStart = date.getSeconds() < 30 ? 0 : 30;
		date.setUTCSeconds(chunkStart);

		return timestampToPts(date.getTime());
	}

	async getChunkData(pts) {
		const time = performance.now();

		const chunkStart = this.getChunkStart(pts);
		const chunkDir = this.getChunkDir(chunkStart);

		const chunkPath = path.join(chunkDir, "chunk.json");
		const readStream = fs.createReadStream(chunkPath);
		const rl = readline.createInterface({
			input: readStream,
			crlfDelay: Infinity
		});

		let frameList = [];
		let firstKeyPTS = "";
		for await (const line of rl) {
			const frame = JSON.parse(line);
			if (!firstKeyPTS && frame.info.key) {
				firstKeyPTS = frame.info.pts[0];
			}
			if (!firstKeyPTS) {
				continue;
			}

			// skip empty frame
			if (!frame.data) {
				continue;
			}

			frameList.push(frame);
		}

		// find key frame closest to pts
		if (ptsToTimestamp(pts) > ptsToTimestamp(firstKeyPTS)) {
			let prevKeyFrameIndex = undefined;
			for (let i = 0; i < frameList.length; i++) {
				const frame = frameList[i];
				if (prevKeyFrameIndex && frame.info.key
				    && ptsToTimestamp(frameList[prevKeyFrameIndex].info.pts[0]) <= ptsToTimestamp(pts)
					&& ptsToTimestamp(pts) < ptsToTimestamp(frame.info.pts[0])
				) {
					break;
				}

				if (frame.info.key) {
					prevKeyFrameIndex = i;
				}
			}

			// remove all frames before pts
			if (prevKeyFrameIndex !== undefined) {
				frameList = frameList.slice(prevKeyFrameIndex);
			}
		}

		console.log(`getChunkData ${performance.now() - time}ms`);

		return {
			start: chunkStart,
			list: frameList
		};
	}

	static async getCoverage(obj) {
		const time = performance.now();

		const list = {};

		try {
			let cameraIdList = await fs.promises.readdir(DATA_DIR, {encoding: "utf8"});
			cameraIdList = cameraIdList.filter((cameraId) => {
				return cameraId === obj;
			});
			if (cameraIdList.length === 0) {
				throw new Error(`obj=${obj} not found in ${DATA_DIR}`);
			}

			const cameraId = cameraIdList[0];

			const streamIdList = await fs.promises.readdir(path.join(DATA_DIR, cameraId), {encoding: "utf8"});
			for (const streamId of streamIdList) {
				const yearList = await fs.promises.readdir(path.join(DATA_DIR, cameraId, streamId), {encoding: "utf8"});
				for (const year of yearList) {
					const monthDateList = await fs.promises.readdir(path.join(DATA_DIR, cameraId, streamId, year), {encoding: "utf8"});
					for (const monthDate of monthDateList) {
						const timeList = await fs.promises.readdir(path.join(DATA_DIR, cameraId, streamId, year, monthDate), {encoding: "utf8"});
						for (const time of timeList) {
							const chunkList = await fs.promises.readdir(path.join(DATA_DIR, cameraId, streamId, year, monthDate, time), {encoding: "utf8"});
							for (const chunk of chunkList) {
								const ptsString = `${year}${monthDate}${time}${chunk.padStart(2, "0")}`;
								const timestamp = ptsToTimestamp(ptsString);
								list[timestamp] = {percentage: 100};
							}
						}
					}
				}
			}

			console.log(`getCoverage ${performance.now() - time}ms`)
		}
		catch (e) {
			console.warn(`getCoverage ${e.message}`);
		}

		return list;
	}

	parsePts(pts) {
		const ptsString = String(pts);
		let year = parseInt(ptsString.substr(2, 2));
		let month = parseInt(ptsString.substr(4, 2));
		let date = parseInt(ptsString.substr(6, 2));
		let hour = parseInt(ptsString.substr(8, 2));
		let minute = parseInt(ptsString.substr(10, 2));
		let second = parseInt(ptsString.substr(12, 2));
		let millisecond = parseInt(ptsString.substr(15, 3));

		return{
			year,
			month,
			date,
			hour,
			minute,
			second,
			millisecond
		}
	}

	/**
	 *
	 * @param {ArrayBuffer|{}} message
	 */
	message(message) {
		if (message.data instanceof ArrayBuffer) {
			this._streamerService.send({...binaryEvent, data: message.data});
		} else {
			let messageData = {};
			try {
				messageData = JSON.parse(message.data);
			}
			catch (e) {
				this._logger.error(this._getTransportCaption(transport) + " invalid json");
			}

			if ("code" in messageData) {
				this._streamerService.send({...codeEvent, ...messageData})
			} else
			if ("event" in messageData) {
				this._streamerService.send({...eventEvent, ...messageData})
			} else
			if ("mime" in messageData) {
				this._streamerService.send({...mimeEvent, ...messageData});
			} else
			if ("pts" in messageData) {
				this._streamerService.send({...ptsEvent, ...messageData})
			}
		}
	}

	read() {

	}
}
