import rfdc from 'rfdc';
import {Tolerance, UUID} from "@solid/types";

export class AjaxError extends Error {
  constructor({code = 0, error = ""}: {code: number, error: string}) {
    const message = `[${code}] ${error}`;

    super(message); // 'Error' breaks prototype chain here
    this.name = 'AjaxError';
    Object.setPrototypeOf(this, new.target.prototype); // restore prototype chain
  }
}

/**
 * get cookie value
 */
export function readCookie(name: string): string | null
{
  let nameEQ = name + "=";
  let cookies = document.cookie.split(';');
  for(let i = 0; i < cookies.length; i++)
  {
    let cookie = cookies[i];
    while (cookie.charAt(0) === ' ')
    {
      cookie = cookie.substring(1, cookie.length);
    }
    if (cookie.indexOf(nameEQ) === 0)
      return decodeURI(cookie.substring(nameEQ.length, cookie.length));
  }
  return null;
}

export function setCookie(name: string, value: string, expires?: number, path?: string, domain?: string, secure?: boolean)
{
  let expires_txt = "0";
  if (expires)
  {
    expires = expires * 1000;
    let today = new Date();
    today.setTime(today.getTime());
    let expires_date = new Date(today.getTime() + (expires));
    expires_txt = expires_date.toUTCString();
  }

  document.cookie = name + "=" + encodeURI(value) +
    ((expires) ? "; expires=" + expires_txt : "") +
    ((path) ? "; path=" + path : "") +
    ((domain) ? "; domain=" + domain : "") +
    ((secure) ? "; secure" : "");
}

export function getCookie(name: string, cookies: string[]) {
  for (const cookiePath of cookies) {
    const [cookie/*, path*/] = cookiePath.split("; ");
    const [key, value] = cookie.split("=");
    if (key === name) {
      return value;
    }
  }
  return undefined;
}

export function getToken(): string
{
  let token = readCookie("token");
  if (!token || token.match(/^([0-9A-Za-z])+$/) === null)
  {
    token = "";
  }
  return token;
}

export const Utils = {
  normalize: function (point: [number, number, number], width: number, height: number) {
    const N: [[number, number, number], [number, number, number], [number, number, number]] = [
      [2 / width, 0,          -1],
      [0,         2 / height, -1],
      [0,         0,           1]
    ];

    return this.multiply(N, point);
  },

  denormalize: function (point: [number, number, number], width: number, height: number): [number, number, number] {
    const dN: [[number, number, number], [number, number, number], [number, number, number]] = [
      [width / 2, 0,          width / 2],
      [0,         height / 2, height / 2],
      [0,         0,          1]
    ];

    return this.multiply(dN, point);
  },

  multiply: function (m: [[number, number, number], [number, number, number], [number, number, number]], v: [number, number, number]): [number, number, number] {
    const result: [number, number, number] = [0, 0, 0];

    result[0] = m[0][0] * v[0] + m[0][1] * v[1] + m[0][2] * v[2];
    result[1] = m[1][0] * v[0] + m[1][1] * v[1] + m[1][2] * v[2];
    result[2] = m[2][0] * v[0] + m[2][1] * v[1] + m[2][2] * v[2];

    return result;
  },

  uuid: function (): string {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
      const r = Math.random() * 16|0;
      const v = c === 'x' ? r : (r&0x3|0x8);
      return v.toString(16);
    });
  },

  shortUuid: function (uuid: string, withTail: boolean): string {
    return uuid.substring(0, 8) + (!withTail ? "" : ("..." + uuid.substring(uuid.length - 8)));
  },

  throttleToDraw: function (callback: (...args: any[]) => void): (...args: any[]) => void {
    let pending: number | null = null;
    return function(...args: any[]) {
      // @ts-ignore
      const context = this;
      if (pending === null) {
        pending = requestAnimationFrame(function() {
          pending = null;
          callback.apply(context, args);
        });
      }
    }
  },

  throttle: (callback: (...args: any[]) => void, limit: number): (...args: any[]) => void => {
    let lastFunc: NodeJS.Timeout;
    let lastRan: number;

    return function (...args: any[]) {
      // @ts-ignore
      const context = this;
      if (!lastRan) {
        callback.apply(context, args);
        lastRan = Date.now()
      } else {
        clearTimeout(lastFunc);
        lastFunc = setTimeout(function () {
          if ((Date.now() - lastRan) >= limit) {
            callback.apply(context, args);
            lastRan = Date.now();
          }
        }, limit - (Date.now() - lastRan));
      }
    }
  },

  debounce: (callback: (...args: any[]) => void, delay: number): (...args: any[]) => void => {
    let inDebounce: NodeJS.Timeout;
    return function (...args: any[]) {
      // @ts-ignore
      const context = this;
      globalThis.clearTimeout(inDebounce);
      inDebounce = globalThis.setTimeout(() => callback.apply(context, args), delay)
    }
  },

  wait: function (delay: number): Promise<any> {
    return new Promise((resolve) => {
      setTimeout(resolve, delay);
    });
  },

  webSocketCloseCodeMessage: function (code: number) {
    let codeMessage = {
      1000: "Normal Closure",
      1001: "Going Away",
      1002: "Protocol error",
      1003: "Unsupported Data",
      1004: "Reserved",
      1005: "No Status Rcvd",
      1006: "Abnormal Closure",
      1007: "Invalid frame payload data",
      1008: "Policy Violation",
      1009: "Message Too Big",
      1010: "Mandatory Ext.",
      1011: "Internal Error",
      1012: "Service Restart",
      1013: "Try Again Later",
      1014: "The server was acting as a gateway or proxy and received an invalid response from the upstream server.",
      1015: "TLS handshake"
    };

    return codeMessage[code]
  },

  promisifyAjax: function (deferred: JQuery.Promise<any>): Promise<any> {
    return new Promise((resolve, reject) => {
      deferred
        .fail(function(jqXHR){
          let code = jqXHR.status;
          let responseJSON = null;
          if (jqXHR.responseJSON) {
            responseJSON = jqXHR.responseJSON;
          } else {
            try {
                  responseJSON = JSON.parse(jqXHR.responseText);
              }
              catch (e) {}
          }
          let message = responseJSON ? responseJSON.error : jqXHR.statusText;
          reject(new AjaxError({code: code, error: message}));
        })
        .done(function(result: any){
          if (typeof result.code === "undefined" || result.code >= 200 && result.code < 300 || result.code === 304) {
            resolve(result);
          } else {
            reject(new AjaxError(result));
          }
        });
    });
  }
};

export function clone<T>(obj: T): T {
  const rfdcClone = rfdc();
  return rfdcClone(obj);
}

export function isElectron(): boolean {
  if (typeof navigator == "undefined" || !navigator.userAgent) {
    return false;
  }
  const userAgent = navigator.userAgent.toLowerCase();
  return userAgent.indexOf(' electron/') > -1;
}

export function isBrowser(): boolean {
  return typeof window !== 'undefined';
}

/**
 * NW.js / Electron is a browser context, but copies some Node.js objects; see
 * http://docs.nwjs.io/en/latest/For%20Users/Advanced/JavaScript%20Contexts%20in%20NW.js/#access-nodejs-and-nwjs-api-in-browser-context
 * https://www.electronjs.org/docs/api/process#processversionselectron-readonly
 * https://www.electronjs.org/docs/api/process#processtype-readonly
 *
 * @returns {boolean}
 */
export function isNode(): boolean {
  return typeof process === "object" &&
    process + "" === "[object process]" &&
    // @ts-ignore
    !process.versions.nw &&
    // @ts-ignore
    !(process.versions.electron && process.type && process.type !== "browser");
}

/**
 * convert YYYYMMDDHHMMSS.XXXXXX to Date
 */
export function ptsToDate(pts: number | string): Date {
  const ptsString = String(pts);
  let year = parseInt(ptsString.substr(0, 4));
  let month = parseInt(ptsString.substr(4, 2));
  let date = parseInt(ptsString.substr(6, 2));
  let hour = parseInt(ptsString.substr(8, 2));
  let minute = parseInt(ptsString.substr(10, 2));
  let second = parseInt(ptsString.substr(12, 2)) || 0;
  let millisecond = parseInt(ptsString.substr(15, 3)) || 0;

  let ptsDate = new Date();
  ptsDate.setUTCFullYear(year);
  ptsDate.setUTCMonth(month - 1);
  ptsDate.setUTCDate(date);
  ptsDate.setUTCHours(hour);
  ptsDate.setUTCMinutes(minute);
  ptsDate.setUTCSeconds(second);
  ptsDate.setUTCMilliseconds(millisecond);

  return ptsDate;
}

/**
 * convert YYYYMMDDHHMMSS.XXXXXX to unix timestamp
 *
 * @param pts
 * @returns timestamp ms
 */
export function ptsToTimestamp(pts: number | string): number {
  const date = ptsToDate(pts);
  return date.getTime();
}

/**
 * convert unix timestamp to YYYYMMDDHHMMSS.XXXXXX
 *
 * @param timestamp ms
 */
export function timestampToPts(timestamp: number): number {
  let date = new Date();
  date.setTime(timestamp);

  let pts = date.getUTCFullYear()
            + String(date.getUTCMonth() + 1).padStart(2, "0")
            + String(date.getUTCDate()).padStart(2, "0")
            + String(date.getUTCHours()).padStart(2, "0")
            + String(date.getUTCMinutes()).padStart(2, "0")
            + String(date.getUTCSeconds()).padStart(2, "0");
  return parseInt(pts);
}

export function bufferToBase64(buffer: ArrayBuffer): string {
  // var base64 = btoa(String.fromCharCode.apply(null, new Uint8Array(arrayBuffer)));
  // var base64 = btoa([].reduce.call(new Uint8Array(bufferArray),function(p,c){return p+String.fromCharCode(c)},''))
  // var base64 = btoa(new Uint8Array(arrayBuffer).reduce((data, byte) => data + String.fromCharCode(byte), ''));
  /*function arrayBufferToBase64( buffer, callback ) {
      var blob = new Blob([buffer],{type:'application/octet-binary'});
      var reader = new FileReader();
      reader.onload = function(evt){
          var dataurl = evt.target.result;
          callback(dataurl.substr(dataurl.indexOf(',')+1));
      };
      reader.readAsDataURL(blob);
  }*/
  /*function arrayBufferToBase64( buffer ) {
      var binary = '';
      var bytes = new Uint8Array( buffer );
      var len = bytes.byteLength;
      for (var i = 0; i < len; i++) {
          binary += String.fromCharCode( bytes[ i ] );
      }
      return window.btoa( binary );
  }*/
  /*function arrayBufferToBase64(ab){
      var dView = new Uint8Array(ab);   //Get a byte view
      var arr = Array.prototype.slice.call(dView); //Create a normal array
      var arr1 = arr.map(function(item){
        return String.fromCharCode(item);    //Convert
      });
      return window.btoa(arr1.join(''));   //Form a string
  }*/
  /*function draw(imgData, frameCount) {
      var r = new FileReader();
      r.readAsBinaryString(imgData);
      r.onload = function(){
          var img=new Image();
          img.onload = function() {
              cxt.drawImage(img, 0, 0, canvas.width, canvas.height);
          }
          img.src = "data:image/jpeg;base64,"+window.btoa(r.result);
      };
  }*/
  let binary = '';
  const bytes = new Uint8Array(buffer);
  const len = bytes.byteLength;
  for (let i = 0; i < len; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return window.btoa(binary);
}

export function base64ToBuffer(base64: string): ArrayBuffer {
  const binary_string = window.atob(base64);
  const len = binary_string.length;
  const bytes = new Uint8Array(len);
  for (let i = 0; i < len; i++) {
    bytes[i] = binary_string.charCodeAt(i);
  }
  return bytes.buffer;
}

type AlignBorderParameters = {
  startTime: number, // ms
  endTime?: number, // ms
  roundToSec?: boolean
}
/**
 * align border to chunk
 */
export function alignBorder({startTime, endTime = undefined, roundToSec = false}: AlignBorderParameters): {start: number, end: number} {
  // align time to 30 sec intervals
  let date = new Date(startTime);

  // align left border (0 or 30 sec)
  date.setUTCSeconds(date.getUTCSeconds() - date.getUTCSeconds() % 30, 0);
  let start = roundToSec ? Math.round(date.getTime() / 1000) : date.getTime();

  if (!endTime) {
    return {start, end: start};
  }

  // align right border (0 or 30 sec)
  date.setTime(endTime);
  date.setUTCSeconds(date.getUTCSeconds() + (30 - date.getUTCSeconds() % 30) % 30, 0);
  let end = roundToSec ? Math.round(date.getTime() / 1000) : date.getTime();

  return {start, end};
}

/**
 * align time to 30 sec intervals
 *
 * @param timestamp ms
 * @returns {number} ms
 */
export function alignTimestampToChunk(timestamp: number): number {
  let date = new Date(timestamp);
  date.setUTCSeconds(date.getUTCSeconds() - date.getUTCSeconds() % 30, 0);

  return date.getTime();
}

export function removeTrailingSlash(str: string): string {
  return str.endsWith("/") ? str.substring(0, str.length - 1) : str;
}

export function getOrigin(): string {
  // @ts-ignore
  return !globalThis.location || globalThis.location.origin.startsWith("file") ? removeTrailingSlash(process.env.API_BACKEND || "") : globalThis.location.origin;
}

export function getSnapshotURL(cameraId: UUID, timestampSec?: number, downscale: boolean = true, status: boolean = false, tolerance: Tolerance = Tolerance.EXACT) {
  const strTimeArg = tolerance !== Tolerance.EXACT && timestampSec
    ? `&start=${timestampSec - tolerance}&end=${timestampSec + tolerance}`
    : (timestampSec ? `&ts=${timestampSec}` : "");
  const cacheTimeArg = !timestampSec ? `&_=${Math.trunc(Date.now() / 1000)}` : "";
  const origin = getOrigin();
  return `${origin}/api/snapshot?objid=${cameraId}${downscale ? "&downscale" : ""}${status ? "&status" : ""}${strTimeArg}${cacheTimeArg}`;
}
