/**
 * @version $Id: log.js 44022 2018-06-25 11:50:42Z astarostin $
 * ------------------------------------------------------------------------------
 * Logging and alert message on screen
 * include <script scr="/sdi/lib/log.js"></script>
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2015
 * ------------------------------------------------------------------------------
 */

"use strict";

const initLog = function () {
	if (logInitiated) {
		return;
	}

	logInitiated = true;

	const styles =
		"#messages {position:fixed;top:0px;right:0px;width:350px;margin:0px;padding:2px;background:transparent;z-index:2000}" +
		"#messages div{color:#fff;padding:7px;margin-bottom:7px;opacity:0.85;}" +
		"#messages .error{background:#98001b}" +
		"#messages .info{background:#0d8529}" +
		"#messages .warning{background:#dd6; color:#333}" +
		"#messages .closeButton{cursor: pointer;float: right;height: 12px;line-height: 12px;margin: 0;padding: 0;text-align: center;vertical-align: middle;width: 12px;}";

	const addLoadEvent = (func: (this: GlobalEventHandlers, ev: Event) => any) => {
		const oldonload = window.onload;
		if (typeof window.onload !== "function") {
			window.onload = func;
		} else {
			window.onload = () => {
				if (oldonload) {
					// @ts-ignore
          oldonload();
				}
				// @ts-ignore
        func();
			};
		}
	};

	const import_style = (src: string) => {
		if ((src === null || src === undefined)) return;
		const imprt = document.createElement("style");
		imprt.setAttribute("type", "text/css");
		// @ts-ignore
    if (imprt.styleSheet) {
			// @ts-ignore
      imprt.styleSheet.cssText = src;
		} else {
			imprt.appendChild(document.createTextNode(src));
		}
		document.getElementsByTagName("head")[0].appendChild(imprt);
	};

	const addAll = () => {
		if (document.getElementById("messages")) {
			return;
		}

		const messageBox = document.createElement("div");
		messageBox.id = "messages";
		if (document.body.firstChild) {
			document.body.insertBefore(messageBox, document.body.firstChild);
		} else {
			document.body.appendChild(messageBox);
		}
		import_style(styles);
	};

	if (document.body === null) {
		return addLoadEvent(() => {
			addAll();
		})
	} else {
		addAll();
	}
};

let logInitiated = false;

const history: [string, string][] = []; // store logs to an array for reference

let isWriteToConsole = false;

type LogMessageType = "info" | "warning" | "error";

const Log = {
	writeToConsole: (status: boolean) => {
		isWriteToConsole = status;
	},

	info: (mtext: string, mtype: LogMessageType = "info", howlong = 15000) => {
		initLog();

		if (isWriteToConsole) {
			switch (mtype) {
				case "info":
					console.info(mtext);
					break;
				case "warning":
					console.warn(mtext);
					break;
				case "error":
					console.error(mtext);
					break;
			}

			return;
		}

		mtype = mtype || "info";
		howlong = howlong || 15000;

		Log.debug(mtype, mtext);

		if (!document.getElementById("messages")) {
			setTimeout(() => {
				Log.info(mtext, mtype, howlong);
			}, 100);
			return;
		}

		const alarm = document.createElement("div");
		alarm.className = mtype;
		alarm.innerHTML = mtext;

		// @ts-ignore
    alarm.del = () => {
			document.getElementById("messages")?.removeChild(alarm);
		};

		const closeButton = document.createElement("div");
		closeButton.className = "closeButton";
		closeButton.innerHTML = "X";

		closeButton.onclick = () => {
			alarm.style.display = "none";
		};

		alarm.appendChild(closeButton);

		document.getElementById("messages")?.appendChild(alarm);
		// @ts-ignore
    setTimeout(alarm.del, howlong);
	},

	/**
	 *
	 * @param mtext
	 * @param [howlong]
	 */
	warning: (mtext: string, howlong = 15000) => {
		Log.info(mtext, "warning", howlong);
	},

	/**
	 *
	 * @param mtext
	 * @param [howlong]
	 */
	error: (mtext: string, howlong = 15000) => {
		Log.info(mtext, "error", howlong);
	},

	isDebug: false,

	// showGlobalError: true,

	getHistory: () => {
		return history;
	},

	debug: function(mtype: LogMessageType, mtext: string) {
		if (Log.isDebug) {
			history.push([mtype, mtext]);
			console.log(Array.prototype.slice.call(arguments));
		}
	}
};

export {Log};

// @ts-ignore
globalThis.oldAlert = globalThis.alert;
// replace alert function with Log error message
globalThis.alert = function() {
	if (isWriteToConsole) {
		// @ts-ignore
    globalThis.oldAlert(arguments[0]);
		return;
	}

	if (arguments && arguments[0]) {
		Log.error(String(arguments[0]).replace(/\n/g, "<br/>"));
	}
};

/*const oldOnError = window.onerror;
// override previous handler.
window.onerror = function (errorMsg, url, lineNumber)
{
	// if exist, call previous handler
	if (oldOnError)
	{
		return oldOnError(errorMsg, url, lineNumber);
	}

	const message = errorMsg + "<br/>line:" + lineNumber + "<br/>url:" + url;
	if (Log.showGlobalError) {
		// TODO: correct handling this error
		if (errorMsg !== "Uncaught Invariant Violation: Expected targetIds to be registered.") {
			Log.error(message);
		}
		console.error(message);
	} else {
		console.error(message);
	}

	return false;
};*/
/*
window.addEventListener("unhandledrejection", ({promise, reason}) => {
	const message = reason.message ? `[${reason.code}] ${reason.message}` : `${reason.message}`;

	if (Log.showGlobalError) {
		Log.error(message);
		console.error(message);
	} else {
		console.error(message);
	}
});
 */
