import i18n, { StringMap } from "i18next";
import Backend from "i18next-http-backend";
import LanguageDetector from "i18next-browser-languagedetector";
import {isNode, setCookie} from "./utils";

export const isI18nActive = !isNode();

/**
 * @typedef Language = type Language = { tag: string,  description: string,  direction: "ltr" | "rtl"}
 * @returns {Language[]}
 */
function getLanguages() {
  return [
    {
      "tag": "ar_AE",
      "description": "العربية",
      "direction": "rtl"
    },
    {
      "tag": "en_US",
      "description": "English",
      "direction": "ltr"
    }
  ];
}

const languageGetter = {
  name: "languageGetter",

  lookup() {
    const localLang = localStorage.getItem("language");
    let language = "en-US";

    if (localLang) {
      language = localLang.replace("_", "-");
    }
    if (!localLang) {
      const languageList = getLanguages();
      const navigatorLanguage = window.navigator.language;
      const isNavigatorLang = languageList.some(lang => lang.tag.replace("_", "-") === navigatorLanguage);
      if (isNavigatorLang) {
        language = navigatorLanguage;
      }
      if (!isNavigatorLang) {
        const navigatorLanguages = window.navigator.languages;
        const defaultLanguage = navigatorLanguages.find(lang => {
          return languageList.find(item => item.tag.replace("_", "-") === lang);
        });
        if (defaultLanguage) {
          language = defaultLanguage;
        }
      }
    }

    localStorage.setItem("language", language);
    // used in old guis
    const oldGuisLanguage = language.replace("-", "_");
    setCookie("language", oldGuisLanguage, undefined, "/");

    return language;
  }
};

if (isI18nActive) {
  const LD = new LanguageDetector();
  LD.addDetector(languageGetter);

  i18n
    .use(Backend)
    .use(LD)
    .init({
      backend: {
        loadPath: `locales/{{lng}}/{{ns}}.json`
      },
      detection: {
        order: ["languageGetter"],
        lookupLocalStorage: "language",
        caches: ["localStorage"]
      },
      supportedLngs: ["ar-AE"],
      returnEmptyString: false,
      returnNull: false,

      keySeparator: false,
      nsSeparator: false,
      contextSeparator: undefined,
    });
}

export const __ = (message: string, options: StringMap = {}) => {
  if (isI18nActive) {
    return i18n.t(message, options);
  } else {
    return message;
  }
};
