import {AODAPI, AODAPIGetRequest} from "./aod";
import {Utils} from "../utils";

type AODModelParamters = {
  objid?: string,
  period?: number, // s
  startTs?: number, // s
  endTs?: number, // s
  sleepInterval?: number // ms
}

const defaultSleepInterval = 500; // ms

export class AODModel {
  isStop = false;
  list: AODAPIGetRequest[] = [];
  requestedList = {};
  parameters: AODModelParamters = {};
  onupdate: (list: AODAPIGetRequest[]) => void = () => {};

  looped = false;

  init(parameters: AODModelParamters = {}, onupdate: (list: AODAPIGetRequest[]) => void = ()=>{}) {
    this.parameters = Object.assign(this.parameters, parameters);
    this.onupdate = onupdate;

    this.list = [];
    this.onupdate(this.list);
  }

  async submit(obj: string, start: number, end: number): Promise<void> {
    let response = await AODAPI.submit(obj, start, end);
    let requestid = response.requests[0].requestid;
    this.requestedList[requestid] = {obj, start, end};
  }

  async get(start: number, end: number): Promise<void> {
    const parameters: { objid?: string, startTs: number, endTs: number } = {
      startTs: start,
      endTs: end
    };
    if (this.parameters.objid) {
      parameters.objid = this.parameters.objid;
    }
    let response = await AODAPI.get(parameters);
    this.list = response.requests;

    this.onupdate(this.list);
  }

  async wait(): Promise<void> {
    if (this.looped) {
      return;
    }
    this.looped = true;
    while (!this.isStop) {
      const period = this.parameters.period ?? 0;
      const start = this.parameters.startTs ?? Math.round((Date.now() - period * 1000) / 1000);
      const end = this.parameters.endTs ?? Math.round(Date.now() / 1000);

      await this.get(start, end);
      await Utils.wait(this.parameters.sleepInterval ?? defaultSleepInterval);
    }
    this.looped = false;
  }

  stop() {
    this.isStop = true;
  }
}
