import {UUID} from "@solid/types";
import {alignBorder, getOrigin, Utils} from "../utils";
import {Events} from "../events";
import $ from "jquery";
import type {Optional} from "@solid/types";

export enum AODStatus {
  COMPLETED = "COMPLETED",
  CANCELED = "CANCELED",
  FAILED = "FAILED",

  NOT_CONFIRMED = "NOT CONFIRMED",
  QUEUED = "QUEUED",
  PROCESSING = "PROCESSING",
  PAUSED = "PAUSED"
}

// const status = ['COMPLETED', 'CANCELED', 'FAILED'];

type RequestMethod = "GET" | "POST" | "PUT";

type RequestType = Optional<AODAPISubmitRequest, any> & {
  promise?: Promise<Record<string, any>>
};

type AODAPISubmitRequest = {
  userid: UUID
  objid: UUID,
  startTs: number, // sec
  endTs: number, // sec
  createdTs: number, // sec
  status: AODStatus,
  isPrivate: boolean,
  streamNum: string,
  realm: UUID,
  requestid: string,
  avatarid: UUID
}

type AODAPISubmitResponse = {
  method: "submit",
  code: number,
  error: string,
  requests: AODAPISubmitRequest[]
}

export type AODAPIGetRequest = {
  userid: UUID
  objid: UUID,
  startTs: number, // sec
  endTs: number, // sec
  createdTs: number, // sec
  status: AODStatus,
  isPrivate: boolean,
  streamNum: number,
  realm: UUID,
  requestid: string,
  avatarid: UUID,
  size: number,
  downloaded: number,
  completedTs: number,
  eventid: string | null
}

export type AODAPIGetResponse = {
  method: "get",
  code: number,
  error: string,
  requests: AODAPIGetRequest[]
}

export const AODAPI = {
  _request: async function (requestMethod: RequestMethod, url: string, parameters: Record<string, any>): Promise<Record<string, any>> {
    let settings: any = {
      url: url,
      type: requestMethod,
      cache: false,
      dataType: "json",
      async: true
    };

    if (requestMethod === "GET") {
      settings.data = parameters;
    } else {
      settings.processData = false;
      settings.contentType = "application/json; charset=utf-8";
      settings.data = JSON.stringify(parameters);
    }

    return Utils.promisifyAjax($.ajax(settings));
  },
  submit: function (objid: UUID, startTs: number, endTs: number, streamNum: string = "1", isPrivate: boolean = true): Promise<AODAPISubmitResponse> {
    // /api/aod/submit
    // {objid: objid, start: startTs, end: endTs}
    // response = {requests}

    const parameters = {
      objid: objid,
      startTs: startTs,
      endTs: endTs,
      streamNum: streamNum,
      isPrivate: isPrivate
    };
    return this._request("POST", `${getOrigin()}/api/aod/submit`, parameters) as Promise<AODAPISubmitResponse>;
  },
  get: function (parameters: Record<string, any>): Promise<AODAPIGetResponse> {
    // /api/aod/get
    // request = {requestid}
    // response = {requests}

    return this._request("GET", `${getOrigin()}/api/aod/get`, parameters) as Promise<AODAPIGetResponse>;
  },
  pause: function (requestid: string) {
    // /api/aod/pause
    // request = {requestid}

    const parameters = {
      requestid: requestid
    };
    return this._request("PUT", `${getOrigin()}/api/aod/pause`, parameters);
  },
  resume: function (requestid: string) {
    // /api/aod/resume
    // request = {requestid}

    const parameters = {
      requestid: requestid
    };
    return this._request("PUT", `${getOrigin()}/api/aod/resume`, parameters);
  },
  cancel: function (requestid: string) {
    // /api/aod/cancel
    // request = {requestid}

    const parameters = {
      requestid: requestid
    };
    return this._request("GET", `${getOrigin()}/api/aod/cancel`, parameters);
  }
};

export class AODWait {
  _events: Events;
  _requestList: RequestType[];
  _isLooped: boolean;
  _checkInterval: number; // ms

  constructor() {
    this._events = new Events();
    this._requestList = [];
    this._isLooped = false;
    this._checkInterval = 5000; // ms
  }

  /**
   *
   * @param {string} topic request.requestid
   * @param {function} listener
   */
  once(topic: string, listener: (...args: any) => void) {
    this._events.once(topic, listener);
  }

  /**
   *
   * @param {string} obj
   * @param {number} startTime ms
   * @param {number} length sec
   * @param {string} streamNum
   * @returns {Promise<*>}
   */
  add(obj: UUID, startTime: number, length: number = 30, streamNum: string = "1", isPrivate: boolean = true): Promise<RequestType> {
    return new Promise((resolve, reject) => {
      let request: RequestType | null = this.searchRequest(obj, startTime, length, streamNum);
      if (!request) {
        const {start, end} = alignBorder({startTime: startTime, endTime: startTime + length * 1000, roundToSec: true});
        let promise = AODAPI.submit(obj, start, end, streamNum, isPrivate);
        request = {
          requestid: "",
          objid: obj,
          startTs: start,
          endTs: end,
          streamNum: streamNum,
          promise: promise,
          isPrivate
        };
        this._requestList.push(request);

        promise.then((result) => {
          request = result.requests[0];

          let index = this.searchRequestIndex(obj, startTime, length, streamNum);
          this._requestList[index].requestid = request.requestid;

          this._wait();

          resolve(request);
        });
      } else {
        if (request.requestid) {
          resolve(request);
        } else {
          (request.promise as Promise<AODAPISubmitResponse> | undefined)?.then((result) => {
            request = result.requests[0];

            let index = this.searchRequestIndex(obj, startTime, length, streamNum);
            this._requestList[index].requestid = request.requestid;

            this._wait();

            resolve(request);
          });
        }
      }
    });
  }

  /**
   * @param {[number]} requestIdList
   * @returns {Promise<Array|*>}
   */
  async get(requestIdList: number[]): Promise<AODAPIGetResponse["requests"]> {
    let response = await AODAPI.get({requestid: JSON.stringify(requestIdList)});
    return response.requests;
  }

  /**
   * remove all listeners
   */
  clear() {
    this._isLooped = false;
    this._events.removeAllListeners();
  }

  /**
   * @returns {Promise<void>}
   * @private
   */
  async _wait() {
    if (this._isLooped) {
      return;
    }

    this._isLooped = true;

    // loop until all requests will be completed or rejected
    while (this._isLooped) {
      let requestIdList: number[] = this._requestList
        .map((row) => {return row.requestid;})
        .filter((requestid) => {return requestid !== "";}) as number[];

      if (this._requestList.length > 0 && requestIdList.length === 0) {
        await Utils.wait(this._checkInterval);
        continue;
      } else
      if (this._requestList.length === 0) {
        this._isLooped = false;
        console.log("exit loop");
        continue;
      }

      let resultList = await this.get(requestIdList);

      resultList.map((result) => {
        if (result.status === AODStatus.COMPLETED) {
          this._events.trigger(`request.${result.requestid}`, {status: true, result: result});

          let index = this._requestList.findIndex((request) => {return request.requestid === result.requestid;});
          this._requestList.splice(index, 1);

          return;
        }
        if ([AODStatus.CANCELED, AODStatus.FAILED].includes(result.status)) {
          this._events.trigger(`request.${result.requestid}`, {status: false, result: result});

          let index = this._requestList.findIndex((request) => {return request.requestid === result.requestid;});
          this._requestList.splice(index, 1);

          return;
        }
      });

      await Utils.wait(this._checkInterval);
    }
  }

  searchRequest(obj: UUID, startTime: number, length: number = 1, streamNum: string = "1"): RequestType | null {
    let index = this.searchRequestIndex(obj, startTime, length, streamNum);
    return index >= 0 ? this._requestList[index] : null;
  }

  /**
   * search for request in already created request list
   *
   * @param {string} obj
   * @param {number} startTime
   * @param {number} length sec
   * @param {string} streamNum
   * @returns {number}
   */
  searchRequestIndex(obj: UUID, startTime: number, length: number = 1, streamNum: string = "1") {
    const {start, end} = alignBorder({startTime: startTime, endTime: startTime + length * 1000, roundToSec: true});

    for (let i = 0; i < this._requestList.length; i++) {
      let request = this._requestList[i];
      if (request.objid === obj && request.streamNum === streamNum
          && (
            (request.startTs === start && request.endTs === end)
            ||
            (request.startTs <= start && end <= request.endTs)
          )) {
        return i;
      }
    }

    return -1;
  }
}
