import { GraphQLError, GraphQLScalarType, Kind } from "graphql";
import { InMemoryCache, NormalizedCacheObject, ApolloClient } from "@apollo/client";
import produce from "immer";

import {Resolvers, StoreQuery, StoreQueryVariables, StoreDocument, AppStore} from "@generated/graphql";
import { Apollo } from "@core/api";
import { toMaybe, PropNameTypeMap, copyInputToQuery } from "utils";

export interface ClientContext {
  client: ApolloClient<NormalizedCacheObject>;
  cache: InMemoryCache;
  getCacheKey: (obj: { __typename: string; id: string | number }) => any;
}

const UUIDScalarType = new GraphQLScalarType({
  name: "UUID",
  description: "Unique Object Identifier",
  serialize(value) {
    if (typeof value !== "string") {
      throw new GraphQLError("Invalid string");
    }
    return value.toString();
  },
  parseValue(value) {
    if (typeof value !== "string") {
      throw new GraphQLError("Invalid string");
    }
    return value.toString();
  },
  parseLiteral(ast) {
    return ast.kind === Kind.STRING ? ast.value : null;
  }
});

const propNameTypeMap: PropNameTypeMap = {
  session: { __typename: "Session", required: true }, // required: true to prevent overwriting with null if null is passed in an input
  info: { __typename: "UserInfo" },
  user: { __typename: "UserData" },
  solidConfiguration: { __typename: "SolidConfiguration" },
  realm: { __typename: "GenericObject" },
  select: { __typename: "ObjUUID" },
  workspace: { __typename: "Workspace", required: true },
  event: { __typename: "EventInfo" },
  entry: { __typename: "AuditEntry" },
  updatedEvent: { __typename: "AuditEntry" },
  socketEvent: { __typename: "SocketEvent" },
  creatingViews: { __typename: "CreatingView" },
  widgets: { __typename: "WidgetInfo" },
  selectedWidget: { __typename: "WidgetInfo" },
  cameraWidgetProps: { __typename: "CameraWidgetProps" },
  witnesses: { __typename: "RealmObject" },
  snapshots: { __typename: "AuditSnapshot" },
  metadata: { __typename: "MetaEntry" },
  tags: { __typename: "KeyValue" },
  metaClass: { __typename: "MetaClassifier" },
  attributes: { __typename: "KeyValue" },
};

const resolvers: Resolvers = {
  Query: {
    store: async (root, args, { cache }, info) => {
      const data = cache.readQuery<StoreQuery, StoreQueryVariables>({ query: StoreDocument });
      return (data || { store: Apollo.defaultStore() }).store as AppStore;
    },
  },
  Mutation: {
    setStore: async (root, args, { cache }, info) => {
      const data = cache.readQuery<StoreQuery, StoreQueryVariables>({ query: StoreDocument });
      const baseStore = data ? data.store : toMaybe(Apollo.defaultStore());
      const store = produce(baseStore, draftStore => {
        copyInputToQuery(draftStore, args.store, propNameTypeMap);
      });
      cache.writeQuery<StoreQuery, StoreQueryVariables>({ query: StoreDocument, data: { store } });
      return true;
    }
  },
  UUID: UUIDScalarType
};

export default resolvers;
