import electron, { BrowserWindow, crashReporter } from "electron"; // Module to control application life.
import { isPackage, loadPackageEnvironment, getPackageVersion } from "./packageUtils";
import Window, { WindowId } from "./Window";
import * as logger from "./logger";
import * as autoUpdate from "./autoUpdate";
import * as viewWindow from "./ViewWindow";
import * as screen from "./screen";
import * as store from "./store";
import * as linkProvision from "./linkProvision";

if (isPackage && require("electron-squirrel-startup")) process.exit(0);

loadPackageEnvironment();

const API_MINIDUMP = process.env.API_MINIDUMP;
crashReporter.start({
  companyName: "titan-systems",
  productName: "solid",
  ignoreSystemCrashHandler: true,
  uploadToServer: !!API_MINIDUMP,
  submitURL: API_MINIDUMP,
  globalExtra: {
    cluster: process.env.API_BACKEND ?? ""
  }
});

const app = electron.app; // Module to create native browser window.

// increase number of contexts for creating more native players
app.commandLine.appendSwitch("max-active-webgl-contexts", "64");

app.commandLine.appendSwitch("force_high_performance_gpu", "1");

/*// GC-related options
app.commandLine.appendSwitch("js-flags", "--expose-gc");
app.commandLine.appendSwitch("js-flags", "--trace-gc");
app.commandLine.appendSwitch("js-flags", "--gc-interval=10");
app.commandLine.appendSwitch("js-flags", "--predictable-gc-schedule");
app.commandLine.appendSwitch("js-flags", "--trace-gc-objects-stats");
app.commandLine.appendSwitch("js-flags", "--max-old-space-size=1024");
*/

// TODO: remove this option, after removing old gui-admin UI from solid
// allow load and change iframe content
// internal pdf viewer will not work with this option
app.commandLine.appendSwitch("disable-site-isolation-trials");

// Renderer Node fs API stops working on page reload
// https://github.com/electron/electron/issues/22119
// app.allowRendererProcessReuse = false;

// Keep a global reference of the window object, if you don't, the window will
// be closed automatically when the JavaScript object is garbage collected.
let mainWindow: BrowserWindow | null = null;

const FORCE_EGL = process.env.FORCE_EGL || "no";
const FORCE_WEBGL = process.env.FORCE_WEBGL || "no";
const OPEN_WEB_TOOLS = process.env.OPEN_WEB_TOOLS || "no";
const ENABLE_TRACING = process.env.ENABLE_TRACING || "no";
const forceEGL = FORCE_EGL === "yes";
const forceWEBGL = FORCE_WEBGL === "yes";
const isOpenWebTools = OPEN_WEB_TOOLS === "yes";
const isTracing = ENABLE_TRACING === "yes";
const tracingTimeout = process.env.TRACING_TIMEOUT ? parseInt(process.env.TRACING_TIMEOUT) : 300000; // 5min by default

if (forceWEBGL) {
  app.commandLine.appendSwitch("enable-webgl-draft-extensions");
  app.commandLine.appendSwitch("ignore-gpu-blocklist");
}

if (forceEGL) {
  // Let's select which implementation of GL the GPU process should use.
  // Options are:
  // desktop: whatever desktop OpenGL the user has installed (Linux and Mac default).
  // egl:     whatever EGL / GLES2 the user has installed (Windows default - actually ANGLE).
  // swiftshader: The SwiftShader software renderer (on some Macs chosen by default and give crappy picture in our case).
  app.commandLine.appendSwitch("use-gl", "egl");
}

function createWindow() {
  const window = new Window(WindowId.Main);
  // Create the browser window.
  mainWindow = window.createMain();

  logger.init(mainWindow);

  if (isPackage) {
    logger.log(`App version: ${getPackageVersion()}`);
  }

  autoUpdate.init(mainWindow);

  screen.init(mainWindow);
  viewWindow.init(mainWindow);
  store.init();
  linkProvision.init(mainWindow);

  window.loadURL();

  if (isOpenWebTools && !isPackage) {
    //mainWindow.webContents.loadURL("chrome://gpu"); // check graphics feature status
    mainWindow.webContents.openDevTools();
  }

  // Emitted when the window is closed.
  mainWindow.on("closed", () => {
    // Dereference the window object, usually you would store windows
    // in an array if your app supports multi windows, this is the time
    // when you should delete the corresponding element.
    mainWindow = null;
  });
}

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.on("ready", createWindow);

// Quit when all windows are closed.
app.on("window-all-closed", () => {
  // On OS X it is common for applications and their menu bar
  // to stay active until the user quits explicitly with Cmd + Q
  if (process.platform !== "darwin") {
    app.quit();
  }
});

app.on("activate", () => {
  // On OS X it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (mainWindow === null) {
    createWindow();
  }
});

if (isTracing) {
  const contentTracing = electron.contentTracing;
  app.whenReady().then(() => {
    (async () => {
        const categories = await contentTracing.getCategories();
        console.log("TRACING CATEGORIES:");
        console.table(categories);
        await contentTracing.startRecording({
            included_categories: [
//                "*",
                "cc",
                "cc.debug",
                "cppgc",
//                "drm",
//                "gpu",
//                "gpu.angle",
//                "gpu.capture",
//                "memory",
                "disabled-by-default-memory-infra",
                "disabled-by-default-memory-infra.v8.code_stats",
                "disabled-by-default-v8.gc",
                "disabled-by-default-v8.gc_stats"
            ]});
        console.log("Tracing started");
        await new Promise(resolve => setTimeout(resolve, tracingTimeout));
        const path = await contentTracing.stopRecording();
        console.log('Tracing data recorded to ' + path);
    })()
  });
}
