import { ipcMain, BrowserWindow } from "electron";
import mdns from "mdns-js";
import axios from "axios";
import https from "https";
import { MessageId } from "./types";

export type LinkInfo = {
  host: string;
  port: number;
  status?: LinkStatus;
};

export type LinkStatus = {
  version: string;
  endpoint: string;
  publickey: string;
  uuid: string;
  provisioned: boolean;
  connected: boolean;
  hostname: string;
  ssh_public_key?: string;
};

export type LinkAddArgs = {
  link: LinkInfo;
  endpoint: string;
};

export type LinkAddReplyArgs = {
  link: LinkInfo;
};

let mainWindow: BrowserWindow;
let mdnsBrowser: any;

type MdnsData = {
  addresses: string[];
  query: string[];
  type: MdnsType[];
  interfaceIndex: number;
  networkInterface: string;
};

type MdnsType = {
  name: string;
  protocol: "tcp" | "udp";
  subtypes: string[];
  description?: string;
};

export function init(mainWin: BrowserWindow): void {
  mainWindow = mainWin;

  ipcMain.on(MessageId.LinkProvisionStart, (event) => {
    stopMdnsBrowser();

    mdnsBrowser = mdns.createBrowser();

    mdnsBrowser.on("ready", () => {
      mdnsBrowser.discover();
    });

    mdnsBrowser.on("update", async (data: MdnsData) => {
      if (data.addresses.length === 0 || !data.type.some(type => type.name === "videonext-link")) {
        return;
      }

      const link: LinkInfo = {
        host: data.addresses[0],
        port: 62100
      };

      try {
        link.status = await getLinkStatus(link);
      }
      catch (e: any) {
        publishLinkError(e);
      }
      publishLinkFound(link);
    });
  });

  ipcMain.on(MessageId.LinkProvisionStop, (event) => {
    stopMdnsBrowser();
  });

  ipcMain.on(MessageId.LinkProvisionAdd, async (event, args: LinkAddArgs) => {
    const { link, endpoint } = args;
    let status = link.status;
    try {
      status = await addLink(link, endpoint);
    }
    catch (e: any) {
      publishLinkError(e);
    }
    finally {
      const replyArgs: LinkAddReplyArgs = { link: { ...link, status } };
      event.reply(MessageId.LinkProvisionAddReply, replyArgs);
    }
  });
}

function stopMdnsBrowser(): void {
  if (!mdnsBrowser) {
    return;
  }
  mdnsBrowser.stop();
  mdnsBrowser = undefined;
}

function publishLinkFound(link: LinkInfo): void {
  if (mainWindow) {
    mainWindow.webContents.send(MessageId.LinkProvisionFound, link);
  }
}

function publishLinkError(error: Error): void {
  if (mainWindow) {
    mainWindow.webContents.send(MessageId.LinkProvisionError, error);
  }
}

async function getLinkStatus(link: LinkInfo): Promise<LinkStatus> {
  try {
    const httpsAgent = new https.Agent({ rejectUnauthorized: false });
    const res = await axios.get<LinkStatus>(`${getLinkBaseUrl(link)}/status`, { httpsAgent });
    if (!res.data) {
      throw new Error("Empty Avatar status response.");
    }
    return res.data;
  }
  catch (e: any) {
    throw new Error(`Avatar ${link.host} error: ${e.message}`);
  }
}

async function addLink(link: LinkInfo, endpoint: string): Promise<LinkStatus> {
  try {
    if (!link.status) {
      throw new Error("Status is unknown.");
    }
    if (link.status.endpoint || link.status.provisioned) {
      throw new Error("Already provisioned.");
    }

    const httpsAgent = new https.Agent({ rejectUnauthorized: false });
    await axios.post(`${getLinkBaseUrl(link)}/provision`, { ENDPOINT: endpoint }, { httpsAgent });

    const status = await getLinkStatus(link);
    if (status.endpoint.toLocaleUpperCase() !== endpoint.toLocaleUpperCase()) {
      throw new Error("Provision is not done.");
    }
    return status;
  }
  catch (e: any) {
    if (e.message.startsWith(`Avatar ${link.host} error`)) {
      throw e;
    }
    throw new Error(`Avatar ${link.host} error: ${e.message}`);
  }
}

function getLinkBaseUrl(link: LinkInfo): string {
  return `https://${link.host}:${link.port}/vnlk`;
}
