import * as os from "os";
import { autoUpdater, BrowserWindow, dialog, MessageBoxOptions } from "electron";
import { getPackageVersion, isFirstRun, isPackage } from "./packageUtils";
import { getPlatform, Platform } from "./util";
import * as logger from "./logger";
import { API } from "@solid/libs";
import { __ } from "./i18n";
import { ApiError } from "@solid/types";

const api = new API();

export async function init(mainWindow: BrowserWindow): Promise<void> {
  if (!isPackage) {
    return;
  }
  if (isFirstRun()) {
    logger.log("Skipping auto update, it is a first application run.");
    return;
  }

  try {
    const updatesHost = await api.getParameters();
    const UPDATES_HOST = updatesHost.list.UPDATES_HOST;
    const backendUrl = new URL(process.env.API_BACKEND ?? "");
    const OVERCAST_EXT_HTTP_URL = backendUrl.hostname;

    const platform = getPlatform();
    if (platform === Platform.Unsupported) {
      throw new Error(`Unsupported platform: ${os.platform()} ${os.arch()}.`);
    }

    const url = `https://${UPDATES_HOST}/${platform}/solid/${OVERCAST_EXT_HTTP_URL}/`;

    const version = getPackageVersion();
    if (!version) {
      throw new Error("Package version is not specified.");
    }

    autoUpdater.setFeedURL({url});
    logger.log("Auto update URL:", url);

    autoUpdater.on("error", (err) => {
      logger.error("Auto update error:", err);
    });

    autoUpdater.once("checking-for-update", () => {
      logger.log("Checking for updates...");
    });

    autoUpdater.once("update-available", () => {
      logger.log("Update available. Downloading...");
    });

    autoUpdater.once("update-not-available", () => {
      logger.log("Update not available.");
    });

    autoUpdater.once("update-downloaded", () => {
      logger.log("Update downloaded.");

      const dialogOpts: MessageBoxOptions = {
        type: "info",
        buttons: [__("Restart"), __("Later")],
        title: __("Application Update"),
        message: __("Attention!\nThe new client software version was downloaded and is ready to use.\nWe strongly recommend restarting your client right now as the old version is not guaranteed to work reliably with the updated cloud services.\nDo you want to restart the client software right now?")
      };

      dialog.showMessageBox(mainWindow, dialogOpts).then(returnValue => {
        if (returnValue.response === 0) {
          autoUpdater.quitAndInstall();
        }
      });
    });

    autoUpdater.checkForUpdates();
  }
  catch (e) {
    const error = e as ApiError;
    logger.error("Could not initialize auto update:", error.code, error.message);
  }
}
