import { BrowserWindow, ipcMain, screen } from "electron";
import type Electron from "electron";
import { MessageId } from "./types";
import Window, { findMainWindow } from "./Window";

export interface ViewWindowArgs {
  viewId: string;
  name: string;
  index: number;
  displayId?: number;
  rect?: Electron.Rectangle;
}

let mainWindow: BrowserWindow;

export function init(mainWin: BrowserWindow): void {
  mainWindow = mainWin;

  ipcMain.on(MessageId.ViewToWindow, (event, args: ViewWindowArgs) => {
    const { viewId, name, index, displayId, rect } = args;

    let window = Window.windows.find(w => w.windowId === viewId);
    if (window) {
      if (window.displayId === displayId) {
        window.activate();
        event.reply(MessageId.ViewToWindowReply, args);
      }
      else {
        window.changeDisplay(displayId);
      }
      return;
    }

    window = new ViewWindow(viewId, name, index, displayId, rect);
    window.createChild();
    window.loadURL();
    event.reply(MessageId.ViewToWindowReply, args);
  });
}

class ViewWindow extends Window {
  constructor(viewId: string, readonly name: string, readonly index: number, override readonly displayId?: number, override readonly rect?: Electron.Rectangle) {
    super(viewId, mainWindow);

    if (!this.displayId) {
      this.displayId = findMainWindow()?.getDisplayId() ?? screen.getPrimaryDisplay().id;
    }
  }

  override createChild(): BrowserWindow {
    const { name, displayId, rect } = this;
    return super.createChild({ title: name, displayId, rect, maximizeOnExternalDisplay: true });
  }

  override loadURL(): void {
    super.loadURL("/view/" + this.windowId);
  }

  protected override init(): void {
    super.init();

    if (!this.window) {
      return;
    }

    this.window.on("page-title-updated", (event) => event.preventDefault());

    const main = this.mainWindow;
    if (!main) {
      return;
    }

    this.window.on("closed", () => {
      const { windowId: viewId, name, index } = this;
      const args: ViewWindowArgs = { viewId, name, index };
      main.webContents.send(MessageId.ViewToMain, args);
    });
  }
}
