/* eslint-disable import/no-extraneous-dependencies */
import moment from "moment";
import { DataProxy, ApolloClient } from "@apollo/client";
import { round } from "utils";
import { __ } from "@solid/libs/i18n";
import { ModuleInfo, RouteParams } from "./types";
import { isElectron } from "@solid/libs";
import { ViewId } from "@generated/graphql";

export function getModule(module: string) {
  const moduleToImport = {
    "ViewLayouts": () => import("components/ViewLayouts"),
    "ViewLayouts/ViewCustom": () => import("components/ViewLayouts/ViewCustom"),
    "View/SelectTemplate": () => import("components/View/SelectTemplate"),
    "View/SelectLayout": () => import("components/View/SelectLayout"),
    "Views/ViewStart": () => import("components/Views/ViewStart"),
    "Widgets/WidgetPlaceholder": () => import("components/Widgets/WidgetPlaceholder"),
    "EventList": () => import("components/EventList"),
    "EventInfo": () => import("components/EventInfo"),
    "CellContainer/CameraCell": () => import("components/CellContainer/CameraCell"),
    "CellContainer/EventVideo": () => import("components/CellContainer/EventVideo"),
    "CameraList": () => import("components/CameraList"),
    "Widgets/WidgetPreview": () => import("components/Widgets/WidgetPreview"),
    "Timeline": () => import("components/Timeline"),
    "ImageMap": () => import("components/ImageMap"),
    "GEOMap": () => import("components/GEOMap"),
    "VMX": () => import("components/VMX"),
    "Admin/LinkProvision": () => import("components/Admin/LinkProvision"),
    "Admin/Users": () => import("components/Admin/Users"),
    "Admin/Groups": () => import("components/Admin/Groups"),
    "Admin/Zones": () => import("components/Admin/Zones"),
    "Admin/Sets": () => import("components/Admin/Sets"),
    "Admin/Policies": () => import("components/Admin/Policies"),
    "Admin/AdminCameraList": () => import("components/Admin/AdminCameraList"),
    "Admin/AdminLinkList": () => import("components/Admin/AdminLinkList"),
    "Admin/Sensors": () => import("components/Admin/Sensors"),
    "Admin/Gateways": () => import("components/Admin/Gateways"),
    "": () => {},
    "NotFound": () => import("components/NotFound"),
    "DeviceEventList": () => import("components/EventList/DeviceEventList"),
    "UserEventList": () => import("components/EventList/UserEventList"),
    "ArchiveEvents": () => import("components/EventList/ArchiveEvents"),
    "AuditEventInfo": () => import("components/EventInfo/AuditEventInfo"),
    "Admin/AvatarActivation": () => import("components/Admin/AvatarActivation"),
    "LabelsAndHierarchies/HierarchyEditor": () => import("components/LabelsAndHierarchies/HierarchyEditor"),
    "LabelsAndHierarchies/LabelEditor": () => import("components/LabelsAndHierarchies/LabelEditor"),
    "NotificationPolicies": () => import("components/NotificationPolicies"),
    "RealmSettings": () => import("components/RealmSettings"),
    "ManageAvatars": () => import("components/ManageAvatars"),
    "WatchlistLPR": () => import("components/WatchlistLPR"),
    "WatchlistFacial": () => import("components/WatchlistFacial"),
    "VideoTouring": () => import("components/VideoTouring"),
    "Admin/ConfigurationStatusReport": () => import("components/Admin/ConfigurationStatusReport"),
    "VideoWalls": () => import("components/VideoWalls"),
    "Subscription": () => import("components/Subscription"),
    "QuickStart": () => import("components/QuickStart"),
    "TrainingVideos": () => import("components/TrainingVideos")
  };

  const error = `Module "${module}" not defined`;
  if (!moduleToImport[module]) {
    console.error(error);
    module = "NotFound";
  }

  console.assert(moduleToImport[module], error);

  return moduleToImport[module];
}

export function formatDateTime(dateTime: Date, language: string = "default"): string {
  return new Intl.DateTimeFormat(
    language,
    {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit"
    }
  ).format(dateTime);
}

export function parseDateTime(value: string): moment.Moment {
  return moment(value, "DD.MM.YYYY, HH:mm:ss");
}

export function getFileSizeText(size: number): string {
  const giga = 1024 * 1024 * 1024;
  const mega = 1024 * 1024;
  const kilo = 1024;

  if (size >= giga) {
    return `${round(size / giga)} ${__("GB")}`;
  }
  if (size >= mega) {
    return `${round(size / mega)} ${__("MB")}`;
  }
  if (size >= kilo) {
    return `${round(size / kilo)} ${__("KB")}`;
  }
  return `${round(size)} ${__("B")}`;
}

export function readQuery<TQuery, TVariables>(client: ApolloClient<any>, query: DataProxy.Query<TVariables, TQuery>): TQuery | null {
  try {
    return client.readQuery<TQuery, TVariables>(query);
  }
  catch {
    return null;
  }
}

export function handleDefaultView(
  views: ModuleInfo[],
  allViews: (ModuleInfo)[],
  params: RouteParams,
  pathname: string
) {
  const modules = views.filter(v => !isElectron() || !v.inWindow);

  if (modules.length > 0) {
    if (params.viewId) {
      const view = allViews.find(v => v?.id === params.viewId);
      return view ? [view] : [modules[0]];
    }

    if (params.linkId) {
      const viewLinks = allViews.find(v => v?.id === ViewId.Avatars);
      return viewLinks ? [viewLinks] : [modules[0]];
    }

    if (params.cameraId
        || pathname.includes("view/camera/import")
        || pathname.includes("/view/camera/add")
        || pathname.includes("/view/camera/discovery")) {
      const viewCameras = allViews.find(v => v?.id === ViewId.Cameras);
      return viewCameras ? [viewCameras] : [modules[0]];
    }

    if (params.userId || pathname.includes("/view/user/add")) {
      const viewUsers = allViews.find(v => v?.id === ViewId.Users);
      return viewUsers ? [viewUsers] : [modules[0]];
    }

    if (params.policyId || pathname.includes("/view/policy/add")) {
      const viewPolicy = allViews.find(v => v?.id === ViewId.Policies);
      return viewPolicy ? [viewPolicy] : [modules[0]];
    }

    if (params.groupId || pathname.includes("/view/group/add")) {
      const viewGroup = allViews.find(v => v?.id === ViewId.Groups);
      return viewGroup ? [viewGroup] : [modules[0]];
    }

    if (params.zoneId || pathname.includes("/view/zone/add")) {
      const viewZone = allViews.find(v => v?.id === ViewId.Zones);
      return viewZone ? [viewZone] : [modules[0]];
    }

    if (params.setId || pathname.includes("/view/set/add")) {
      const viewZone = allViews.find(v => v?.id === ViewId.Sets);
      return viewZone ? [viewZone] : [modules[0]];
    }

    if (params.sensorId || pathname.includes("/view/sensor/add")) {
      const viewSensor = allViews.find(v => v.id === ViewId.Sensors);
      return viewSensor ? [viewSensor] : [modules[0]];
    }

    if (params.gatewayId || pathname.includes("/view/gateway/add")) {
      const viewSensor = allViews.find(v => v.id === ViewId.Gateways);
      return viewSensor ? [viewSensor] : [modules[0]];
    }
  }

  return modules;
}
