import React, {useCallback, useEffect} from "react";
import {
  AppStore,
  useStoreQuery,
  useSetStoreMutation,
  AppStoreInput,
  SetStoreMutation,
  SetStoreMutationVariables,
  SetStoreDocument
} from "@generated/graphql";
import { Apollo } from "@core/api";
import {isElectron} from "@solid/libs/utils";
import { MessageId } from "electron/types";
import { StoreChangeArgs } from "electron/store";
import type Electron from "electron";

let electronWindowId: string | undefined;
const ipcSubscribe = async () => {
  if (isElectron()) {
    window.ipcRenderer.on(MessageId.SendWindowId, (event: Electron.IpcRendererEvent, windowId: string) => {
      electronWindowId = windowId;
    });

    window.ipcRenderer.on(MessageId.StoreToView, (event: Electron.IpcRendererEvent, { store }: StoreChangeArgs) => {
      if (Apollo.client) {
        Apollo.client.mutate<SetStoreMutation, SetStoreMutationVariables>({
          mutation: SetStoreDocument,
          variables: { store }
        });
      }
    });
  }
};
ipcSubscribe();

type StoreResult = {
  store: AppStore;
  setStore: (store: AppStoreInput) => Promise<void>;
  updateLoading: boolean;
  updateError?: Error;
};

export function useStore(): StoreResult {
  const { data, loading, error } = useStoreQuery();
  // TODO: create an issue about useSetStoreMutation, it should not return result for removed component
  const [updateStore, { loading: updateLoading, error: updateError }] = useSetStoreMutation();

  useEffect(() => {
    error && console.error("Error executing store query:", error);
    !error && !data && console.error("Store query returned empty data.");
  }, [loading, error, data]);

  useEffect(() => {
    updateError && console.error("Could not update app store:", updateError);
  }, [updateError]);

  const setStore = useCallback(async (store: AppStoreInput): Promise<void> => {
    // TODO: check if this call should be await before next logic
    await updateStore({ variables: { store } });
    if (isElectron() && electronWindowId) {
      let args: StoreChangeArgs | string = { windowId: electronWindowId, store };
      if (args.store && args.store["workspace"] && args.store["workspace"].event && args.store["workspace"].event.entry) {
        args = JSON.stringify(args);
      }
      window.ipcRenderer.send(MessageId.StoreToMain, args);
    }
  }, [updateStore]);

  const store = data ? data.store as AppStore : Apollo.defaultStore();
  return { store, setStore, updateLoading, updateError };
}

/*export const connect = <TProps, NS>(
  Component: React.ComponentType<TProps>,
  mapStateToProps?: (state: AppState) => NS
): React.ComponentType<TProps> => (props) => {
  let [state, actions] = useStore(mapStateToProps);
  let _props = {...props, state, actions};
  return <Component {..._props}/>;
};*/

type GetProps<C> = C extends React.ComponentType<infer P> ? P : never;

/**
 * A property P will be present if:
 * - it is present in DecorationTargetProps
 *
 * Its value will be dependent on the following conditions
 * - if property P is present in InjectedProps and its definition extends the definition
 *   in DecorationTargetProps, then its definition will be that of DecorationTargetProps[P]
 * - if property P is not present in InjectedProps then its definition will be that of
 *   DecorationTargetProps[P]
 * - if property P is present in InjectedProps but does not extend the
 *   DecorationTargetProps[P] definition, its definition will be that of InjectedProps[P]
 */
export type Matching<InjectedProps, DecorationTargetProps> = {
  [P in keyof DecorationTargetProps]: P extends keyof InjectedProps
    ? InjectedProps[P] extends DecorationTargetProps[P]
      ? DecorationTargetProps[P]
      : InjectedProps[P]
    : DecorationTargetProps[P]
};

export const connect = <MappedProps, OwnProps>(
  mapProps: (store: AppStore) => MappedProps
) => <C extends React.ComponentType<Matching<MappedProps, GetProps<C>>>>(
    WrappedComponent: C
  ): React.ComponentType<JSX.LibraryManagedAttributes<C, Omit<GetProps<C>, keyof MappedProps>> & OwnProps> => {
  return (props: any) => {
    const { store } = useStore();
    const mappedProps = mapProps(store);
    return <WrappedComponent {...props} {...mappedProps}/>;
  };
};

export type WithStoreProps = Pick<StoreResult, "store" | "setStore">;

export const withStore = <C extends React.ComponentType<GetProps<C> & WithStoreProps>>(
  WrappedComponent: C
): React.ComponentType<JSX.LibraryManagedAttributes<C, Omit<GetProps<C>, keyof WithStoreProps>>> => {
  return (props: any) => {
    const { store, setStore } = useStore();
    return <WrappedComponent {...props} store={store} setStore={setStore} />;
  };
};
