import { useEffect, useRef, useCallback, RefObject } from "react";
import produce from "immer";
import { SolidConfig } from "./config";
import { RSplitPanel, RBoxPanel, RWidgetLoader, LayoutChangedArgs, LayoutType, RTabPanel, ModulesUpdateArgs } from "components/Layout";
import { ModuleInfo } from "@core/types";
import { parseJSON, isArray } from "utils";

export type LayoutOptions = {
  layout: string;
  views: ModuleInfo[];
  allViews: ModuleInfo[];
  mainPanelRef: RefObject<RSplitPanel | RBoxPanel>;
  tabPanelRef?: RefObject<RTabPanel>;
  viewLoaderRef: RefObject<RWidgetLoader>;
  setConfig?: (config: SolidConfig, localOnly?: boolean) => void;
  isLoaded?: boolean;
};

type LayoutResult = {
  allViewsLayoutRef: RefObject<LayoutType>;
  loadLayout: (config: SolidConfig, layoutJSON: string, viewsChanged: boolean) => void;
  savePendingLayout: (config: SolidConfig) => void;
};

type Layout = LayoutType & {
  allViews?: LayoutType;
};

export function useLayout({ layout, views, allViews, mainPanelRef, tabPanelRef, viewLoaderRef, setConfig, isLoaded }: LayoutOptions): LayoutResult {
  const allViewsRef = useRef(allViews);
  const allViewsLayoutRef = useRef<LayoutType>(getAllViewsLayout(layout));
  const dbLayoutRef = useRef("");
  const newLayoutRef = useRef("");
  const newLayoutLocalRef = useRef<boolean | undefined>();
  const layoutSubRef = useRef("");
  const layoutIntervalRef = useRef<NodeJS.Timeout>();
  const layoutLoadedRef = useRef(!!isLoaded);

  useEffect(useCallback(() => {
    layoutIntervalRef.current = globalThis.setInterval(() => saveLayoutToDb(), 2000);
    const mainPanel = mainPanelRef.current;
    const viewLoader = viewLoaderRef.current;
    if (mainPanel && layoutSubRef.current) {
      mainPanel.layoutChanged.subscribe(layoutSubRef.current, onLayoutChanged);
    }
    const id = `${Date.now()}.${Math.random()}`;
    if (viewLoader) {
      viewLoader.modulesUpdated.subscribe(id, onViewsUpdated);
    }

    return function cleanup() {
      if (mainPanel && layoutSubRef.current) {
        mainPanel.layoutChanged.unsubscribe(layoutSubRef.current);
      }
      if (viewLoader) {
        viewLoader.modulesUpdated.unsubscribe(id);
      }
      if (layoutIntervalRef.current) {
        clearInterval(layoutIntervalRef.current);
      }
    };
  }, [views, layout, setConfig]));

  useEffect(() => {
    if (newLayoutRef.current) {
      return;
    }
    const layoutObj = parseJSON<Layout>(layout);
    if (layoutObj?.allViews) {
      allViewsLayoutRef.current = layoutObj.allViews;
      restoreAllViewsLayout(layoutObj.allViews);
    }
  }, [layout]);

  useEffect(useCallback(() => {
    const oldViews = allViewsRef.current;
    allViewsRef.current = allViews;

    const newViews = allViews.filter(v => !oldViews.some(ov => ov.id === v.id) && allViews.some(av => av.id === v.originalViewId));
    const allLayout = allViewsLayoutRef.current;
    let updated = false;
    for (const view of newViews) {
      if (view.originalViewId && !allLayout[view.id] && allLayout[view.originalViewId]) {
        allLayout[view.id] = copyLayout(view.originalViewId, view.id, allLayout[view.originalViewId]);
        viewLoaderRef?.current?.restoreWidgetLayout(view.id, allLayout[view.id]);
        updated = true;
      }
    }

    if (updated && setConfig) {
      const layoutObj = parseJSON<LayoutType>(newLayoutRef.current || layout) ?? {};
      const newLayout = { ...layoutObj, allViews: allViewsLayoutRef.current };
      setConfig({ layout: JSON.stringify(newLayout) });
    }
  }, [allViews, layout, setConfig]), [allViews]);

  function getAllViewsLayout(layout: string): LayoutType {
    const layoutObj = parseJSON<LayoutType>(layout);
    if (!layoutObj) {
      return {};
    }
    return layoutObj.allViews ?? {};
  }

  const onLayoutChanged = useCallback(({ id: widgetId, layout }: LayoutChangedArgs): void => {
    const current: Layout = { ...layout, allViews: allViewsLayoutRef.current };
    let localOnly = false;
    const newLayout = produce(current, draftLayout => {
      if (mainPanelRef.current) {
        if (!draftLayout.allViews) {
          draftLayout.allViews = {};
        }
        const allViews = allViewsRef.current.concat(views);
        for (const { id, isLocal } of allViews) {
          const viewLayout = mainPanelRef.current.getWidgetLayout(id);
          if (viewLayout && viewLayout[id]) {
            draftLayout.allViews[id] = viewLayout[id];
            if (isWidgetInViewLayout(widgetId, viewLayout[id]) && isLocal) {
              localOnly = true;
            }
          }
        }
      }
    });
    allViewsLayoutRef.current = newLayout.allViews ?? {};
    newLayoutRef.current = JSON.stringify(newLayout);
    newLayoutLocalRef.current = (newLayoutLocalRef.current ?? true) && localOnly;
  }, [views]);

  function isWidgetInViewLayout(widgetId: string, viewLayout: LayoutType): boolean {
    if (!viewLayout.widgets) {
      return false;
    }
    if (viewLayout.widgets.hasOwnProperty(widgetId)) {
      return true;
    }
    for (const id in viewLayout.widgets) {
      if (isWidgetInViewLayout(widgetId, viewLayout.widgets[id])) {
        return true;
      }
    }
    return false;
  }

  const saveLayoutToDb = useCallback((): void => {
    if (newLayoutRef.current) {
      setConfig && setConfig({ layout: newLayoutRef.current }, newLayoutLocalRef.current);
      newLayoutRef.current = "";
      newLayoutLocalRef.current = undefined;
    }
  }, [setConfig]);

  const onViewsUpdated = useCallback(({ modules }: ModulesUpdateArgs): void => {
    if (dbLayoutRef.current) {
      restoreLayout(dbLayoutRef.current);
      dbLayoutRef.current = "";
    }
    const obj = parseJSON<Layout>(newLayoutRef.current || layout);
    if (obj && layoutLoadedRef.current) {
      const layoutObj = obj;
      const { allViews } = layoutObj;
      if (allViews && viewLoaderRef?.current) {
        const suffixes = ["_selectWidgets", "_preview"];
        for (const suffix of suffixes) {
          const filteredModules: ModuleInfo[] = modules.filter(m => m.id.endsWith(suffix) && !allViews[m.id]);
          for (const module of filteredModules) {
            const viewId = module.id.replace(suffix, "");
            if (allViews[viewId]) {
              allViews[module.id] = copyToSuffixedLayout(viewId, allViews[viewId], suffix);
              viewLoaderRef.current.restoreWidgetLayout(module.id, allViews[module.id]);
            }
          }
        }
      }
      tabPanelRef?.current?.restoreTabOrder(layoutObj);
    }
    subscribeLayoutChanged();
  }, [views, layout]);

  function restoreAllViewsLayout(allViewsLayout: LayoutType): void {
    for (const id in allViewsLayout) {
      viewLoaderRef?.current?.restoreWidgetLayout(id, allViewsLayout[id]);
    }
  }

  function loadLayout(config: SolidConfig, layoutJSON: string, viewsChanged: boolean): void {
    if (layoutLoadedRef.current) {
      return;
    }
    layoutLoadedRef.current = true;
    const layout = parseJSON<Layout>(layoutJSON);
    if (layout && layout.allViews) {
      allViewsLayoutRef.current = layout.allViews;
    }
    if (!viewsChanged || !viewLoaderRef.current) {
      restoreLayout(layoutJSON);
      subscribeLayoutChanged();
    }
    else {
      dbLayoutRef.current = layoutJSON || "";
    }
  }

  const subscribeLayoutChanged = useCallback(() => {
    if (mainPanelRef.current) {
      if (!layoutSubRef.current) {
        layoutSubRef.current = `${Date.now()}.${Math.random()}`;
      }
      mainPanelRef.current.layoutChanged.subscribe(layoutSubRef.current, onLayoutChanged);
    }
  }, [views]);

  function restoreLayout(layoutJSON: string | undefined): void {
    if (!mainPanelRef.current || newLayoutRef.current) {
      return;
    }
    const layout = parseJSON<Layout>(layoutJSON);
    if (layout) {
      mainPanelRef.current.restoreLayout(layout);
    }
  }

  function savePendingLayout(config: SolidConfig): void {
    if (newLayoutRef.current) {
      config.layout = newLayoutRef.current;
      newLayoutRef.current = "";
    }
  }

  return { allViewsLayoutRef, loadLayout, savePendingLayout };
}

export function copyFromSuffixedLayout(layout: LayoutType, suffix: string): LayoutType {
  const result = JSON.parse(JSON.stringify(layout));
  if (result.widgets) {
    for (const widgetId in result.widgets) {
      const newId = widgetId.replace(suffix, "");
      result.widgets[newId] = copyFromSuffixedLayout(result.widgets[widgetId], suffix);
      delete result.widgets[widgetId];
    }
  }
  return result;
}

export function copyFromSelectWidgetsLayout(layout: LayoutType): LayoutType {
  return copyFromSuffixedLayout(layout, "_selectWidgets");
}

export function copyToSuffixedLayout(viewId: string, layout: LayoutType, suffix: string): LayoutType {
  return copyLayout(viewId, viewId + suffix, layout);
}

export function copyToSelectWidgetsLayout(viewId: string, layout: LayoutType): LayoutType {
  return copyToSuffixedLayout(viewId, layout, "_selectWidgets");
}

function copyLayout(viewId: string, newViewId: string, layout: LayoutType): LayoutType {
  const result = JSON.parse(JSON.stringify(layout));
  if (result.widgets) {
    for (const widgetId in result.widgets) {
      const newId = widgetId.replace(viewId, newViewId);
      result.widgets[newId] = copyLayout(viewId, newViewId, result.widgets[widgetId]);
      delete result.widgets[widgetId];
    }
  }
  return result;
}

export function getViewLayout(viewId: string, layout: string): LayoutType | undefined {
  const layoutObj = parseJSON<Layout>(layout);
  return layoutObj?.allViews ? layoutObj?.allViews[viewId] : undefined;
}

export function compareLayout(layout1: LayoutType, layout2: LayoutType, threshold: number = 0.01): boolean {
  for (const key of Object.keys(layout1).concat(Object.keys(layout2))) {
    if (!!layout1[key] !== !!layout2[key]) {
      return false;
    }
    if (key === "sizes") {
      const arr1 = layout1[key];
      const arr2 = layout2[key];
      if (isArray<number>(arr1, "number") && isArray<number>(arr2, "number")) {
        if (arr1.length !== arr2.length) {
          return false;
        }
        for (let i = 0; i < arr1.length; i++) {
          if (Math.abs(arr1[i] - arr2[i]) >= threshold) {
            return false;
          }
        }
      }
    }
    else if (typeof layout1[key] === "object" && typeof layout2[key] === "object") {
      if (!compareLayout(layout1[key], layout2[key], threshold)) {
        return false;
      }
    }
    else if (layout1[key] !== layout2[key]) {
      return false;
    }
  }
  return true;
}
