import React, { useState, useEffect, useRef, useCallback } from "react";
import { useLocation, useNavigate, useParams } from "react-router-dom";
// import { Apollo } from "@core/api";
import { RWidget, RBoxPanel, RWidgetLoader } from "components/Layout";
import { useStore } from "@core/store";
import { useConfig, useLayout, SolidConfig } from "@core/store/actions";
import WithQueryStatus from "components/WithQueryStatus";
import "utils/icons";
import SidePanel from "components/SidePanel";
import StatusBar from "components/StatusBar";
import { MessageId } from "electron/types";
import {SIDE_PANEL_WIDTH} from "components/SidePanel/info";
import CloseConfirm from "components/View/Manager/CloseConfirm";
import { ModuleInfo, RouteParams } from "@core/types";
import { useDeviceActions, useImageMapSubscription, useLabelsSubscription, useHierarchiesSubscription, useAuditEntriesSubscription } from "@core/actions";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { ShepherdTour } from "react-shepherd";
import "shepherd.js/dist/css/shepherd.css";
import {linkActivationSteps} from "tour/link-activation-tour";
import {tourOptions} from "tour/tourOptions";
import { handleDefaultView } from "@core/utils";
import { SocketEventType } from "@generated/graphql";
//import {createDeviceTour} from "tour/create-device-tour";

function Root() {
  const params = useParams<RouteParams>();
  const { pathname } = useLocation();
  const navigate = useNavigate();
  const { store: { session: { isLoggedIn, info: sessionInfo }, socketEvent } } = useStore();
  const { updateCachedDevices } = useDeviceActions({});
  const { views, allViews, layout, loading, error, updateError, setConfig, loadFromDbEvent, routesViews } = useConfig({ loadFromDb: isLoggedIn, onLoadedLayout });
  const mainPanelRef = useRef<RBoxPanel>(null);
  const viewLoaderRef = useRef<RWidgetLoader>(null);
  const sidePanelWidgetRef = useRef<RWidget>(null);
  const { allViewsLayoutRef, loadLayout, savePendingLayout } = useLayout({ layout, views, allViews, mainPanelRef, viewLoaderRef, setConfig });
  const [closeView, setCloseView] = useState<ModuleInfo | undefined>();

  useImageMapSubscription();
  useLabelsSubscription();
  useHierarchiesSubscription();
  useAuditEntriesSubscription();

  const modules = handleDefaultView(views, routesViews, params, pathname);

  useEffect(() => {
    if (pathname === "/main" && modules && modules.length > 0) {
      navigate(`/view/${modules[0].id}`, { replace: true });
    }
  }, [pathname, modules]);

  // useEffect(() => {
  //   if (Apollo.subscriptionClient) {
  //     Apollo.subscriptionClient.close(false, true);
  //   }
  // }, [isLoggedIn]);

  useEffect(() => {
    if (!socketEvent) return;
    const { type, timeStamp } = socketEvent;
    if (type === SocketEventType.Reconnected) {
      updateCachedDevices(timeStamp);
    }
  }, [socketEvent]);

  useEffect(() => {
    if (sessionInfo?.realm?.name) {
      const title = `Solid - ${sessionInfo?.realm?.name}`;
      if (window.ipcRenderer) {
        window.ipcRenderer.send(MessageId.SetMainWindowTitle, title);
      }
      else {
        document.title = title;
      }
    }
  }, [sessionInfo]);

  useEffect(() => {
    updateError && Log.error(`${__("Configuration update error")}: ${updateError.message}`, 3000);
  }, [updateError]);

  function onLoadedLayout(config: SolidConfig, layoutJSON: string, viewsChanged: boolean): void {
    loadLayout(config, layoutJSON, viewsChanged);
  }

  function onWriteToDb(config: SolidConfig): void {
    savePendingLayout(config);
  }

  const onChildClosing = useCallback(({ id }) => {
    const view = views.find(v => v.id === id);
    if (!view || !view.hasUnsavedChanges) {
      viewLoaderRef.current?.closeChild(id);
      return;
    }
    setCloseView(view);
  }, [views]);

  function onCloseConfirm(id: string): void {
    viewLoaderRef.current?.closeChild(id);
    setCloseView(undefined);
  }

  return (
    <WithQueryStatus loading={loading} error={error}>
      { isLoggedIn &&
        <ShepherdTour steps={linkActivationSteps as any} tourOptions={tourOptions}>
          <RBoxPanel id="main" rootElementId="root" ref={mainPanelRef} direction="top-to-bottom">
            <RBoxPanel id="main-content" direction="left-to-right">
              <RWidget id="main_left" name="Views" minWidth={SIDE_PANEL_WIDTH} maxWidth={SIDE_PANEL_WIDTH} ref={sidePanelWidgetRef}>
                <SidePanel sidePanelWidgetRef={sidePanelWidgetRef} mainPanelRef={mainPanelRef} viewLoaderRef={viewLoaderRef} onWriteToDb={onWriteToDb}
                  loadFromDbEvent={loadFromDbEvent}/>
              </RWidget>
              <RBoxPanel id="main_right" direction="top-to-bottom">
                <RBoxPanel id="main_box" direction="top-to-bottom">
                  <RWidgetLoader ref={viewLoaderRef} modules={modules} layoutRef={allViewsLayoutRef} showHeader onChildClosing={onChildClosing}/>
                </RBoxPanel>
              </RBoxPanel>
            </RBoxPanel>
            <RWidget id="main_status" minHeight={28} maxHeight={28}><StatusBar loaderRef={viewLoaderRef}/></RWidget>
          </RBoxPanel>
        </ShepherdTour>
      }

      { !!closeView && <CloseConfirm view={closeView} onConfirm={onCloseConfirm} onCancel={() => setCloseView(undefined)}/> }
    </WithQueryStatus>
  );
}

export default Root;
