import { useEffect, useMemo, useState } from "react";
import moment from "moment";
import { ApolloError } from "@apollo/client";
import { useStore } from "@core/store";
import { formatDateTime } from "@core/utils";
import {
  AodRequest,
  AodRequestStatus,
  AodRequestType,
  DeviceFunctionalAspectType,
  DeviceListByAspectTypesQuery,
  useAodCancelMutation,
  useDeviceListByAspectTypesQuery,
  useAodUpdateSubscription,
  AodRequestFilesListEntry,
  useAodUpdatedListLazyQuery
} from "@generated/graphql";
import { Log, __ } from "@solid/libs";
import { origin } from "@core/api";
import { DeviceList } from "./deviceActions";

const currentLanguage: string = localStorage.getItem("language") ?? "";

export enum PeriodValues {
  EightHours = 8,
  OneDay = 24,
  TwoDays = 48
}

export enum VDMContentType {
  "Download" = "Download",
  "Request" = "Request"
}

export type VDMRequestStatus = AodRequestStatus | "ALL";

export type FilterPropType = PeriodValues | VDMRequestStatus | AodRequestType | string | boolean;

export type VDMFilterState = {
  interval: PeriodValues;
  status: VDMRequestStatus;
  type: AodRequestType;
  onlyEvents: boolean;
  searchText: string;
};

export const periodOptions = [
  { key: "8", value: PeriodValues.EightHours, text: __("8 hours") },
  { key: "24", value: PeriodValues.OneDay, text: __("one day") },
  { key: "48", value: PeriodValues.TwoDays, text: __("two days") }
];

export const statusOptions = [
  { key: "ALL", value: "ALL", text: __("ALL") },
  { key: AodRequestStatus.Completed, value: AodRequestStatus.Completed, text: __("COMPLETED") },
  { key: AodRequestStatus.Failed, value: AodRequestStatus.Failed, text: __("FAILED") },
  { key: AodRequestStatus.Paused, value: AodRequestStatus.Paused, text: __("PAUSED") },
  { key: AodRequestStatus.Processing, value: AodRequestStatus.Processing, text: __("PROCESSING") },
  { key: AodRequestStatus.Queued, value: AodRequestStatus.Queued, text: __("QUEUED") },
];

export const requestTypeOptions = [
  { key: AodRequestType.Manual, value: AodRequestType.Manual, text: __("Manual") },
  { key: AodRequestType.Auto, value: AodRequestType.Auto, text: __("Auto") },
];

export const defaultVDMFilter: VDMFilterState = {
  interval: PeriodValues.EightHours,
  status: "ALL",
  type: AodRequestType.Manual,
  onlyEvents: false,
  searchText: ""
};

export const statusCancelsRequests = [AodRequestStatus.Queued, AodRequestStatus.Processing, AodRequestStatus.Exporting, AodRequestStatus.Paused];
export const statusPauseRequest = [AodRequestStatus.Processing, AodRequestStatus.Queued, AodRequestStatus.Exporting];

export function formatBytes(bytes: number | undefined | null): string | undefined {
  if (!bytes) return undefined;
  if (bytes === 0) return "0 Bytes";

  const count = 1024;
  const sizes = ["Bytes", "KB", "MB", "GB"];

  const log = Math.floor(Math.log(bytes) / Math.log(count));

  return `${parseFloat((bytes / Math.pow(count, log)).toFixed(2))} ${sizes[log]}`;
}

export function formatDownloadingTime(eta: number | undefined | null): string {
  if (eta === null || typeof eta === "undefined" || eta === -1) return __("Stalled");
  if (eta === 0) return __("Very soon");

  let expTime = "~ ";
  const d = Math.floor(eta / (24 * 3600));
  if (d > 0) {
    return expTime += `${d} day(s)`;
  }

  const h = Math.floor(eta / 3600);
  if (h > 0) {
    return expTime += `${h} hrs`;
  }

  const m = Math.floor(eta / 60);
  if (m > 0) {
    return expTime += `${m} min`;
  }

  return expTime += `${eta} sec`;
}

export function formatExpirationTime(expiredDate: Date): string {
  const duration = getDuration(new Date(Date.now()), expiredDate);

  if (duration.hours() > 0 || duration.days() > 0) {
    const hours = duration.hours() + duration.days() * 24;
    return __("Expire after {{time}} hour(s)", { time: hours });
  }
  if (duration.minutes() > 0) {
    return __("Expire after {{time}} minute(s)", { time: duration.minutes() });
  }

  return __("Expire very soon");
}

function getDevice(deviceId: string, cameraList?: DeviceListByAspectTypesQuery) {
  const currentDevice = cameraList && cameraList.devicesByAspectTypes
    .find(device => device.id === deviceId);
  return currentDevice;
}

function getDeviceNameById(deviceId: string, cameraList?: DeviceListByAspectTypesQuery): string | undefined {
  const currentDevice = cameraList && cameraList.devicesByAspectTypes
    .filter(device => device.id === deviceId)
    .map(device => device.name)
    .toString();
  return currentDevice;
}

function getDuration(start: Date, end: Date): moment.Duration {
  const startTime = moment(start);
  const endTime = moment(end);
  const duration = moment.duration(endTime.diff(startTime));

  return duration;
}

function formatDuration(start: Date, end: Date) {
  const duration = getDuration(start, end);

  const h = duration.hours() > 0 ? `${duration.hours().toString()}h ` : "";
  const m = duration.minutes() > 0 ? `${duration.minutes().toString()}m` : "";
  const s = `${duration.minutes() === 0 && duration.hours() === 0 ? duration.seconds().toString() + " sec" : ""}`;

  return `${h}${m}${s}`;
}

function getDownloadFileContent(key: string, data: AodRequestFilesListEntry): string {
  switch (key) {
    case "from":
      return formatDateTime(new Date(data.startTime), currentLanguage);
    case "duration":
      return formatDuration(new Date(data.startTime), new Date(data.endTime));
    default:
      return "";
  }
}

function getRequestContent(key: string, data: AodRequest, devData: DeviceListByAspectTypesQuery | undefined) {
  switch (key) {
    case "id":
      return data.id;
    case "eventid":
      return data.eventId || "N/A";
    case "deviceName":
      return getDeviceNameById(data?.deviceId, devData) ?? "";
    case "from":
      return data.startTime ? formatDateTime(new Date(data.startTime), currentLanguage) : "";
    case "duration":
      return formatDuration(data.startTime, data.endTime);
    case "status":
      return data.status;
    default:
      return "";
  }
}

function sortList(
  list: AodRequest[] | undefined,
  cameraList: DeviceListByAspectTypesQuery | undefined,
  filter: VDMFilterState,
): AodRequest[] {
  if (!list) {
    return [];
  }

  const { onlyEvents, searchText, type, status } = filter;
  let filteredList = list;

  if (onlyEvents) {
    filteredList = filteredList.filter(request => request.eventId);
  }
  if (status !== "ALL") {
    if (status === AodRequestStatus.Processing) {
      filteredList = filteredList.filter(request => [AodRequestStatus.Processing, AodRequestStatus.Exporting].indexOf(request.status) !== -1);
    }
    else {
      filteredList = filteredList.filter(request => request.status === status);
    }
  }
  if (searchText.length > 0) {
    const text = searchText.toLowerCase();
    filteredList = filteredList.filter(request =>
      getRequestContent("deviceName", request, cameraList).toLowerCase().includes(text) ||
      getRequestContent("id", request, cameraList).toLowerCase().includes(text) ||
      // getRequestContent("eventid", request, undefined).toLowerCase().includes(text) ||
      getRequestContent("from", request, undefined).toLowerCase().includes(text) ||
      getRequestContent("duration", request, undefined).toLowerCase().includes(text) ||
      getRequestContent("status", request, undefined).toLowerCase().includes(text)
    );
  }

  return filteredList
    .filter(request => request.type === type && request.status !== AodRequestStatus.Canceled)
    .sort((a, b) => new Date(b.updatedAt ?? b.createdAt).getTime() - new Date(a.updatedAt ?? a.createdAt).getTime());
}

function mergeRequests(list: AodRequest[], updatedList: AodRequest[]): AodRequest[] {
  if (updatedList.length > 0) {
    const prevList = list.filter(req => !updatedList?.some(updReq => updReq?.id === req.id));

    if (prevList) {
      return [...prevList, ...updatedList];
    }
  }

  if (list) {
    return list;
  }

  return [];
}

function getUTCOffset(): string {
  const currentLocalDate = new Date();
  const tzOffset = currentLocalDate.getTimezoneOffset();
  return tzOffset > 0 ? `-${Math.abs(tzOffset)}` : Math.abs(tzOffset).toString();
}

export function downloadAll(files: AodRequestFilesListEntry[]) {
  for (let i = 0; i < files.length; i++) {
    downloadOne(files[i]);
  }
}

export function downloadOne(file: AodRequestFilesListEntry) {
  const url = `${origin}${file.href}?tz=${getUTCOffset()}`;
  const a = document.createElement("a");
  a.href = url;
  a.click();
  a.remove();
}


interface CellContentData {
  content: VDMContentType;
  rowData: AodRequest | AodRequestFilesListEntry;
}
interface RequestContent extends CellContentData {
  content: VDMContentType.Request;
  rowData: AodRequest;
}

interface DownloadFileContent extends CellContentData {
  content: VDMContentType.Download;
  rowData: AodRequestFilesListEntry;
}

type CellContentTextProps = {
  dataKey: string;
  data: RequestContent | DownloadFileContent;
};

type VideoDeliveryArgs = {
  waitRequest?: boolean;
  interval?: number;
  filter?: VDMFilterState;
};

type VideoDeliveryResult = {
  error: ApolloError | undefined;
  pending: boolean;
  list: AodRequest[];
  userRequests: AodRequest[] | undefined;
  getCellContentText: (data: CellContentTextProps) => string;
  updateRequests: (requests: AodRequest[]) => void;
  cancelAll: () => Promise<void>;
  getDeviceItem: (data: AodRequest) => DeviceList | undefined;
};

function useVideoDelivery({
  waitRequest = false,
  interval,
  filter
}: VideoDeliveryArgs): VideoDeliveryResult {
  const { store: { session: { info } } } = useStore();
  const [cancelRequest] = useAodCancelMutation();
  const [getList, { data, loading, error }] = useAodUpdatedListLazyQuery();
  const { loading: devLoading, error: devError, data: devData } = useDeviceListByAspectTypesQuery({
    variables: { types: [{ type: DeviceFunctionalAspectType.Media}] }
  });
  const { data: subData } = useAodUpdateSubscription({
    variables: {
      filter: { interval: PeriodValues.TwoDays }
    },
    skip: !waitRequest
  });
  const [dispatchedRequest, setDispatchedRequest] = useState<AodRequest[]>();

  const updatedRequests: AodRequest[] = subData?.aodUpdate && subData.aodUpdate.length > 0 ? subData.aodUpdate : [];

  const userRequests: AodRequest[] | undefined = useMemo(() => {
    if (!info?.user || !updatedRequests) {
      return undefined;
    }

    return updatedRequests.filter(request =>
      request.userId === info.user.id &&
      request.type === AodRequestType.Manual &&
      (request.status === AodRequestStatus.Completed || request.status ===  AodRequestStatus.Failed)
    );
  }, [updatedRequests, info]);

  const list = useMemo(() => {
    if (!filter) {
      return [];
    }

    let list = data?.aodGetUpdatedList || [];

    if (dispatchedRequest) {
      list = mergeRequests(list, dispatchedRequest);
    }

    if (updatedRequests.length > 0) {
      let filteredRequests = [...updatedRequests];

      if (dispatchedRequest?.some(req => req.status === AodRequestStatus.Canceled)) {
        const canceledIds = dispatchedRequest
          .filter(req => req.status === AodRequestStatus.Canceled)
          .map(req => req.id);

        filteredRequests = filteredRequests.filter(req => !canceledIds.includes(req.id));
      }

      list = mergeRequests(list, filteredRequests);
    }

    return sortList(list, devData, filter);
  }, [data, devData, filter, updatedRequests, dispatchedRequest]);

  useEffect(() => {
    if (!interval) {
      return;
    }

    const updatedFrom = new Date(Date.now() - interval * 3600 * 1000);
    getList({
      variables: { updatedFrom },
      fetchPolicy: "cache-first"
    });
  }, [interval]);

  function updateRequests(requests: AodRequest[]) {
    setDispatchedRequest(requests);
  }

  async function cancelAll(): Promise<void> {
    if (!info || list.length === 0 || !filter) {
      return;
    }

    const { user: { id, isAdmin } } = info;
    const requestsToCancel = list.filter(req =>
      statusCancelsRequests.indexOf(req.status) !== -1 &&
      req.type === filter.type
    );

    if (requestsToCancel.length === 0) {
      Log.info(__("No requests to cancel"));
      return;
    }

    if (isAdmin) {
      const reqIds: string[] = requestsToCancel.map(req => req.id);
      try {
        const result = await cancelRequest({ variables: { requestIds: reqIds } });
        if (result.data?.aodCancel) {
          const canceledRequests = requestsToCancel.map((req) => {
            return {...req, ...{status: AodRequestStatus.Canceled}};
          });
          setDispatchedRequest(canceledRequests);
          return;
        }
      }
      catch (e: any) {
        Log.error(e);
        return;
      }
    }

    const availableRequests = requestsToCancel.filter(req => req.userId === id);
    if (availableRequests.length === 0) {
      Log.info(__("No requests to cancel"));
      return;
    }
    const reqIds = availableRequests.map(req => req.id);
    try {
      const result = await cancelRequest({ variables: { requestIds: reqIds } });
      if (result.data?.aodCancel) {
        const canceledRequests = availableRequests.map((req) => {
          return { ...req, ...{ status: AodRequestStatus.Canceled } };
        });
        setDispatchedRequest(canceledRequests);
        return;
      }
    }
    catch (e: any) {
      Log.error(e);
    }
  }

  function getDeviceItem(data: AodRequest) {
    const device = getDevice(data.deviceId, devData);
    return device;
  }

  function getCellContentText({ dataKey, data }: CellContentTextProps) {
    switch (data.content) {
      case VDMContentType.Download:
        return getDownloadFileContent(dataKey, data.rowData);
      case VDMContentType.Request:
        return getRequestContent(dataKey, data.rowData, devData);
      default:
        return "";
    }
  }


  return {
    error: error || devError,
    pending: loading || devLoading,
    list,
    getCellContentText,
    userRequests,
    updateRequests,
    cancelAll,
    getDeviceItem
  };
}

export default useVideoDelivery;
