import { useEffect } from "react";
import { ApolloClient, useApolloClient } from "@apollo/client";
import { v1 as uuid } from "uuid";
import { UpdateImageMapConfig } from "@libs/imagemap";
import * as imt from "@libs/imagemap";
import {
  InsertImageMapSetDocument,
  InsertImageMapSetMutation,
  InsertImageMapSetMutationVariables,
  RenameImageMapSetDocument,
  RenameImageMapSetMutation,
  RenameImageMapSetMutationVariables,
  DeleteImageMapSetDocument,
  DeleteImageMapSetMutation,
  DeleteImageMapSetMutationVariables,
  InsertImageMapLayerDocument,
  InsertImageMapLayerMutation,
  InsertImageMapLayerMutationVariables,
  RenameImageMapLayerDocument,
  RenameImageMapLayerMutation,
  RenameImageMapLayerMutationVariables,
  DeleteImageMapLayerDocument,
  DeleteImageMapLayerMutation,
  DeleteImageMapLayerMutationVariables,
  UpdateImageMapDocument,
  UpdateImageMapMutation,
  UpdateImageMapMutationVariables,
  InsertImageMapObjectDocument,
  InsertImageMapObjectMutation,
  InsertImageMapObjectMutationVariables,
  UpdateImageMapObjectDocument,
  UpdateImageMapObjectMutation,
  UpdateImageMapObjectMutationVariables,
  DeleteImageMapObjectDocument,
  DeleteImageMapObjectMutation,
  DeleteImageMapObjectMutationVariables,
  useImageMapUpdateSubscription,
  ImageMapConfigQuery,
  ImageMapConfigQueryVariables,
  ImageMapConfigDocument,
  ImageMapConfig,
  ImageMapUpdateAction,
  ImageMapImageDocument,
  ImageMapImageQuery,
  ImageMapImageQueryVariables
} from "@generated/graphql";
import { useStore } from "@core/store";
import { clone } from "@solid/libs/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

const version = "v2";
export const imageMapVersion = version;
export const lazyLoadImages = true;

type ImageMapActionsResult = {
  saveImageMapConfig: UpdateImageMapConfig;
  updateImageMapCache: (config: imt.ImageMapConfig) => void;
  getImageMapImage: (mapId: string) => Promise<string>;
};

const updateId = uuid();

/* eslint-disable arrow-body-style, no-return-await, no-empty */

export function useImageMapActions(): ImageMapActionsResult {
  const client = useApolloClient();

  const saveImageMapConfig: UpdateImageMapConfig = {
    appendSet: withErrorHandler(async (setId, { name, layer }: imt.ImageMapSet): Promise<boolean> => {
      const { data, errors } = await client.mutate<InsertImageMapSetMutation, InsertImageMapSetMutationVariables>({
        mutation: InsertImageMapSetDocument,
        variables: {
          version,
          updateId,
          set: {
            id: setId,
            name,
            layers: Object.keys(layer).map(id => {
              const { name, map } = layer[id];
              return {
                id,
                name,
                maps: Object.keys(map).map(id => {
                  const { name, order, base64, object } = map[id];
                  return {
                    id,
                    name,
                    order,
                    base64,
                    objects: Object.keys(object).map(id => {
                      const { obj, position } = object[id];
                      return { id, objectId: obj, position };
                    })
                  };
                })
              };
            })
          }
        }
      });
      if (data) {
        return data.insertImageMapSet;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
    renameSet: withErrorHandler(async (setId: string, name: string): Promise<boolean> => {
      const { data, errors } = await client.mutate<RenameImageMapSetMutation, RenameImageMapSetMutationVariables>({
        mutation: RenameImageMapSetDocument,
        variables: { version, updateId, setId, name }
      });
      if (data) {
        return data.renameImageMapSet;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
    deleteSet: withErrorHandler(async (setId: string): Promise<boolean> => {
      const { data, errors } = await client.mutate<DeleteImageMapSetMutation, DeleteImageMapSetMutationVariables>({
        mutation: DeleteImageMapSetDocument,
        variables: { version, updateId, setId }
      });
      if (data) {
        return data.deleteImageMapSet;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),

    appendLayer: withErrorHandler(async (setId: string, layerId: string, { name, map }: imt.ImageMapLayer): Promise<boolean> => {
      const { data, errors } = await client.mutate<InsertImageMapLayerMutation, InsertImageMapLayerMutationVariables>({
        mutation: InsertImageMapLayerDocument,
        variables: {
          version,
          updateId,
          setId,
          layer: {
            id: layerId,
            name,
            maps: Object.keys(map).map(id => {
              const { name, order, base64, object } = map[id];
              return {
                id,
                name,
                order,
                base64,
                objects: Object.keys(object).map(id => {
                  const { obj, position } = object[id];
                  return { id, objectId: obj, position };
                })
              };
            })
          }
        }
      });
      if (data) {
        return data.insertImageMapLayer;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
    renameLayer: withErrorHandler(async (layerId: string, name: string): Promise<boolean> => {
      const { data, errors } = await client.mutate<RenameImageMapLayerMutation, RenameImageMapLayerMutationVariables>({
        mutation: RenameImageMapLayerDocument,
        variables: { version, updateId, layerId, name }
      });
      if (data) {
        return data.renameImageMapLayer;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
    deleteLayer: withErrorHandler(async (layerId: string): Promise<boolean> => {
      const { data, errors } = await client.mutate<DeleteImageMapLayerMutation, DeleteImageMapLayerMutationVariables>({
        mutation: DeleteImageMapLayerDocument,
        variables: { version, updateId, layerId }
      });
      if (data) {
        return data.deleteImageMapLayer;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),

    updateMap: withErrorHandler(async (mapId: string, { name, order, base64, object }: imt.ImageMap): Promise<boolean> => {
      const { data, errors } = await client.mutate<UpdateImageMapMutation, UpdateImageMapMutationVariables>({
        mutation: UpdateImageMapDocument,
        variables: {
          version,
          updateId,
          map: {
            id: mapId,
            name,
            order,
            base64,
            objects: Object.keys(object).map(id => {
              const { obj, position } = object[id];
              return { id, objectId: obj, position };
            })
          }
        }
      });
      if (data) {
        return data.updateImageMap;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),

    appendObject: withErrorHandler(async (mapId: string, objectId: string, { obj, position }: imt.ImageMapObject): Promise<boolean> => {
      const { data, errors } = await client.mutate<InsertImageMapObjectMutation, InsertImageMapObjectMutationVariables>({
        mutation: InsertImageMapObjectDocument,
        variables: { version, updateId, mapId, object: { id: objectId, objectId: obj, position } }
      });
      if (data) {
        return data.insertImageMapObject;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
    updateObject: withErrorHandler(async (objectId: string, { obj, position }: imt.ImageMapObject): Promise<boolean> => {
      const { data, errors } = await client.mutate<UpdateImageMapObjectMutation, UpdateImageMapObjectMutationVariables>({
        mutation: UpdateImageMapObjectDocument,
        variables: { version, updateId, object: { id: objectId, objectId: obj, position } }
      });
      if (data) {
        return data.updateImageMapObject;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
    deleteObject: withErrorHandler(async (objectId: string): Promise<boolean> => {
      const { data, errors } = await client.mutate<DeleteImageMapObjectMutation, DeleteImageMapObjectMutationVariables>({
        mutation: DeleteImageMapObjectDocument,
        variables: { version, updateId, objectId }
      });
      if (data) {
        return data.deleteImageMapObject;
      }
      if (errors) {
        throw errors[0];
      }
      return false;
    }),
  };

  const updateImageMapCache = (config: imt.ImageMapConfig): void => {
    let query: ImageMapConfigQuery | null = null;
    try {
      query = client.readQuery<ImageMapConfigQuery, ImageMapConfigQueryVariables>({
        query: ImageMapConfigDocument,
        variables: { version, lazyLoadImages }
      });
    }
    catch {}

    const oldConfig = query?.imageMapConfig;
    if (!oldConfig) {
      return;
    }

    const newConfig: ImageMapConfig = {
      __typename: "ImageMapConfig",
      sets: Object.keys(config.set).map(setId => {
        const { name, layer } = config.set[setId];
        return {
          __typename: "ImageMapSet",
          id: setId,
          name,
          layers: Object.keys(layer).map(layerId => {
            const { name, map } = layer[layerId];
            return {
              __typename: "ImageMapLayer",
              id: layerId,
              name,
              maps: Object.keys(map).map(mapId => {
                const { name, order, base64, object } = map[mapId];
                return {
                  __typename: "ImageMap",
                  id: mapId,
                  name,
                  order,
                  base64: getBase64(setId, layerId, mapId, base64, oldConfig),
                  objects: Object.keys(object).map(id => {
                    const { obj, position } = object[id];
                    return {
                      __typename: "ImageMapObject",
                      id,
                      objectId: obj,
                      position: { x: position.x, y: position.y, __typename: "Position" }
                    };
                  })
                };
              })
            };
          })
        };
      })
    };

    client.writeQuery<ImageMapConfigQuery, ImageMapConfigQueryVariables>({
      query: ImageMapConfigDocument,
      variables: { version, lazyLoadImages },
      data: { imageMapConfig: newConfig }
    });
  };

  const getImageMapImage = async (mapId: string): Promise<string> => {
    try {
      const { data } = await client.query<ImageMapImageQuery, ImageMapImageQueryVariables>({
        query: ImageMapImageDocument,
        variables: { version, mapId }
      });
      return data.imageMapImage;
    }
    catch (e: any) {
      Log.error(`${__("Image load error")}: ${e.message}`);
      throw e;
    }
  };

  return { saveImageMapConfig, updateImageMapCache, getImageMapImage };
}

export function useImageMapSubscription(): void {
  const client = useApolloClient();
  const { store: { session: { isLoggedIn } } } = useStore();
  const { data, error } = useImageMapUpdateSubscription({ variables: { version }, skip: !isLoggedIn });

  useEffect(() => {
    error && console.error("Image map subscription error:", error);

    if (!data || data.imageMapUpdate.updateId === updateId) {
      return;
    }

    let query: ImageMapConfigQuery | null = null;
    try {
      query = client.readQuery<ImageMapConfigQuery, ImageMapConfigQueryVariables>({
        query: ImageMapConfigDocument,
        variables: { version, lazyLoadImages }
      });
    }
    catch {}

    if (!query?.imageMapConfig) {
      return;
    }

    const config: ImageMapConfig = clone(query.imageMapConfig);

    const { action, set, layer, map, object, setId, layerId, mapId, objectId, setName, layerName } = data.imageMapUpdate;
    switch (action) {
      case ImageMapUpdateAction.InsertSet:
        if (!set) {
          return;
        }
        if (config.sets.some(s => s.id === set.id)) {
          return;
        }
        config.sets.push(set);
        break;

      case ImageMapUpdateAction.UpdateSet:
        {
          if (!set) {
            return;
          }
          const setIndex = config.sets.findIndex(s => s.id === set.id);
          if (setIndex < 0) {
            return;
          }
          config.sets[setIndex] = set;
        }
        break;

      case ImageMapUpdateAction.DeleteSet:
        {
          const setIndex = config.sets.findIndex(s => s.id === setId);
          if (setIndex < 0) {
            return;
          }
          config.sets.splice(setIndex, 1);
        }
        break;

      case ImageMapUpdateAction.RenameSet:
        {
          if (setName === undefined || setName === null) {
            return;
          }
          const setIndex = config.sets.findIndex(s => s.id === setId);
          if (setIndex < 0) {
            return;
          }
          config.sets[setIndex].name = setName;
        }
        break;

      case ImageMapUpdateAction.InsertLayer:
        {
          if (!layer) {
            return;
          }
          const setIndex = config.sets.findIndex(s => s.id === setId);
          if (setIndex < 0 || config.sets[setIndex].layers.some(l => l.id === layer.id)) {
            return;
          }
          config.sets[setIndex].layers.push(layer);
        }
        break;

      case ImageMapUpdateAction.UpdateLayer:
        {
          if (!layer || !setId) {
            return;
          }
          const indices = getLayerIndices(config, setId, layer.id);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex } = indices;
          config.sets[setIndex].layers[layerIndex] = layer;
        }
        break;

      case ImageMapUpdateAction.DeleteLayer:
        {
          if (!setId || !layerId) {
            return;
          }
          const indices = getLayerIndices(config, setId, layerId);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex } = indices;
          config.sets[setIndex].layers.splice(layerIndex, 1);
        }
        break;

      case ImageMapUpdateAction.RenameLayer:
        {
          if (layerName === undefined || layerName === null || !setId || !layerId) {
            return;
          }
          const indices = getLayerIndices(config, setId, layerId);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex } = indices;
          config.sets[setIndex].layers[layerIndex].name = layerName;
        }
        break;

      case ImageMapUpdateAction.UpdateMap:
        {
          if (!map || !setId || !layerId) {
            return;
          }
          const indices = getMapIndices(config, setId, layerId, map.id);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex, mapIndex } = indices;
          config.sets[setIndex].layers[layerIndex].maps[mapIndex] = map;
        }
        break;

      case ImageMapUpdateAction.InsertObject:
        {
          if (!object || !setId || !layerId || !mapId) {
            return;
          }
          const indices = getMapIndices(config, setId, layerId, mapId);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex, mapIndex } = indices;
          if (config.sets[setIndex].layers[layerIndex].maps[mapIndex].objects.some(o => o.id === object.id)) {
            return;
          }
          config.sets[setIndex].layers[layerIndex].maps[mapIndex].objects.push(object);
        }
        break;

      case ImageMapUpdateAction.UpdateObject:
        {
          if (!object || !setId || !layerId || !mapId) {
            return;
          }
          const indices = getObjectIndices(config, setId, layerId, mapId, object.id);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex, mapIndex, objectIndex } = indices;
          config.sets[setIndex].layers[layerIndex].maps[mapIndex].objects[objectIndex] = object;
        }
        break;

      case ImageMapUpdateAction.DeleteObject:
        {
          if (!setId || !layerId || !mapId || !objectId) {
            return;
          }
          const indices = getObjectIndices(config, setId, layerId, mapId, objectId);
          if (!indices) {
            return;
          }
          const { setIndex, layerIndex, mapIndex, objectIndex } = indices;
          config.sets[setIndex].layers[layerIndex].maps[mapIndex].objects.splice(objectIndex, 1);
        }
        break;
    }

    client.writeQuery<ImageMapConfigQuery, ImageMapConfigQueryVariables>({
      query: ImageMapConfigDocument,
      variables: { version, lazyLoadImages },
      data: { imageMapConfig: config }
    });
  }, [data, error]);
}

export async function getImageMapConfig(client: ApolloClient<any>): Promise<ImageMapConfig> {
  const result = await client.query<ImageMapConfigQuery, ImageMapConfigQueryVariables>({
    query: ImageMapConfigDocument,
    variables: { version, lazyLoadImages }
  });
  return result.data.imageMapConfig;
}

function withErrorHandler<T extends (...args: any[]) => Promise<boolean>>(func: T): T {
  const wrapper = (...args: any[]): Promise<boolean> => {
    return new Promise<boolean>(async (resolve) => {
      try {
        const result = await func(...args);
        resolve(result);
      }
      catch (e: any) {
        console.error(e);
        Log.error(`${__("Image map update error")}: ${e.message}`);
        resolve(false);
      }
    });
  };
  return (wrapper as unknown) as T;
}

function getBase64(setId: string, layerId: string, mapId: string, base64: string, oldConfig: ImageMapConfig): string {
  if (base64 !== imt.ImageMapBase64.Loading && base64 !== imt.ImageMapBase64.Error) {
    return base64;
  }
  const indices = getMapIndices(oldConfig, setId, layerId, mapId);
  if (!indices) {
    return "";
  }
  const { setIndex, layerIndex, mapIndex } = indices;
  return oldConfig.sets[setIndex].layers[layerIndex].maps[mapIndex].base64;
}

function getLayerIndices(config: ImageMapConfig, setId: string, layerId: string): { setIndex: number, layerIndex: number } | undefined {
  const setIndex = config.sets.findIndex(set => set.id === setId);
  const layerIndex = setIndex >= 0 ? config.sets[setIndex].layers.findIndex(layer => layer.id === layerId) : -1;
  return setIndex >= 0 && layerIndex >= 0 ? { setIndex, layerIndex } : undefined;
}

function getMapIndices(config: ImageMapConfig, setId: string, layerId: string, mapId: string): { setIndex: number, layerIndex: number, mapIndex: number } | undefined {
  const setIndex = config.sets.findIndex(set => set.id === setId);
  const layerIndex = setIndex >= 0 ? config.sets[setIndex].layers.findIndex(layer => layer.id === layerId) : -1;
  const mapIndex = layerIndex >= 0 ? config.sets[setIndex].layers[layerIndex].maps.findIndex(map => map.id === mapId) : -1;
  return setIndex >= 0 && layerIndex >= 0 && mapIndex >= 0 ? { setIndex, layerIndex, mapIndex } : undefined;
}

function getObjectIndices(config: ImageMapConfig, setId: string, layerId: string, mapId: string, objectId: string):
{ setIndex: number, layerIndex: number, mapIndex: number, objectIndex: number } | undefined
{
  const setIndex = config.sets.findIndex(set => set.id === setId);
  const layerIndex = setIndex >= 0 ? config.sets[setIndex].layers.findIndex(layer => layer.id === layerId) : -1;
  const mapIndex = layerIndex >= 0 ? config.sets[setIndex].layers[layerIndex].maps.findIndex(map => map.id === mapId) : -1;
  const objectIndex = mapIndex >= 0 ?  config.sets[setIndex].layers[layerIndex].maps[mapIndex].objects.findIndex(object => object.id === objectId) : -1;
  return setIndex >= 0 && layerIndex >= 0 && mapIndex >= 0 && objectIndex >= 0 ? { setIndex, layerIndex, mapIndex, objectIndex } : undefined;
}
