import { useState, useEffect, useRef, useCallback } from "react";
import { useStore } from "@core/store";
import {
  AuditEntriesQuery,
  AuditEntriesWithSnapshotsQuery,
  AuditEntriesSubscriptionSubscription,
  useAuditEntriesLazyQuery,
  useAuditEntriesWithSnapshotsLazyQuery,
  ErrorInfo,
  SocketEventType,
  SocketEvent
} from "@generated/graphql";
import { PropType } from "utils";
import { UUID } from "@solid/types";
import { useAuditEntries } from "core/actions";

export type AuditEntry = PropType<AuditEntriesQuery, "auditEntries">[0];
export type AuditEntryEvent = PropType<AuditEntriesSubscriptionSubscription, "auditEntries">;
export type AuditEntryWithSnapshots = PropType<AuditEntriesWithSnapshotsQuery, "auditEntries">[0];

export function isWithSnapshots(entry: AuditEntry | AuditEntryWithSnapshots): entry is AuditEntryWithSnapshots {
  return !!(entry as AuditEntryWithSnapshots).snapshots;
}

export type EventLogOptions = {
  from?: Date;
  to?: Date;
  fromDays?: number;
  witnesses: UUID[];
  categories?: string[];
  start?: number;
  limit?: number;
  skipSubscription?: boolean;
  skipQuery?: boolean;
  withSnapshots?: boolean;
  widgetId: string;
};

export type EventLogResult = {
  entry?: AuditEntryEvent,
  entries?: AuditEntriesQuery["auditEntries"],
  queryLoading: boolean;
  subscriptionLoading: boolean;
  queryError?: Error;
  subscriptionError?: Error;
};

const resubscribeInterval = 5000;
const socketErrorInterval = 10000;

export function isError(obj: AuditEntryEvent): obj is ErrorInfo {
  return (obj as ErrorInfo).error !== undefined && (obj as ErrorInfo).error !== null;
}

export function useEventLog({
  from,
  to,
  fromDays = 2,
  witnesses,
  categories = ["47", "61"],
  start = 0,
  limit = 100,
  skipSubscription = false,
  skipQuery = false,
  withSnapshots = false,
  widgetId
}: EventLogOptions): EventLogResult {
  const [entryError, setEntryError] = useState<string>();
  const [subErrMsg, setSubErrMsg] = useState("");
  const lastResubscribeTime = useRef<Date>();
  const socketErrorTimeout = useRef<NodeJS.Timeout>();
  const lastSocketEventTimeStamp = useRef(0);
  const { store: { socketEvent } } = useStore();
  const [getAuditEntries, { data: evData, loading: evLoading, error: evError }] = useEventsQuery(withSnapshots);
  const { subscribe, unsubscribe, subscription: { data: subData, loading: subLoading, error: subError } } = useAuditEntries();
  const mountedRef = useRef<boolean>(false);

  useEffect(() => {
    mountedRef.current = true;

    return function cleanup() {
      mountedRef.current = false;
      !skipSubscription && unsubscribe(widgetId);
    };
  }, []);

  useEffect(() => {
    !skipSubscription && witnesses.length !== 0 && subscribe(widgetId);
  }, [skipSubscription, witnesses]);

  const executeEventLogQuery = useCallback((): void => {
    if (witnesses.length > 0 && !skipQuery && fromDays > 0) {
      getAuditEntries({
        variables: {
          filter: {
            from: from ?? new Date(Date.now() - fromDays * 24 * 60 * 60 * 1000),
            to: to ?? new Date(),
            witnesses,
            categories,
            start,
            limit
          }
        }
      });
    }
  }, [from, to, fromDays, witnesses, categories, start, limit, skipQuery, getAuditEntries]);


  const handleSubscriptionError = useCallback((error: Error | string): void => {
    const now = new Date();
    if (!lastResubscribeTime.current || now.getTime() - lastResubscribeTime.current.getTime() >= resubscribeInterval) {
      console.error("Subscription error:", error);
      if (typeof error === "string") {
        setSubErrMsg(error);
      }
      lastResubscribeTime.current = now;
      executeEventLogQuery();
      return;
    }
    setTimeout(() => handleSubscriptionError(error), resubscribeInterval);
  }, [executeEventLogQuery]);

  useEffect(useCallback(() => {
    if (!subLoading && !subError && subData?.auditEntries) {
      const entry = subData.auditEntries;
      if (isError(entry)) {
        setEntryError(entry.error);
        return;
      }

      setEntryError(undefined);
      setSubErrMsg("");
    }
  }, [handleSubscriptionError, subLoading, subError, subData]), [subLoading, subError, subData]);

  const handleSocketError = useCallback((event: SocketEvent) => {
    if (mountedRef.current) {
      socketErrorTimeout.current = undefined;
      handleSubscriptionError("Socket " + event.type);
      setSubErrMsg("");
    }
  }, [handleSubscriptionError]);

  useEffect(() => {
    if (socketEvent &&
        socketEvent.type === SocketEventType.Error &&
        socketEvent.timeStamp !== lastSocketEventTimeStamp.current &&
        !socketErrorTimeout.current) {
      lastSocketEventTimeStamp.current = socketEvent.timeStamp;
      const event = socketEvent;
      socketErrorTimeout.current = globalThis.setTimeout(() => handleSocketError(event), socketErrorInterval);
    }
  }, [socketEvent]);

  useEffect(() => {
    const error = evError || subError || entryError;
    error && handleSubscriptionError(error);
  }, [evError, subError, entryError]);

  return {
    entry: subData?.auditEntries,
    entries: evData?.auditEntries,
    queryLoading: evLoading,
    subscriptionLoading: subLoading,
    queryError: evError,
    subscriptionError: subError || (subErrMsg ? new Error(subErrMsg) : undefined),
  };
}

export function useEventsQuery(withSnapshots: boolean) {
  const [getAuditEntries1, { data: evData1, loading: evLoading1, error: evError1 }] = useAuditEntriesLazyQuery({ fetchPolicy: "no-cache" });
  const [getAuditEntries2, { data: evData2, loading: evLoading2, error: evError2 }] = useAuditEntriesWithSnapshotsLazyQuery({ fetchPolicy: "no-cache" });

  const getAuditEntries = withSnapshots ? getAuditEntries2 : getAuditEntries1;
  const data = withSnapshots ? evData2 : evData1;
  const loading = withSnapshots ? evLoading2 : evLoading1;
  const error = withSnapshots ? evError2 : evError1;

  return [getAuditEntries, { data, error, loading }] as const;
}
