import { makeVar, useReactiveVar } from "@apollo/client";
import { useEffect, useMemo, useRef } from "react";

import { AcknowledgeAuditEntryInput, AuditEntriesSubscriptionSubscriptionHookResult, AuditEntry, DeviceFunctionalAspectType, RequireAcknowledgeEntriesQuery, SocketEventType, useAuditEntriesSubscriptionSubscription, useDeviceListByAspectTypesQuery } from "@generated/graphql";
import { useStore } from "@core/store";
import { PropType, queryToInput } from "utils";

import { Entry } from "components/EventList";

export const AuditEntriesAutoUpdateWidgets = makeVar<boolean>(true);
const AuditEntriesSubscription = makeVar<AuditEntriesSubscriptionSubscriptionHookResult>({ loading: false });
const SkipAuditEntriesSubscription = makeVar<boolean>(true);
const AuditEntriesSubscribers = makeVar<Map<string, boolean>>(new Map());

const resubscribeInterval = 5000;

const queryTypes = [
  { type: DeviceFunctionalAspectType.Media },
  { type: DeviceFunctionalAspectType.Sensor },
];

export const useAuditEntriesSubscription = () => {
  const { store: { socketEvent } } = useStore();
  const skip = useReactiveVar(SkipAuditEntriesSubscription);
  const subscribers = useReactiveVar(AuditEntriesSubscribers);
  const { data: devData } = useDeviceListByAspectTypesQuery({
    variables: {
      types: queryTypes
    }
  });

  const subscribersExist = [...subscribers.values()].includes(true);
  const skipSubscription = subscribersExist ? skip : true;
  const witnesses = useMemo<string[]>(() => devData?.devicesByAspectTypes.map(dev => dev.id) ?? [], [devData]);
  const { data, loading, error } = useAuditEntriesSubscriptionSubscription({
    variables: {
      filter: {
        witnesses,
        categories: ["47", "61"]
      }
    },
    shouldResubscribe: true,
    skip: skipSubscription || witnesses.length === 0
  });
  const resubscribeTimeout = useRef<NodeJS.Timeout>();

  useEffect(() => {
    AuditEntriesSubscription({ data, loading, error });
  }, [data, loading, error]);

  useEffect(() => {
    if (!subscribersExist) return;
    const socketError = !!socketEvent &&
      (socketEvent.type === SocketEventType.Reconnected || socketEvent.type === SocketEventType.Error) &&
      Date.now() - new Date(socketEvent.timeStamp).getTime() < resubscribeInterval;

    if ((socketError || error) && !resubscribeTimeout.current) {
      resubscribeTimeout.current = setTimeout(() => resubscribe(), resubscribeInterval);
    }
  }, [socketEvent, error, subscribersExist]);

  function resubscribe() {
    console.info("Resubscribing to audit entries...");
    resubscribeTimeout.current = undefined;
    // Change subscription options to force resubscribe.
    SkipAuditEntriesSubscription(!skip);
    SkipAuditEntriesSubscription(skip);
  }

  return {};
};

type AuditEntriesSubscriptionManagerResult = {
  subscription: AuditEntriesSubscriptionSubscriptionHookResult
  subscribe: (widgetId: string) => void;
  unsubscribe: (widgetId: string) => void;
};

export const useAuditEntries = (): AuditEntriesSubscriptionManagerResult => {
  const skip = useReactiveVar(SkipAuditEntriesSubscription);
  const subscribers = useReactiveVar(AuditEntriesSubscribers);
  const subscription = useReactiveVar(AuditEntriesSubscription);

  const subscribe = (widgetId: string) => {
    const newSubscribers = new Map(subscribers);
    newSubscribers.set(widgetId, true);
    AuditEntriesSubscribers(newSubscribers);
    skip && SkipAuditEntriesSubscription(false);
  };
  const unsubscribe = (widgetId: string) => {
    const newSubscribers = new Map(subscribers);
    newSubscribers.set(widgetId, false);
    AuditEntriesSubscribers(newSubscribers);
  };

  return {
    subscription,
    subscribe,
    unsubscribe
  };
};

type AnyEntry = Entry | AuditEntry | PropType<RequireAcknowledgeEntriesQuery, "requireAcknowledgeEntries">[0];

export function isAlert(entry: AnyEntry): boolean {
  const { metadata } = entry;
  const eventType = metadata?.find(data => data.key === "EVENT-TYPE");
  if (!eventType) return false;
  return eventType.metaClass.value === "1";
}

export function isExternal(entry: AnyEntry): boolean {
  const { metadata } = entry;
  const eventType = metadata?.find(data => data.key === "EXTERNAL");
  if (!eventType) return false;
  return eventType.metaClass.value === "1";
}

export function isAckRequired(entry: AnyEntry): boolean {
  const { metadata } = entry;
  const eventType = metadata?.find(data => data.key === "ACK-REQUIRE");
  if (!eventType) return false;
  return eventType.metaClass.value === "1";
}

export function isConfirmed(entry: AnyEntry): boolean {
  const { metadata } = entry;
  const eventType = metadata?.find(data => data.key === "ACK-CONFIRM");
  if (!eventType) return false;
  return eventType.metaClass.value === "1";
}

export function isAcknowledged(entry: AnyEntry): boolean {
  const { metadata } = entry;
  const eventType = metadata?.find(data => data.key === "ACK");
  if (!eventType) return false;
  return eventType.metaClass.value === "1";
}

export function getAuditEntryInput(entry: AnyEntry): AcknowledgeAuditEntryInput {
  const { context, triggeredAt, category, modifiedAt, witnesses, note } = entry;
  const entryInput: AcknowledgeAuditEntryInput = {
    context,
    triggeredAt,
    modifiedAt,
    note,
    categoryId: category.id,
    witnesses: witnesses.map(w => w.id),
    external: isExternal(entry)
  };
  const queryInput = queryToInput<AcknowledgeAuditEntryInput, AcknowledgeAuditEntryInput>(entryInput);
  return queryInput;
}

export function getStoreEvent<EntryType extends AnyEntry>(entry: EntryType): EntryType {
  const ackMetaIndex = entry.metadata.findIndex(data => data.key === "ACK");
  if (ackMetaIndex >= 0) {
    entry.metadata[ackMetaIndex].metaClass.value = "1";
  }
  return { ...entry, isAcknowledged: true };
}
