import {UUID} from "@solid/types";

export type Event = {
  context: string;
  triggeredAt: number;
  deviceIds: UUID[];
};

type EventBucket = {
  start: number;
  end: number;
};

const tooLargeTime = new Date("2500-01-01").getTime();
const eventLimit = 100;
const eventRequestStep = 4 * 60 * 60 * 1000;

class StepBackForward {
  private deviceIds: UUID[] = [];
  private streamNumbers: number[] = [];
  private archiveStartTime = 0;
  private eventBuckets: EventBucket[] = [];
  private eventSet = new Set<string>();
  private events: Event[] = [];
  private lastTime = 0;
  private lastBoundaryTime = 0;
  private lastBoundaryForward?: boolean;
  private onGetEvents?: (start: number, end: number, limit: number) => Promise<Event[]>;
  private onGetBoundary?: (time: number, forward: boolean, deviceIds: UUID[], streamNumbers: number[]) => Promise<number>;
  private onLoading?: (loading: boolean) => void;

  setDeviceIds(ids: string[]): void {
    this.deviceIds = ids;
  }

  setStreamNumbers(streamNumbers: number[]): void {
    this.streamNumbers = streamNumbers;
  }

  setArchiveStartTime(time: number): void {
    this.archiveStartTime = time;
  }

  setEvents(start: number, end: number, events: Event[]): void {
    end = Math.min(end, Date.now());
    if (!this.eventBuckets.some(bucket => bucket.start <= start && bucket.end >= end)) {
      this.eventBuckets = this.eventBuckets.filter(bucket => !(bucket.start >= start && bucket.end <= end));
      this.eventBuckets.push({ start, end });
      for (const event of events) {
        if (!this.eventSet.has(event.context)) {
          this.eventSet.add(event.context);
          this.events.push(event);
        }
      }
    }
  }

  setOnGetEvents(onGetEvents: (start: number, end: number, limit: number) => Promise<Event[]>): void {
    this.onGetEvents = onGetEvents;
  }

  setOnGetBoundary(onGetBoundary: (time: number, forward: boolean, deviceIds: UUID[], streamNumbers: number[]) => Promise<number>): void {
    this.onGetBoundary = onGetBoundary;
  }

  setOnLoading(onLoading: (loading: boolean) => void): void {
    this.onLoading = onLoading;
  }

  clearCache(): void {
    this.eventBuckets = [];
    this.events = [];
    this.eventSet.clear();
  }

  stepBackward(time: number): Promise<number> {
    return this.step(time, false);
  }

  stepForward(time: number): Promise<number> {
    return this.step(time, true);
  }

  private async step(time: number, forward: boolean): Promise<number> {
    const now = Date.now();
    let newTime = 0;
    let loaded = false;
    do {
      if (forward) {
        newTime = this.events.filter(ev =>
          ev.triggeredAt > time && (this.lastTime === 0 || ev.triggeredAt > this.lastTime) &&
          ev.deviceIds.some(id => this.deviceIds.includes(id)))
          .reduce((accumulator, ev) => Math.min(accumulator, ev.triggeredAt), tooLargeTime);
      }
      else {
        newTime = this.events.filter(ev =>
          ev.triggeredAt < time && (this.lastTime === 0 || ev.triggeredAt < this.lastTime) &&
          ev.deviceIds.some(id => this.deviceIds.includes(id)))
          .reduce((accumulator, ev) => Math.max(accumulator, ev.triggeredAt), 0);
      }

      if (newTime === 0 || newTime === tooLargeTime) {
        loaded = await this.loadEvents(time, forward, now);
      }
    }
    while ((newTime === 0 || newTime === tooLargeTime) && loaded);

    if (newTime > 0 && newTime < tooLargeTime) {
      this.lastTime = newTime;
      return newTime;
    }

    newTime = await this.getBoundary(time, forward);

    if (newTime > 0 && newTime < tooLargeTime) {
      this.lastTime = newTime;
      return newTime;
    }
    return time;
  }

  private async loadEvents(time: number, forward: boolean, currentTime: number): Promise<boolean> {
    let start = 0;
    let end = 0;
    if (forward) {
      const max = this.eventBuckets.reduce((acc, b) => Math.max(acc, b.end), 0);
      if (max > 0 && max >= currentTime) {
        return false; // End of archive reached
      }
      // Find gap in events
      start = this.eventBuckets.filter(b => b.start <= time + 1 && b.end >= time).reduce((acc, b) => Math.max(acc, b.end), 0);
      end = this.eventBuckets.filter(b => b.start > time).reduce((acc, b) => Math.min(acc, b.start), tooLargeTime);
      // There are not buckets for the given time
      if (start === 0) {
        start = time + 1;
      }
      // Gap not found
      if (end === tooLargeTime) {
        end = start + eventRequestStep;
      }
    }
    else {
      const min = this.eventBuckets.reduce((acc, b) => Math.min(acc, b.start), tooLargeTime);
      if (min < tooLargeTime && this.archiveStartTime > 0 && min <= this.archiveStartTime) {
        return false; // Start of archive reached
      }
      // Find gap in events
      start = this.eventBuckets.filter(b => b.end < time).reduce((acc, b) => Math.max(acc, b.end), 0);
      end = this.eventBuckets.filter(b => b.start <= time && b.end >= time - 1).reduce((acc, b) => Math.min(acc, b.start), tooLargeTime);
      // There are not buckets for the given time
      if (end === tooLargeTime) {
        end = time - 1;
      }
      // Gap not found
      if (start === 0) {
        start = end - eventRequestStep;
      }
    }
    const events = await this.getEvents(start, end);
    this.setEvents(start, end, events);
    return true;
  }

  private async getEvents(start: number, end: number): Promise<Event[]> {
    if (!this.onGetEvents) {
      return [];
    }
    let result: Event[] = [];
    this.onLoading && this.onLoading(true);
    try {
      let eventCount = 0;
      let to = end;
      do {
        const events = (await this.onGetEvents(start, to, eventLimit))
          .sort((a, b) => new Date(a.triggeredAt).getTime() - new Date(b.triggeredAt).getTime());
        eventCount = events.length;
        if (events.length > 0) {
          to = events[0].triggeredAt - 1;
        }
        result = result.concat(events);
      }
      while (eventCount === eventLimit);
    }
    catch (e) {
      console.error("Get events error:", e);
    }
    finally {
      this.onLoading && this.onLoading(false);
    }
    return result;
  }

  private async getBoundary(time: number, forward: boolean): Promise<number> {
    if (!this.onGetBoundary) {
      return 0;
    }

    let newTime = 0;
    this.onLoading && this.onLoading(true);
    try {
      const startTime = this.lastBoundaryTime > 0 && this.lastBoundaryForward === forward ?
        (forward ? this.lastBoundaryTime + 1000 : this.lastBoundaryTime - 1000) : time;

      newTime = await this.onGetBoundary(startTime, forward, this.deviceIds, this.streamNumbers);

      this.lastBoundaryForward = forward;
      if (newTime > 0) {
        this.lastBoundaryTime = newTime;
      }
    }
    catch (e) {
      console.error("Get boundary error:", e);
    }
    finally {
      this.onLoading && this.onLoading(false);
    }
    return newTime;
  }
}

export default StepBackForward;
