import React from "react";
import { ApolloClient } from "@apollo/client";
import { WidgetId, WidgetInfo } from "@generated/graphql";
import { ModuleInfo, DragObjectType } from "@core/types";
import { parseJSON } from "utils";
import { WidgetProps, Widgets } from "components/Widgets";
import WidgetStub from "components/Widgets/WidgetStub";
import WidgetPropsEditor from "components/Widgets/WidgetPropsEditor";
import {__} from "@solid/libs/i18n";

export type WidgetOptions = {
  props?: WidgetPropDesc[];
  acceptsConstructorProps?: boolean;
  countLimit?: number | ((widgets: WidgetInfo[]) => number);
  disableCondition?: (widgets: WidgetInfo[]) => boolean;
  hideCondition?: (widgets: WidgetInfo[]) => boolean;
  compatibleWidgets?: WidgetId[];
  getViewValidationError?: (widgets: WidgetInfo[]) => string | undefined;
  headerDragObjectType?: DragObjectType;
  minWidth?: number;
  minHeight?: number;
  getWidgetWarning?: (widget: Widget, props: object, size: WidgetSize) => WidgetWarning | undefined;
};

export type WidgetWarning = {
  code: string;
  text: string;
};

export type WidgetSize = {
  width: number;
  height: number;
};

export type WidgetParamsCallback = (props: Record<string, any>) => void;

export type WidgetPropGroup = {
  id: string;
  name: string;
  key: string;
  content?: JSX.Element;
  acceptsFilter?: boolean;
  scrollable?: boolean;
  extraTitleContent?: JSX.Element;
};

export enum ControlType {
  Checkbox,
  Dropdown,
  Input,
  Button
}

export type WidgetPropValue = boolean | string | number | string[];

export type WidgetPropDesc = {
  name: string;
  label: string;
  controlType: ControlType;
  defaultValue?: WidgetPropValue;
  getDefaultValue?: (widgets: WidgetInfo[], props: object) => WidgetPropValue;
  dropdownItems?: DropdownItem[];
  getDropdownItems?: (client: ApolloClient<any>, widgets: WidgetInfo[], props: object) => Promise<DropdownItem[]>;
  disableCondition?: (widgets: WidgetInfo[], props: object) => boolean;
  hideCondition?: (widgets: WidgetInfo[], props: object) => boolean;
  minWidth?: number;
  buttonModal?: React.ComponentType<ButtonModalProps>;
  buttonLabel?: string;
  multiple?: boolean; // For dropdown controls.
};

export type ButtonModalProps = {
  open: boolean;
  onClose: () => void;
};

export type DropdownItem = {
  value: string | number;
  text: string;
};

export class Widget {
  viewId = "";
  index = 0;
  widgets: WidgetInfo[] = [];
  options: WidgetOptions = {};

  constructor(
    public readonly id: WidgetId,
    public readonly name: string,
    public readonly title: string,
    public readonly module: string,
    public readonly icon: string,
    options?: WidgetOptions) {
    if (options) {
      this.options = options;
    }
  }

  protected createNew(): Widget {
    return new Widget(this.id, this.name, this.title, this.module, this.icon, this.options);
  }

  createInstance(viewId: string, index: number, widgets: WidgetInfo[]): Widget {
    const instance = this.createNew();
    instance.viewId = viewId;
    instance.index = index;
    instance.widgets = widgets;
    return instance;
  }

  getModuleInfo(forStore: boolean = false) : ModuleInfo {
    const { id, name, module, viewId, index } = this;
    let widgetName = forStore ? "" : name;
    if (!forStore && this.widgets.filter(w => w.widgetId === id).length > 1) {
      const i = this.widgets.filter((w, i) => w.widgetId === id && i < index).length + 1;
      widgetName = `${name} ${i}`;
    }
    const m: ModuleInfo = { id: `${viewId}_${id}${index}`, name: widgetName, module, widgetId: this.id };
    const w = index >= 0 && index < this.widgets.length ? this.widgets[index] : undefined;
    if (w && w.propsJSON) {
      m.props = parseJSON(w.propsJSON);
    }
    return m;
  }

  renderStub(): React.ComponentType<WidgetProps> {
    return Stub;
  }

  getPropGroups(): WidgetPropGroup[] {
    return [{ id: "Properties", name: __("Properties"), key: this.id + "_Properties" }];
  }

  renderPropGroupEditor(group: WidgetPropGroup, callback: WidgetParamsCallback, params?: object): JSX.Element | undefined {
    if (group.id === "Properties") {
      return <WidgetPropsEditor key={`${this.viewId}_${this.index}`} widget={this} callback={callback} props={params}/>;
    }
    return undefined;
  }

  getPropDesc(): WidgetPropDesc[] {
    let props: WidgetPropDesc[] = [
      { name: "limitWidth", label: __("Freeze width"), controlType: ControlType.Checkbox },
      { name: "limitHeight", label: __("Freeze height"), controlType: ControlType.Checkbox },
    ];

    if (this.options.props) {
      props = props.concat(this.options.props);
    }
    return props;
  }

  isEnabled(): boolean {
    let countInLimit = true;
    if (this.options.countLimit !== undefined) {
      const count = this.widgets.reduce((accumulator, currentValue) => (currentValue.widgetId === this.id ? accumulator + 1 : accumulator), 0);
      const limit = typeof this.options.countLimit === "number" ? this.options.countLimit : this.options.countLimit(this.widgets);
      countInLimit = limit > 0 && count < limit;
    }

    let enabled = true;
    if (this.options.disableCondition) {
      enabled = this.options.disableCondition(this.widgets);
    }

    let compatible = true;
    if (this.options.compatibleWidgets) {
      compatible = !this.widgets.some(w => w.widgetId && !this.options.compatibleWidgets?.some(id => id === w.widgetId));
    }

    compatible = compatible && !this.widgets.some(w => {
      const widget = Widgets.find(wi => wi.id === w.widgetId);
      return widget && widget.options.compatibleWidgets && !widget.options.compatibleWidgets.some(id => id === this.id);
    });

    return countInLimit && enabled && compatible;
  }

  isHidden(): boolean {
    return !!this.options.hideCondition && this.options.hideCondition(this.widgets);
  }

  isVisible(): boolean {
    return !this.isHidden();
  }
}

const Stub = (props: WidgetProps) => {
  return <WidgetStub {...props}/>;
};
