import React, { useState, useEffect, useRef } from "react";
import { Icon } from "semantic-ui-react";
import { ReactSVG } from "react-svg";
import classNames from "classnames";
import { useDrag, DragObjectWithType } from "react-dnd";
import { ViewId } from "@generated/graphql";
import { ModuleInfo } from "@core/types";
import { useStore } from "@core/store";
import {
  WidgetProps,
  CellProps,
  CellCloseEventPubSub,
  Widgets,
  WidgetWarning,
  CommonWidgetEvent,
  WidgetWarningEventArgs,
  HeaderClickEventPubSub
} from "components/Widgets";
import WidgetStub from "../WidgetStub";
import { useCurrentView } from "components/View";
import { WidgetRenderContent, WidgetRenderContentProps } from "./WidgetRenderContent";
import { Utils } from "@solid/libs/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { useAccessability } from "@core/store/actions/accessability";

import "./style.css";

export type WidgetRenderProps = {
  module?: ModuleInfo;
  component?: React.ComponentType<WidgetProps>;
  componentProps?: object;
} & WidgetProps;

const WidgetRender = ({ module: mod, component, componentProps, ...widgetProps }: WidgetRenderProps) => {
  const { store: { session: { info } } } = useStore();
  const { config: { limitedAuditAccess } } = useAccessability();
  const { module, name, widgetId, props: moduleProps } = mod ?? { module: "" };
  const { viewId, index, nextActiveViewId, alwaysUpdateWidgetContent, tabPanelRef, loader, widgetEvent } = widgetProps;
  const viewLoaderRef = useRef(loader ?? null);
  const { currentViewId } = useCurrentView({ tabPanelRef, viewLoaderRef });
  const nextActiveViewIdRef = useRef(nextActiveViewId);
  const [isViewActive, setIsViewActive] = useState(getIsViewActive());
  const widget = Widgets.find(w => w.id === widgetId);
  const defaultTitle = widget?.name || name || `${__("Widget")} ${index ?? 1}`;
  const [title, setTitle] = useState<React.ReactNode>(defaultTitle);
  const [closable, setClosable] = useState(false);
  const [hideHeader, setHideHeader] = useState(false);
  const [hideIcon, setHideIcon] = useState(false);
  const [icon, setIcon] = useState("");
  const [dragObject, setDragObject] = useState<DragObjectWithType>({ type: "" });
  const [isAllow, setIsAllow] = useState<boolean>(true);
  const windowCloseEventRef = useRef(new CellCloseEventPubSub());
  const headerClickEventRef = useRef(new HeaderClickEventPubSub());
  const resizeObserverRef = useRef<ResizeObserver>();
  const rootRef = useRef<HTMLDivElement>(null);
  const widgetWarningRef = useRef<WidgetWarning | undefined>();
  const widgetWarningTimeoutRef = useRef<NodeJS.Timeout | undefined>();

  const [{ dragging }, dragRef] = useDrag<DragObjectWithType, {}, { dragging: boolean }>({
    item: { type: widget?.options.headerDragObjectType ?? "" },
    collect: monitor => ({
      dragging: monitor.isDragging(),
    }),
    begin: () => dragObject,
    canDrag: () => !!dragObject.type
  });

  useEffect(() => {
    const root = rootRef.current;
    const addObserver = !!viewId && !viewId.endsWith("_preview") && !!moduleProps && !!widget && !!widget.options.getWidgetWarning;
    if (root && addObserver) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === root) {
            if (widgetWarningTimeoutRef.current) {
              clearTimeout(widgetWarningTimeoutRef.current);
            }
            const { width, height } = entry.contentRect;
            widgetWarningTimeoutRef.current = setTimeout(() => publishWidgetWarning(width, height), 1000);
            break;
          }
        }
      }));
      resizeObserverRef.current.observe(root);
    }

    return function cleanup() {
      root && addObserver && resizeObserverRef.current?.unobserve(root);
    };
  }, []);

  useEffect(() => {
    if (!info) {
      return;
    }

    if (limitedAuditAccess && currentViewId === ViewId.UserAudit) {
      setIsAllow(true);
      return;
    }

    let isAllow: boolean | undefined = undefined;
    const allowedWidgets = info.user.solidConfiguration.widgets.map(widgetConf => widgetConf.widgetId.toLowerCase());
    if (allowedWidgets && allowedWidgets?.length > 0) {
      if (widgetId) {
        isAllow = allowedWidgets.includes(widgetId.toLowerCase());
      }
      if (widgetProps.widgetId) {
        isAllow = allowedWidgets.includes(widgetProps.widgetId.toLowerCase());
      }
    }

    if (typeof isAllow !== "undefined") {
      setIsAllow(isAllow);
    }
  }, [info, widgetProps.widgetId, widgetId, currentViewId, limitedAuditAccess]);

  useEffect(() => {
    const isActive = getIsViewActive();
    if (isActive !== isViewActive) {
      setIsViewActive(isActive);
    }
    if (nextActiveViewIdRef.current && currentViewId === nextActiveViewIdRef.current) {
      nextActiveViewIdRef.current = undefined;
    }
  }, [currentViewId]);

  function getIsViewActive(): boolean {
    if (viewId && currentViewId) {
      if (nextActiveViewIdRef.current) {
        return viewId === nextActiveViewIdRef.current;
      }

      return currentViewId === viewId;
    }

    return true;
  }

  function setCellProps(props: CellProps): void {
    if (props.title !== undefined) {
      setTitle(prevTitle => JSON.stringify(prevTitle) === JSON.stringify(props.title) ? prevTitle : props.title);
    }
    if (props.closable !== undefined) {
      setClosable(props.closable);
    }
    if (props.hideHeader !== undefined) {
      setHideHeader(props.hideHeader);
    }
    if (props.hideIcon !== undefined) {
      setHideIcon(props.hideIcon);
    }
    if (props.icon !== undefined) {
      setIcon(props.icon);
    }
    if (props.dragObject !== undefined) {
      setDragObject(props.dragObject);
    }
  }

  function onWindowClose(e: React.MouseEvent): void {
    e.stopPropagation();
    windowCloseEventRef.current.publish({});
  }

  function onHeaderClick(): void {
    headerClickEventRef.current.publish({});
  }

  function publishWidgetWarning(width: number, height: number): void {
    const getWarning = widget?.options.getWidgetWarning;
    const warning = widget && getWarning && getWarning(widget, moduleProps ?? {}, { width, height });
    if (JSON.stringify(widgetWarningRef.current) !== JSON.stringify(warning)) {
      widgetWarningRef.current = warning;
      warning && Log.warning(warning.text);
      const args: WidgetWarningEventArgs = { widgetIndex: index ?? -1, warning };
      widgetEvent?.publish({ event: CommonWidgetEvent.WidgetWarning, args });
    }
  }

  const contentProps: WidgetRenderContentProps = {
    renderModule: !!module,
    module,
    moduleProps,
    component,
    componentProps,
    widgetProps: {
      ...widgetProps,
      isViewActive,
      widgetId: widgetProps.widgetId ?? widgetId,
      cellProps: widgetProps.cellProps ?? { title: defaultTitle, closable: false, hideHeader: false, hideIcon: false, icon: "", dragObject: { type: "" } },
      setCellProps: widgetProps.setCellProps ?? setCellProps,
      cellCloseEventPubSub: widgetProps.cellCloseEventPubSub ?? windowCloseEventRef.current,
      headerClickEventPubSub: widgetProps.headerClickEventPubSub ?? headerClickEventRef.current
    },
    alwaysUpdate: alwaysUpdateWidgetContent,
  };

  const titleIcon = icon || widget?.icon;

  return (
    <div ref={rootRef} className="WidgetRender-Root">
      {!!module && !hideHeader &&
      <div ref={dragRef} className={classNames("WidgetRender-Header", { "WidgetRender-Header_dragging": dragging })} onClick={onHeaderClick}>
        { !!titleIcon && !hideIcon && <ReactSVG role="small-icon" src={titleIcon}/> }
        <div className="WidgetRender-Title">{title}</div>
        { closable && <Icon name="close" onClick={onWindowClose}/> }
      </div>}
      {isAllow ?
        <WidgetRenderContent {...contentProps}/>
        :
        <WidgetStub widgetId={widgetId || widgetProps.widgetId} isAllow={isAllow}/>}
    </div>
  );
};

export default WidgetRender;
