import React, { useState, useEffect, useRef, useMemo } from "react";
import { useApolloClient } from "@apollo/client";
import { Segment, Checkbox, Dropdown, DropdownItemProps, Button } from "semantic-ui-react";
import classNames from "classnames";
import { Widget, WidgetPropDesc, WidgetParamsCallback, WidgetPropValue, ControlType, Widgets, DropdownItem, HierarchyEditPropName } from "components/Widgets";
import { WidgetId, WidgetInfo } from "@generated/graphql";
import Loading from "components/Loading";
import { isArray, EventPubSub } from "utils";
import { useStore } from "@core/store";

import "./style.css";

type WidgetPropsEditorProps = {
  widget: Widget;
  callback: WidgetParamsCallback;
  props?: Record<string, any>;
};

class ModalCloseEvent extends EventPubSub<{}> {}

const WidgetPropsEditor = ({ widget, callback, props = {} }: WidgetPropsEditorProps) => {
  const { store: { session: { info } } } = useStore();
  const widgetElemId = `${widget.viewId.replace("_selectWidgets", "")}_widget${widget.index}`;
  const [disabled, setDisabled] = useState<Record<string, boolean>>(getDisabled());
  const [hidden, setHidden] = useState<Record<string, boolean>>(getHidden());
  const [loading, setLoading] = useState(false);
  const loadingControlsRef = useRef(new Set<string>());
  const loadingTimeoutRef = useRef<NodeJS.Timeout | undefined>();
  const modalCloseEventRef = useRef(new ModalCloseEvent());

  const widgetEditProps: WidgetPropDesc[] = useMemo(() => {
    let props: WidgetPropDesc[] = widget.getPropDesc().filter(item => !hidden[item.name]);

    if (info) {
      const allowedWidgets = info.user.solidConfiguration.widgets.map(widgetConf => widgetConf.widgetId);

      if (allowedWidgets && allowedWidgets.length === 0) {
        return props;
      }
      if (!allowedWidgets.includes(widget.id)) {
        return [];
      }
      if (!allowedWidgets.includes(WidgetId.HierarchyEdit)) {
        props = props.filter(prop => !Object.keys(HierarchyEditPropName).includes(prop.name));
      }
    }

    return props;
  }, [widget, info]);

  useEffect(() => {
    setDisabled(getDisabled());
    setHidden(getHidden());
  }, [widget.widgets]);

  function onChange(prop: WidgetPropDesc, value: WidgetPropValue): void {
    if (prop.name === "limitWidth" || prop.name === "limitHeight") {
      const widgetElem = document.getElementById(widgetElemId);
      if (widgetElem) {
        const rect = widgetElem.getBoundingClientRect();
        if (prop.name === "limitWidth") {
          props.fixedWidth = value ? rect.width : undefined;
        }
        else if (prop.name === "limitHeight") {
          props.fixedHeight = value ? rect.height : undefined;
        }
      }
    }

    props[prop.name] = value;

    setDisabled(getDisabled());
    setHidden(getHidden());

    callback(props);
  }

  function getDisabled(): Record<string, boolean> {
    const dis: Record<string, boolean> = {};
    for (const p of widget.getPropDesc().filter(pd => !!pd.disableCondition)) {
      dis[p.name] = !!p.disableCondition && p.disableCondition(widget.widgets, props);
    }
    return dis;
  }

  function getHidden(): Record<string, boolean> {
    const hdn: Record<string, boolean> = {};
    for (const p of widget.getPropDesc().filter(pd => !!pd.hideCondition)) {
      hdn[p.name] = !!p.hideCondition && p.hideCondition(widget.widgets, props);
    }
    return hdn;
  }

  function onLoading(name: string, loading: boolean): void {
    const loadingSet = loadingControlsRef.current;
    if (loading) {
      loadingSet.add(name);
    }
    else {
      loadingSet.delete(name);
    }
    if (loadingTimeoutRef.current) {
      globalThis.clearTimeout(loadingTimeoutRef.current);
      loadingTimeoutRef.current = globalThis.setTimeout(() => setLoading(loadingSet.size > 0), 250);
    }
  }

  return (
    <Segment className="WidgetPropsEditor">
      {widgetEditProps?.length > 0 &&
      <>
        {loading && <Loading/>}

        <div>{Widgets.find(w => w.id === widget.id)?.title}</div><hr/>

        {widgetEditProps.map(item =>
          <PropControl
            key={item.name}
            propDesc={item}
            value={props[item.name]}
            disabled={disabled[item.name]}
            widgets={widget.widgets}
            props={props}
            modalCloseEvent={modalCloseEventRef.current}
            onChange={onChange}
            onLoading={loading => onLoading(item.name, loading)}
        />)}
      </>}
    </Segment>
  );
};

type PropControlProps = {
  propDesc: WidgetPropDesc;
  value?: WidgetPropValue;
  disabled?: boolean;
  widgets: WidgetInfo[];
  props: object;
  modalCloseEvent: ModalCloseEvent;
  onChange: (propDesc: WidgetPropDesc, value: WidgetPropValue) => void;
  onLoading: (loading: boolean) => void;
};

const PropControl = ({ propDesc, value: propValue, disabled, widgets, props, modalCloseEvent, onChange, onLoading }: PropControlProps) => {
  const { name, label, controlType, minWidth, buttonLabel, buttonModal, multiple } = propDesc;
  const client = useApolloClient();
  const [value, setValue] = useState(getPropDefaultValue());
  const [dropdownItems, setDropdownItems] = useState<DropdownItem[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    modalCloseEvent.subscribe(id, loadDropdownItems);

    return () => {
      modalCloseEvent.unsubscribe(id);
    };
  });

  useEffect(() => {
    if (propDesc.getDropdownItems) {
      loadDropdownItems();
      return;
    }
    setDropdownItems(propDesc.dropdownItems ?? []);
  }, [propDesc.dropdownItems, propDesc.getDropdownItems, widgets, props]);

  useEffect(() => {
    if (propValue !== undefined && !disabled) {
      setValue(propValue);
    }
    if (disabled) {
      setValue(getPropDefaultValue());
    }
    if (!disabled && propValue === undefined) {
      setValue(getPropDefaultValue());
    }
  }, [propValue, disabled]);

  function getPropDefaultValue(): WidgetPropValue | undefined {
    return (propDesc.getDefaultValue ? propDesc.getDefaultValue(widgets, props) : propDesc.defaultValue) ?? (multiple ? [] : undefined);
  }

  function getDefaultValue<T extends WidgetPropValue>(type: "boolean" | "string" | "number"): T {
    switch (type) {
      case "boolean": return false as T;
      case "string": return "" as T;
      case "number": return 0 as T;
      default: return false as T;
    }
  }

  function getValue<T extends WidgetPropValue>(value?: WidgetPropValue): T {
    switch (controlType) {
      case ControlType.Checkbox: return typeof value === "boolean" ? value as T : getDefaultValue<T>("boolean");

      case ControlType.Dropdown: return getDropdownValue(value) as T;

      case ControlType.Input: return typeof value === "string" ? value as T : getDefaultValue<T>("string");

      default: return false as T;
    }
  }

  function getDropdownValue(value?: WidgetPropValue): string | number | string[] | undefined {
    if (Array.isArray(value)) {
      if (multiple) {
        return value;
      }
      return value.length > 0 ? value[0] : (dropdownItems.length > 0 ? dropdownItems[0].value : undefined);
    }
    if (multiple) {
      return [];
    }
    if ((typeof value === "string" || typeof value === "number") && dropdownItems.some(it => it.value === value)) {
      return value;
    }
    return dropdownItems.length > 0 ? dropdownItems[0].value : undefined;
  }

  function onChanged(e: React.SyntheticEvent | undefined, value: WidgetPropValue | WidgetPropValue[] | undefined): void {
    let val: WidgetPropValue | undefined;
    if (Array.isArray(value)) {
      if (value.length > 0) {
        if (controlType === ControlType.Dropdown && multiple && isArray<string>(value, "string")) {
          val = value;
        }
        else {
          val = value[0];
        }
      }
      else if (multiple) {
        val = [];
      }
    }
    else {
      val = value;
    }
    if (val !== undefined) {
      setValue(val);
      onChange(propDesc, val);
    }
  }

  function getDropdownItem({ value, text }: DropdownItem): DropdownItemProps {
    return { key: value, value, text };
  }

  function loadDropdownItems(): void {
    if (propDesc.getDropdownItems) {
      onLoading(true);
      propDesc.getDropdownItems(client, widgets, props)
        .then(items => {
          setDropdownItems(items);
          if (Array.isArray(propValue)) {
            const newValue = propValue.filter(val => items.some(item => item.value === val));
            if (JSON.stringify(propValue) !== JSON.stringify(newValue)) {
              onChanged(undefined, newValue);
            }
          }
          else if (propValue !== undefined && !items.some(item => item.value === propValue)) {
            onChanged(undefined, getDefaultValue("string"));
          }
        })
        .catch(e => { console.error("Get dropdown items error:", e); })
        .finally(() => { onLoading(false); });
    }
  }

  switch (controlType) {
    case ControlType.Checkbox:
      return (
        <div>
          <Checkbox
            className={`WidgetPropsEditor-Checkbox WidgetPropsEditor__${name}`}
            label={label}
            checked={getValue<boolean>(value)}
            disabled={disabled}
            onChange={(e, data) => onChanged(e, data.checked ?? false)}
            toggle
          />
        </div>
      );

    case ControlType.Dropdown:
      return (
        <div className="WidgetPropsEditor-Field">
          <span className={classNames({ "WidgetPropsEditor-Field_disabled": disabled })}>{label}</span>
          <Dropdown
            selection
            multiple={multiple}
            options={dropdownItems.map(item => getDropdownItem(item))}
            value={getDropdownValue(value)}
            disabled={disabled}
            style={minWidth ? { minWidth: `${minWidth}px` } : undefined}
            onChange={(e, { value }) => onChanged(e, value)}
          />
        </div>
      );

    case ControlType.Button:
      const Modal = buttonModal;
      return (
        <div className="WidgetPropsEditor-Field">
          <span className={classNames({ "WidgetPropsEditor-Field_disabled": disabled })}>{label}</span>
          <Button
            className="WidgetPropsEditor-Button"
            disabled={disabled}
            onClick={() => setIsModalOpen(true)}>
            {buttonLabel ?? label}
          </Button>

          {Modal &&
          <Modal {...props} open={isModalOpen} onClose={() => {
            setIsModalOpen(false);
            modalCloseEvent.publish({});
          }}/>}
        </div>
      );

    default:
      return null;
  }
};

export default WidgetPropsEditor;
