import React, { useState, useEffect, useRef, useCallback } from "react";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { Button, Icon, Popup } from "semantic-ui-react";
import { CreatingView, CreatingViewInput, WidgetInfo, WidgetInfoInput } from "@generated/graphql";
import {
  WidgetDragObjectType,
  WidgetDragObject,
  Widget,
  Widgets,
  WidgetProps,
  ViewConstructorProps,
  CellProps,
  WidgetWarning,
  CommonWidgetEvent,
  WidgetWarningEventArgs
} from "components/Widgets";
import { withStore, WithStoreProps } from "@core/store";
import { queryToInput, parseJSON } from "utils";
import WidgetRenderWrapper from "components/Widgets/WidgetRenderWrapper";
import { ViewLayoutItemOrientation, ViewLayoutItem } from "@core/types";
import { ViewLayout } from "components/ViewLayouts";
import { AddRowColArgs } from "components/Layout";
import { PanelSide } from "components/View/AddRowColPanel";
import { Utils, clone } from "@solid/libs/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

type WidgetPlaceholderProps = WithStoreProps & {
  creatingViewId: string;
  creatingWidgetIndex: number;
} & WidgetProps;

const WidgetPlaceholder = ({
  creatingViewId: viewId,
  creatingWidgetIndex: index,
  store: { workspace: { creatingViews } },
  setStore,
  ...widgetProps
}: WidgetPlaceholderProps) => {
  const { loader, setCellProps, cellCloseEventPubSub, headerClickEventPubSub, addRowColEvent, paddingClickEvent, aspectRatio, widgetEvent } = widgetProps;

  const getWidgets = useCallback((): WidgetInfo[] => {
    const view = creatingViews ? creatingViews.find(v => v.viewId === viewId) : undefined;
    return view && view.widgets ? clone(view.widgets) : [];
  }, [creatingViews, viewId]);

  const getWidget = useCallback((): Widget | undefined => {
    const widgets = getWidgets();
    return index < widgets.length ? Widgets.find(w => w.id === widgets[index].widgetId) : undefined;
  }, [creatingViews, viewId, index]);

  const getComponentProps = useCallback((): Record<string, any> | undefined => {
    const widgets = getWidgets();
    if (index < widgets.length) {
      const propsJSON = widgets[index].propsJSON;
      if (propsJSON) {
        return JSON.parse(propsJSON);
      }
    }
    return undefined;
  }, [creatingViews, viewId, index]);

  const [widget, setWidget] = useState(getWidget());
  const [componentProps, setComponentProps] = useState(getComponentProps());
  const [selected, setSelected] = useState(false);
  const [constructorProps, setConstructorProps] = useState<ViewConstructorProps | undefined>();
  const [title, setTitle] = useState<React.ReactNode | undefined>();
  const [popupContent, setPopupContent] = useState("");
  const widgetWarningRef = useRef<WidgetWarning | undefined>();
  const isMountedRef = useRef<boolean>(false);
  const [hideIcon, setHideIcon] = useState<boolean | undefined>(undefined);

  useEffect(() => {
    return () => {
      setWidget(undefined);
      setComponentProps(undefined);
      setSelected(false);
    };
  }, []);

  const [{ dragOver }, dropRef] = useDrop<WidgetDragObject, {}, { dragOver: boolean }>({
    accept: [WidgetDragObjectType],

    drop: useCallback((item, monitor) => {
      setTitle("");
      setPopupContent(getPopupText(true));
      const w = updateWidget(item.id);
      if (w) {
        setWidgetToStore(w);
      }
      return undefined;
    }, [creatingViews, viewId, index]),

    canDrop: useCallback((item, monitor) => {
      const widgets = getWidgets();
      return canDrop(item, viewId, index, widgets);
    }, [getWidgets]),

    collect: monitor => ({
      dragOver: monitor.isOver()
    })
  });

  const resizeObserverRef = useRef<ResizeObserver>();

  const rootId = `${viewId}_widget${index}`;
  const splitHorzBtnId = `${viewId}_widget${index}_splitHorzBtn`;
  const splitVertBtnId = `${viewId}_widget${index}_splitVertBtn`;
  const hiddenClassName = "WidgetPlaceholder-Button_hidden";

  useEffect(useCallback(() => {
    isMountedRef.current = true;

    window.requestAnimationFrame(() => {
      if (!isMountedRef.current) {
        return;
      }

      setPopupContent(getPopupText());
    });

    return () => {
      isMountedRef.current = false;
    };
  }, [creatingViews, viewId, index]), []);

  const showHideSplitCellButtons = () => {
    const root = document.getElementById(rootId);
    if (!root) {
      return;
    }

    const style = window.getComputedStyle(root, null);
    const minWidth = parseInt(style.minWidth) || 0;
    const minHeight = parseInt(style.minHeight) || 0;
    const rect = root.getBoundingClientRect();

    if (minWidth !== 0) {
      const horizontalSplitterButton = document.getElementById(splitHorzBtnId);
      if (rect.width <= minWidth) {
        root.classList.add("fixed-min-width");
      } else {
        root.classList.remove("fixed-min-width");
      }
      if (rect.width <= minWidth * 1.5) {
        horizontalSplitterButton?.classList.add(hiddenClassName);
      } else {
        horizontalSplitterButton?.classList.remove(hiddenClassName);
      }
    }

    if (minHeight !== 0) {
      const verticalSplitterButton = document.getElementById(splitVertBtnId);
      if (rect.height <= minHeight) {
        root.classList.add("fixed-min-height");
      } else {
        root.classList.remove("fixed-min-height");
      }
      if (rect.height <= minHeight * 1.5) {
        verticalSplitterButton?.classList.add(hiddenClassName);
      } else {
        verticalSplitterButton?.classList.remove(hiddenClassName);
      }
    }
  };

  useEffect(() => {
    window.requestAnimationFrame(() => showHideSplitCellButtons());
  }, []);

  const resizeObserverCallback = useCallback(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
    if (!isMountedRef.current) {
      return;
    }

    const root = document.getElementById(rootId);
    for (const entry of entries) {
      if (entry.target === root) {
        showHideSplitCellButtons();

        const props = getComponentProps();
        if (props) {
          let update = false;
          if (props.limitWidth && props.fixedWidth !== entry.contentRect.width) {
            props.fixedWidth = entry.contentRect.width;
            update = true;
          }
          if (props.limitHeight && props.fixedHeight !== entry.contentRect.height) {
            props.fixedHeight = entry.contentRect.height;
            update = true;
          }
          if (update) {
            const w = getWidget();
            if (w) {
              setWidgetToStore(w, false, props);
            }
          }
        }

        updateWidgetWarning(getWidget(), props, entry.contentRect.width, entry.contentRect.height);
        break;
      }
    }
  }), [creatingViews, viewId, index]);

  useEffect(useCallback(() => {
    const id = `${Date.now()}.${Math.random()}`;
    cellCloseEventPubSub?.subscribe(id, onCellClose);
    headerClickEventPubSub?.subscribe(id, select);
    if (index === 0) {
      addRowColEvent?.subscribe(id, addRowCol);
      paddingClickEvent?.subscribe(id, ({ id }) => id === viewId + "_selectWidgets" && unselectWidget());
    }

    const root = document.getElementById(rootId);
    if (root) {
      resizeObserverRef.current = new ResizeObserver(resizeObserverCallback);
      resizeObserverRef.current.observe(root);
    }

    return function cleanup() {
      cellCloseEventPubSub?.unsubscribe(id);
      headerClickEventPubSub?.unsubscribe(id);
      addRowColEvent?.unsubscribe(id);
      paddingClickEvent?.unsubscribe(id);

      if (root) {
        resizeObserverRef.current?.unobserve(root);
      }
    };
  }, [resizeObserverCallback]));

  const updateWidget = useCallback((widgetId?: string, remove?: boolean): Widget | undefined => {
    let id = widgetId || "";
    const widgets = getWidgets();
    if (!id && index < widgets.length) {
      id = widgets[index].widgetId;
    }
    let w = !remove ? Widgets.find(w => w.id === id) : undefined;
    setWidget(w);
    const props = !remove ? getComponentProps() : undefined;
    setComponentProps(props);
    if (w) {
      w = w.createInstance(viewId, index, widgets);
    }

    setTitle(prevTitle => {
      const newTitle = w && !prevTitle ? w.name : prevTitle;
      // Warning: bad setState()
      // if (setCellProps) {
      //   setCellProps({ title: newTitle, closable: !!w, hideHeader: !w, icon: w?.icon });
      // }
      return JSON.stringify(prevTitle) === JSON.stringify(newTitle) ? prevTitle : newTitle;
    });

    updateWidgetWarning(w, props);

    return w;
  }, [creatingViews, viewId, index]);

  useEffect(() => {
    if (!title || !setCellProps) {
      return;
    }

    const widgets = getWidgets();
    const id = widgets[index].widgetId;
    let w = Widgets.find(w => w.id === id);
    if (w) {
      w = w.createInstance(viewId, index, widgets);
    }

    setCellProps({ title, closable: !!w, hideHeader: !w, icon: w?.icon });
  }, [title, index, updateWidget]);

  useEffect(useCallback(() => {
    updateWidget();
    setSelected(!!creatingViews && creatingViews.some(
      v => v.viewId === viewId
      && v.selectedWidget
      && v.selectedWidget.index === index
    ));

    window.requestAnimationFrame(() => {
      if (!isMountedRef.current) {
        return;
      }

      setPopupContent(getPopupText());
    });
  }, [updateWidget]), [creatingViews, viewId, index]);

  useEffect(() => {
    setConstructorProps(widget && widget.options.acceptsConstructorProps ?
      { inConstructor: true, selectedInConstructor: selected } : undefined);
  }, [widget, selected]);

  useEffect(() => {
    setSelectedBorder(index, selected);
  }, [selected]);

  useEffect(() => {
    loader?.setMinMaxSize(rootId, widget?.options.minWidth, undefined, widget?.options.minHeight, undefined);
  }, [widget?.options.minWidth, widget?.options.minHeight]);

  const setWidgetToStore = useCallback((w: Widget | undefined, select?: boolean, props?: object): void => {
    const widgets = getWidgets();
    if (creatingViews && index < widgets.length) {
      const i = creatingViews.findIndex(v => v.viewId === viewId);
      if (i >= 0) {
        const newViews = queryToInput<CreatingView[], CreatingViewInput[]>(creatingViews);
        const newWidgets = Array.from(widgets);
        const widget: WidgetInfoInput = {
          widgetId: w ? w.id : "",
          propsJSON: props ? JSON.stringify(props) : null,
          index
        };
        newWidgets[index] = widget;
        const current = newViews[i].selectedWidget;
        let selectedWidget = w ? (select ? widget : current) : (current?.index === index ? null : current);

        // Remove widgets which become hidden due to condition
        for (let j = 0; j < newWidgets.length; j++) {
          if (j === index) {
            continue;
          }
          const wi = newWidgets[j];
          let widget = Widgets.find(w => w.id === wi.widgetId);
          if (!widget) {
            continue;
          }
          widget = widget.createInstance(newViews[i].viewId, j, newWidgets);
          if (widget.isHidden()) {
            newWidgets[j] = { widgetId: "", propsJSON: null, index: j };
            if (selectedWidget?.index === j) {
              selectedWidget = null;
            }
          }
        }

        newViews[i] = { ...newViews[i], widgets: newWidgets, selectedWidget };
        setStore({ workspace: { creatingViews: newViews } });
      }
    }
  }, [creatingViews, viewId, index]);

  const select = useCallback((): void => {
    if (!widget) {
      return;
    }
    setSelected(true);
    if (!creatingViews) {
      return;
    }
    const newViews = queryToInput<CreatingView[], CreatingViewInput[]>(creatingViews);
    const viewIndex = newViews.findIndex(v => v.viewId === viewId);
    if (viewIndex < 0) {
      return;
    }
    const view = newViews[viewIndex];
    if (!view.widgets || index < 0 || index >= view.widgets.length) {
      return;
    }
    newViews[viewIndex] = { ...view, selectedWidget: { ...view.widgets[index], index }};
    setStore({ workspace: { creatingViews: newViews } });
  }, [creatingViews, viewId, index]);

  const setSelectedBorder = useCallback((index: number, selected: boolean): void => {
    const message = `Could not set selected state for widget #${index} of view ${viewId}.`;
    if (!loader) {
      console.error(message, "Loader reference is undefined.");
      return;
    }
    let set = loader.setSelected(`${viewId}_widget${index}`, selected);
    if (!set) {
      console.warn(message, `Widget element with index ${index} for view ${viewId} is not found.`);
    }
    if (!set && index === 0) {
      set = loader.setSelected(`${viewId}_selectWidgets`, selected);
    }
    if (!set) {
      console.error(message, `Widget element with index ${index} for view ${viewId} is not found.`);
    }
    if (selected) {
      const view = creatingViews?.find(v => v.viewId === viewId);
      if (view && view.widgets && view.widgets.length > 1) {
        for (let i = 0; i < view.widgets.length; i++) {
          if (i !== index) {
            setSelectedBorder(i, false);
          }
        }
      }
    }
  }, [creatingViews, viewId, index]);

  const onCellClose = useCallback((): void => {
    setSelected(false);
    updateWidget(undefined, true);
    setWidgetToStore(undefined);
  }, [updateWidget, setWidgetToStore]);

  const widgetSetCellProps = useCallback(({ title, hideIcon }: CellProps): void => {
    if (!title) {
      return;
    }

    setHideIcon(hideIcon);
    setTitle(prevTitle => {
      if (JSON.stringify(prevTitle) === JSON.stringify(title)) {
        return prevTitle;
      }
      // Warning: bad setState()
      // if (setCellProps) {
      //   setCellProps({ title, hideIcon });
      // }
      return title;
    });
  }, [setWidgetToStore]);

  useEffect(() => {
    if (!title || !setCellProps) {
      return;
    }
    setCellProps({ title, hideIcon });
  }, [title, hideIcon, widgetSetCellProps]);

  const storeLayout = useCallback((callback: (layout: ViewLayout, view: CreatingViewInput, widgets: WidgetInfo[]) => boolean): void => {
    if (!creatingViews) {
      return;
    }

    const newViews = queryToInput<CreatingView[], CreatingViewInput[]>(creatingViews);
    const viewIndex = newViews.findIndex(v => v.viewId === viewId);
    if (viewIndex < 0) {
      return;
    }

    const view = newViews[viewIndex];
    const widgets = getWidgets();
    const newWidgets = Array.from(widgets);
    const layout = new ViewLayout(parseJSON<ViewLayoutItem[]>(view.layoutJSON));
    if (callback(layout, view, newWidgets)) {
      view.layoutJSON = JSON.stringify(layout.layout);
      newViews[viewIndex] = { ...view, widgets: newWidgets };
      setStore({ workspace: { creatingViews: newViews } });
    }
  }, [creatingViews, viewId, index]);

  const splitCell = useCallback((orientation: ViewLayoutItemOrientation): void => {
    checkAspectRatio(orientation === ViewLayoutItemOrientation.Horizontal ? "left" : "top");
    storeLayout((layout, view, widgets) => layout.splitCell(index, orientation, view, widgets));
  }, [storeLayout]);

  const removeCell = useCallback((): void => {
    storeLayout((layout, view, widgets) => layout.removeCell(index, view, widgets));
  }, [storeLayout]);

  const checkAspectRatio = useCallback((side: PanelSide): void => {
    if (!aspectRatio) {
      return;
    }

    const widget = document.getElementById(rootId);
    const view = document.getElementById(`${viewId}_selectWidgets_0`) ?? widget;
    if (!view || !widget) {
      return;
    }

    const rect = view.getBoundingClientRect();

    const vStyle = window.getComputedStyle(view, null);
    const minViewWidth = parseInt(vStyle.minWidth) || 0;
    const minViewHeight = parseInt(vStyle.minHeight) || 0;

    const wStyle = window.getComputedStyle(view, null);
    const minWidgetWidth = parseInt(wStyle.minWidth) || 0;
    const minWidgetHeight = parseInt(wStyle.minHeight) || 0;

    if (side === "top" || side === "bottom") {
      if (rect.width && minViewHeight && rect.width / (minViewHeight + minWidgetHeight) < aspectRatio - 0.05) {
        Log.warning(__("Too many rows. View aspect ratio cannot be maintained."));
      }
    }
    else if (rect.height && minViewWidth && (minViewWidth + minWidgetWidth) / rect.height > aspectRatio + 0.05) {
      Log.warning(__("Too many columns. View aspect ratio cannot be maintained."));
    }
  }, [aspectRatio]);

  const addRowCol = useCallback(({ side }: AddRowColArgs): void => {
    checkAspectRatio(side);
    switch (side) {
      case "top": storeLayout((layout, view, widgets) => layout.addRowColBefore(ViewLayoutItemOrientation.Vertical, view, widgets)); break;
      case "bottom": storeLayout((layout, view, widgets) => layout.addRowColAfter(ViewLayoutItemOrientation.Vertical, view, widgets)); break;
      case "left": storeLayout((layout, view, widgets) => layout.addRowColBefore(ViewLayoutItemOrientation.Horizontal, view, widgets)); break;
      case "right": storeLayout((layout, view, widgets) => layout.addRowColAfter(ViewLayoutItemOrientation.Horizontal, view, widgets)); break;
    }
  }, [storeLayout, checkAspectRatio]);

  const dragWidgetsText = __("Drag widgets from the left panel to the widget placeholders");
  const selectWidgetText = __("Click on a widget to select and configure widget");

  const getPopupText = useCallback((dropped?: boolean): string => {
    const widgets = getWidgets();
    if (dropped && !creatingViews?.find(v => v.viewId === viewId)?.selectedWidget && !widgets.filter(w => !!w.widgetId).length) {
      return selectWidgetText;
    }

    if (index === 0 && !widgets.filter(w => !!w.widgetId).length) {
      return dragWidgetsText;
    }

    if (!creatingViews?.find(v => v.viewId === viewId)?.selectedWidget && widgets.findIndex(w => !!w.widgetId) === index) {
      return selectWidgetText;
    }

    return "";
  }, [creatingViews, viewId, index]);

  const unselectWidget = useCallback(() => {
    if (!creatingViews) {
      return;
    }

    const newViews = queryToInput<CreatingView[], CreatingViewInput[]>(creatingViews);
    const viewIndex = newViews.findIndex(v => v.viewId === viewId);
    if (viewIndex < 0 || !newViews[viewIndex].selectedWidget) {
      return;
    }

    const selectedIndex = newViews[viewIndex].selectedWidget?.index;
    newViews[viewIndex] = { ...newViews[viewIndex], selectedWidget: null };
    setStore({ workspace: { creatingViews: newViews } });
    if (selectedIndex !== undefined) {
      setSelectedBorder(selectedIndex, false);
    }
  }, [creatingViews, viewId, index]);

  function canDrop(item: WidgetDragObject, viewId: string, index: number, widgets: WidgetInfo[]): boolean {
    let w = Widgets.find(w => w.id === item.id);
    if (!w) {
      return false;
    }
    w = w.createInstance(viewId, index, widgets);
    return w.isEnabled();
  }

  function updateWidgetWarning(widget?: Widget, props?: object, width?: number, height?: number): void {
    if (!widget?.options.getWidgetWarning) {
      widgetWarningRef.current = undefined;
      return;
    }
    if (width === undefined || height === undefined) {
      const root = document.getElementById(rootId);
      if (!root) {
        return;
      }
      const rect = root.getBoundingClientRect();
      width = rect.width;
      height = rect.height;
    }
    const warning = widget.options.getWidgetWarning(widget, props ?? {}, { width, height });
    if (JSON.stringify(widgetWarningRef.current) !== JSON.stringify(warning)) {
      widgetWarningRef.current = warning;
      const args: WidgetWarningEventArgs = { widgetIndex: index, warning };
      widgetEvent?.publish({ event: CommonWidgetEvent.WidgetWarning, args });
    }
  }

  return (
    <div
      ref={dropRef}
      className={classNames("WidgetPlaceholder-Root", "ui", "segment", { "WidgetPlaceholder-Root_dragOver": dragOver })}
      onClick={select}>
      { widget
        ? <Popup open={!!popupContent} content={popupContent} position="top center" offset={[0, -90]} trigger={
          <WidgetRenderWrapper
            component={widget.renderStub()}
            componentProps={componentProps}
            {...widgetProps}
            {...constructorProps}
            widgetId={widget.id}
            setCellProps={widgetSetCellProps}
          />
        }/> :
        <div className="WidgetPlaceholder-Empty" onClick={unselectWidget}>
          <div>
            <Popup open={!!popupContent} content={popupContent} position="top center" offset={[0, 20]} trigger={
              <div>
                <Popup
                  trigger={
                    <Button id={splitHorzBtnId} icon onClick={() => splitCell(ViewLayoutItemOrientation.Horizontal)}>
                      <Icon name="columns"/>
                    </Button>
                  }
                  content={__("Split Horizontally")}
                />
                <Popup
                  trigger={
                    <Button id={splitVertBtnId} icon onClick={() => splitCell(ViewLayoutItemOrientation.Vertical)}>
                      <Icon name="columns" style={{ writingMode: "vertical-lr", textOrientation: "sideways" }}/>
                    </Button>
                  }
                  content={__("Split Vertically")}
                />

                { getWidgets().length > 1 &&
                  <Popup
                    trigger={
                      <Button icon onClick={removeCell}><Icon name="trash"/></Button>
                    }
                    content={__("Remove Cell")}
                  />}
              </div>}
            />
          </div>
        </div>}
    </div>
  );
};

export default withStore(WidgetPlaceholder);
