import { ViewLayoutItem, ViewLayoutItemType, ViewLayoutItemOrientation } from "@core/types";
import { CreatingViewInput, WidgetInfo } from "@generated/graphql";
import { copyObject } from "utils";

export type LayoutCallback = (
  item: ViewLayoutItem,
  parentItem: ViewLayoutItem | undefined,
  layout: ViewLayoutItem[],
  parentLayout: ViewLayoutItem[] | undefined,
  index: number,
  widgetIndex: number,
  level: number) => boolean;

class ViewLayout {
  public layout: ViewLayoutItem[];

  constructor(layout: ViewLayoutItem[] = []) {
    this.layout = copyObject(layout) as ViewLayoutItem[];
  }

  getWidgetCount(inputLayout?: ViewLayoutItem[]): number {
    const layout = inputLayout ?? this.layout;
    let count = 0;
    for (const item of layout) {
      if (item.type === ViewLayoutItemType.Split && item.items) {
        count += this.getWidgetCount(item.items);
      }
      else if (item.type === ViewLayoutItemType.Cell) {
        count++;
      }
    }
    return count;
  }

  getLayoutToSave(): ViewLayoutItem[] {
    return this.layout;
  }

  foreach(
    callback: LayoutCallback,
    inputLayout?: ViewLayoutItem[],
    level: number = 0,
    getWidgetIndex?: () => number,
    parentItem?: ViewLayoutItem,
    parentLayout?: ViewLayoutItem[]
  ): boolean {
    const layout = inputLayout ?? this.layout;
    let widgetIndex = 0;
    let getWidgetIndexFunc = getWidgetIndex;
    if (!getWidgetIndexFunc) {
      getWidgetIndexFunc = () => widgetIndex++;
    }

    for (let i = 0; i < layout.length; i++) {
      const item = layout[i];
      const widgetIndex = item.type === ViewLayoutItemType.Cell ? getWidgetIndexFunc() : -1;
      if (callback(item, parentItem, layout, parentLayout, i, widgetIndex, level)) {
        return true;
      }
      if (item.items) {
        if (this.foreach(callback, item.items, level + 1, getWidgetIndexFunc, item, layout)) {
          return true;
        }
      }
    }
    return false;
  }

  splitCell(widgetIndex: number, orientation: ViewLayoutItemOrientation, view: CreatingViewInput, widgets: WidgetInfo[]): boolean {
    if (widgetIndex < 0) {
      return false;
    }

    return this.foreach((item, parent, layout, parentLayout, i, wi, level) => {
      if (widgetIndex !== wi) {
        return false;
      }

      if (parent?.type === ViewLayoutItemType.Split && parent?.orientation === orientation && parent?.items) {
        parent.items.splice(i + 1, 0, { type: ViewLayoutItemType.Cell, size: item.size / 2 });
        item.size /= 2;
      }
      else {
        layout.splice(i, 1, { type: ViewLayoutItemType.Split, orientation, size: item.size,
          items: [item, { type: ViewLayoutItemType.Cell, size: 0.5 }] });
        item.size = 0.5;
      }

      widgets.splice(widgetIndex + 1, 0, { widgetId: "", propsJSON: null, index: widgetIndex + 1 });
      for (let i = widgetIndex + 2; i < widgets.length; i++) {
        widgets[i].index++;
      }
      if (view.selectedWidget && view.selectedWidget.index > widgetIndex) {
        view.selectedWidget.index++;
      }
      return true;
    });
  }

  removeCell(widgetIndex: number, view: CreatingViewInput, widgets: WidgetInfo[]): boolean {
    if (widgetIndex < 0) {
      return false;
    }

    return this.foreach((item, parent, layout, parentLayout, i, wi, level) => {
      if (widgetIndex !== wi) {
        return false;
      }
      if (layout.length < 2) {
        return false;
      }

      layout.splice(i, 1);
      if (layout.length === 1 && parent?.type === ViewLayoutItemType.Split && parentLayout) {
        const parentIndex = parentLayout.indexOf(parent);
        parentLayout.splice(parentIndex, 1, layout[0]);
        layout[0].size = parent.size;
      }
      else if (i > 0) {
        layout[i - 1].size += item.size;
      }
      else if (i < layout.length) {
        layout[i].size += item.size;
      }

      widgets.splice(widgetIndex, 1);
      for (let i = widgetIndex; i < widgets.length; i++) {
        widgets[i].index--;
      }
      if (view.selectedWidget && view.selectedWidget.index >= widgetIndex) {
        view.selectedWidget.index--;
      }
      return true;
    });
  }

  addRowColBefore(orientation: ViewLayoutItemOrientation, view: CreatingViewInput, widgets: WidgetInfo[]): boolean {
    if (this.layout.length === 0) {
      return false;
    }

    const item = this.layout[0];
    if (item.type === ViewLayoutItemType.Split && item.orientation === orientation && item.items) {
      item.items.splice(0, 0, { type: ViewLayoutItemType.Cell, size: Math.min(1 / (item.items.length || 1), 0.2) });
    }
    else {
      this.layout.splice(0, 1, { type: ViewLayoutItemType.Split, orientation, size: item.size,
        items: [{ type: ViewLayoutItemType.Cell, size: 0.2 }, item] });
      item.size = 0.8;
    }

    widgets.splice(0, 0, { widgetId: "", propsJSON: null, index: 0 });
    for (let i = 1; i < widgets.length; i++) {
      widgets[i].index++;
    }
    if (view.selectedWidget) {
      view.selectedWidget.index++;
    }
    return true;
  }

  addRowColAfter(orientation: ViewLayoutItemOrientation, view: CreatingViewInput, widgets: WidgetInfo[]): boolean {
    if (this.layout.length === 0) {
      return false;
    }

    const item = this.layout[0];
    if (item.type === ViewLayoutItemType.Split && item.orientation === orientation && item.items) {
      item.items.push({ type: ViewLayoutItemType.Cell, size: Math.min(1 / (item.items.length || 1), 0.2) });
    }
    else {
      this.layout.splice(0, 1, { type: ViewLayoutItemType.Split, orientation, size: item.size,
        items: [item, { type: ViewLayoutItemType.Cell, size: 0.2 }] });
      item.size = 0.8;
    }

    widgets.push({ widgetId: "", propsJSON: null, index: widgets.length });
    return true;
  }
}

export default ViewLayout;
