import { useState, useEffect, useRef, useCallback, RefObject } from "react";
import { useNavigate } from "react-router-dom";
import { v1 as uuid } from "uuid";
import { ModuleInfo, ViewLayoutItem } from "@core/types";
import { UUID } from "@solid/types";
import { useStore } from "@core/store";
import { useConfig, SolidConfig, ConfigResult, copyFromSelectWidgetsLayout, LoadFromDbEvent, LoadFromDbArgs, getViewLayout, compareLayout } from "@core/store/actions";
import { CreatingView, CreatingViewInput, WidgetId, WidgetInfo, WidgetInfoInput } from "@generated/graphql";
import { ViewLayouts, ViewLayout } from "components/ViewLayouts";
import { Widgets, getEditModeWidgets } from "components/Widgets";
import {
  RTabPanel,
  CurrentChangedArgs,
  RWidgetLoader,
  WidgetCloseArgs,
  ModulesUpdateArgs,
  CanChangeToDisabledTabArgs,
  RSplitPanel,
  RBoxPanel,
  LayoutType
} from "components/Layout";
import { queryToInput, parseJSON, copyObject } from "utils";
import {isElectron} from "@solid/libs/utils";
import { MessageId } from "electron/types";
import { WindowRectChangedArgs } from "electron/Window";
import { ViewWindowArgs } from "electron/ViewWindow";
import { useDisplays } from "@core/actions";
import { DisplayInfo } from "electron/screen";
import {__} from "@solid/libs/i18n";
import type Electron from "electron";

export type ViewManagerParams = {
  mainPanelRef?: RefObject<RSplitPanel | RBoxPanel>;
  tabPanelRef?: RefObject<RTabPanel>;
  viewLoaderRef?: RefObject<RWidgetLoader>;
  onViewSelected?: (id: string, index: number) => void;
  onWriteToDb?: (config: SolidConfig) => void;
  loadFromDbEvent?: LoadFromDbEvent;
};

export type ViewManagerResult = {
  config: ConfigResult;
  currentViewId: string;
  currentView?: ModuleInfo;
  editMode: boolean;
  editViewId: string;
  saveViewId: string;
  cancelViewId: string;
  creatingViews?: CreatingView[] | null;
  userId?: UUID;
  displays: DisplayInfo[];
  selectView: (view: ModuleInfo) => void;
  createView: () => void;
  editView: (view: ModuleInfo) => void;
  cancelEdit: () => void;
  saveView: (name: string, saveAs?: boolean) => void;
  renameView: (name: string) => void;
  deleteView: (view: ModuleInfo) => void;
  viewToWindow: (view: ModuleInfo, displayId?: number) => void;
  viewToMain: (view: ModuleInfo) => void;
  shareView: (view: ModuleInfo, name: string) => void;
};

const viewNamePrefix = __("View #");

enum Step {
  None,
  Template,
  Layout,
  Widgets
}

type State = {
  step: Step;
  name: string;
  selectedWidget?: WidgetInfo;
};

type EditInfo = {
  item: CreatingView;
  view: ModuleInfo;
  layout?: LayoutType;
  layoutUpdated?: boolean;
};

export function useViewManager({ mainPanelRef, tabPanelRef, viewLoaderRef, onViewSelected, onWriteToDb, loadFromDbEvent }: ViewManagerParams): ViewManagerResult {
  const { store: { workspace: { creatingViews }, session: { info } } } = useStore();
  const config = useConfig({ onWriteToDb });
  const { views, allViews, startView, recentViews, layout, setConfig } = config;
  const { displays } = useDisplays();
  const viewStateRef = useRef(new Map<string, State>());
  const [saveViewId, setSaveViewId] = useState("");
  const [editViewId, setEditViewId] = useState("");
  const [cancelViewId, setCancelViewId] = useState("");
  const [editMode, setEditMode] = useState(false);
  const editInfoRef = useRef<EditInfo | undefined>();

  const navigate = useNavigate();

  useEffect(useCallback(() => {
    const id = `${Date.now()}.${Math.random()}`;
    const tabPanel = tabPanelRef?.current;
    const viewLoader = viewLoaderRef?.current;

    if (tabPanel) {
      tabPanel.currentChanged.subscribe(id, onCurrentViewChanged);
      tabPanel.canChangeToDisabledTab.subscribe(id, canChangeToDisabledTab);
    }

    if (viewLoader) {
      viewLoader.childClose.subscribe(id, onWidgetClose);
      viewLoader.modulesUpdated.subscribe(id, onModulesUpdated);
      viewLoader.editView.subscribe(id, args => editView(args.view));
      viewLoader.shareView.subscribe(id, args => shareView(args.view, args.name ?? args.view.name));
      viewLoader.deleteView.subscribe(id, args => deleteView(args.view));
      viewLoader.viewToWindow.subscribe(id, args => viewToWindow(args.view, args.displayId));
    }

    loadFromDbEvent?.subscribe(id, onLoadedFromDb);

    if (isElectron()) {
      window.ipcRenderer.on(MessageId.ViewToWindowReply, onViewToWindowReply);
      window.ipcRenderer.on(MessageId.ViewToMain, onViewToMain);
      window.ipcRenderer.on(MessageId.WindowRectChanged, onWindowRectChanged);
    }

    return function cleanup() {
      if (tabPanel) {
        tabPanel.currentChanged.unsubscribe(id);
        tabPanel.canChangeToDisabledTab.unsubscribe(id);
      }

      if (viewLoader) {
        viewLoader.childClose.unsubscribe(id);
        viewLoader.modulesUpdated.unsubscribe(id);
        viewLoader.editView.unsubscribe(id);
        viewLoader.shareView.unsubscribe(id);
        viewLoader.deleteView.unsubscribe(id);
        viewLoader.viewToWindow.unsubscribe(id);
      }

      loadFromDbEvent?.unsubscribe(id);

      if (isElectron()) {
        window.ipcRenderer.off(MessageId.ViewToWindowReply, onViewToWindowReply);
        window.ipcRenderer.off(MessageId.ViewToMain, onViewToMain);
        window.ipcRenderer.off(MessageId.WindowRectChanged, onWindowRectChanged);
      }
    };
  }, [views, allViews, recentViews, creatingViews, setConfig]));

  useEffect(useCallback(() => {
    updateSaveViewId();
    if (!creatingViews) {
      return;
    }
    for (const item of creatingViews.filter(v =>
      v.template && !v.layout && getViewState(v.viewId).step === Step.Template)) {
      selectLayout(item);
    }
    for (const item of creatingViews.filter(v =>
      v.template && v.layout && getViewState(v.viewId).step === Step.Layout)) {
      selectWidgets(item);
    }
    for (const item of creatingViews.filter(v =>
      v.template && v.layout && v.widgets && getViewState(v.viewId).step === Step.Widgets)) {
      const state = getViewState(item.viewId);
      if (item.selectedWidget) {
        const { widgetId, index } = item.selectedWidget;
        if (!state.selectedWidget || state.selectedWidget.widgetId !== widgetId || state.selectedWidget.index !== index) {
          updateWidgetTitles(item);
          state.selectedWidget = item.selectedWidget;
        }
      }
    }
  }, [views, allViews, recentViews, creatingViews, setConfig]), [creatingViews]);

  useEffect(useCallback(() => {
    updateEditViewId();
  }, [views, allViews]), [allViews]);

  useEffect(useCallback(() => {
    updateCancelViewId();
  }, [views]), [views]);

  useEffect(() => {
    viewLoaderRef?.current?.setLayoutChangeEnabled(editMode);
  }, [editMode]);

  const addRemoveView = useCallback((
    view?: ModuleInfo,
    idsToRemove?: string[],
    modifyAll?: boolean,
    creatingViews?: CreatingViewInput[],
    index?: number,
    updateRecent: boolean = false,
    layout?: string,
    local: boolean = false
  ): void => {
    let localOnly = true;
    let remove = false;
    let newViews = Array.from(views);

    const tabPanel = tabPanelRef?.current;
    if (tabPanel) {
      newViews.sort((a, b) => {
        let i1 = tabPanel.getTabIndex(a.id);
        if (i1 < 0) {
          i1 = newViews.indexOf(a);
        }
        let i2 = tabPanel.getTabIndex(b.id);
        if (i2 < 0) {
          i2 = newViews.indexOf(b);
        }
        return i1 - i2;
      });
    }

    let removedCount = 0;
    let newAllViews = Array.from(allViews);
    if (idsToRemove && (newViews.some(v => idsToRemove.includes(v.id)) || (modifyAll && newAllViews.some(v => idsToRemove.includes(v.id))))) {
      remove = true;
      localOnly = localOnly && !modifyAll && !newViews.some(v => idsToRemove.includes(v.id) && !v.isLocal);
      if (index !== undefined) {
        removedCount = newViews.filter((v, i) => idsToRemove.includes(v.id) && i < index).length;
      }
      newViews = newViews.filter(v => !idsToRemove.includes(v.id));
      if (modifyAll) {
        newAllViews = newAllViews.filter(v => !idsToRemove.includes(v.id));
      }
    }

    let i = index ?? -1;
    if (view) {
      if (i >= 0) {
        i -= removedCount;
      }

      if (!tabPanel) {
        newViews = [];
        i = 0;
      }

      localOnly = localOnly && !!view.isLocal;

      if (i < 0 || i >= newViews.length) {
        i = newViews.push(view) - 1;
      }
      else {
        newViews.splice(i, 0, view);
      }

      if (modifyAll) {
        const idx = newAllViews.findIndex(v => v.id === view.id);
        if (idx < 0) {
          newAllViews.push(view);
        }
        else {
          newAllViews[idx] = view;
        }
      }

      const tabCount = isElectron() ? newViews.filter(v => !v.inWindow).length : newViews.length;
      if (view.id !== startView.id && tabCount > 1) {
        const idx = newViews.findIndex(v => v.id === startView.id);
        if (idx >= 0) {
          newViews.splice(idx, 1);
          if (idx < i) {
            i--;
          }
        }
      }
    }

    if (view || remove) {
      const recentViewIds = view && updateRecent ? getRecentViewIds(view.id) : undefined;
      const allViews = modifyAll ? newAllViews : undefined;
      setConfig({ views: newViews, allViews, recentViewIds, layout }, localOnly || local, creatingViews);
    }

    if (view) {
      const viewLoader = viewLoaderRef ? viewLoaderRef.current : undefined;
      if (viewLoader) {
        viewLoader.nextActiveViewId = view.id;
        const id = `${Date.now()}.${Math.random()}`;
        viewLoader.modulesUpdated.subscribe(id, () => {
          viewLoader.modulesUpdated.unsubscribe(id);
          viewSelected(view.id, i);
        });
      } else {
        viewSelected(view.id, i);
      }
    }
  }, [views, allViews, recentViews, setConfig]);

  const selectView = useCallback((view: ModuleInfo): void => {
    if (isElectron() && view.inWindow) {
      if (window.ipcRenderer.sendSync(MessageId.ActivateWindow, view.id)) {
        setConfig({ recentViewIds: getRecentViewIds(view.id) });
        return;
      }

      setModuleProps(view.id, { inWindow: false });
    }

    if (!views.some(v => v.id === view.id)) {
      const updateRecent = !view.isAdmin;
      const local = view.isAdmin;
      addRemoveView(view, undefined, undefined, undefined, undefined, updateRecent, undefined, local);
    } else {
      viewSelected(view.id);
      setConfig({ recentViewIds: getRecentViewIds(view.id) });
    }
  }, [views, allViews, recentViews, setConfig, addRemoveView]);

  const viewSelected = useCallback((id: string, index: number = -1): void => {
    let i = -1;

    const tabPanel = tabPanelRef?.current;
    if (tabPanel) {
      i = index >= 0 ? index : tabPanel.getTabIndex(id);
      if (i < 0) {
        return;
      }

      const {tabCount} = tabPanel;
      if (i >= tabCount) {
        i = tabCount - 1;
      }
      if (tabCount) {
        tabPanel.currentIndex = i;
      }
    }
    else if (views.length > 0) {
      i = 0;
    }

    if (i >= 0 && onViewSelected) {
      onViewSelected(id, i);
    }
  }, [views, allViews]);

  const getRecentViewIds = useCallback((viewId: string): string[] => {
    const ids = recentViews.map(view => view.id);
    const index = recentViews.findIndex(view => view.id === viewId);
    if (index >= 0) {
      ids.splice(index, 1);
    }
    ids.splice(0, 0, viewId);
    while (ids.length > 10) {
      ids.splice(ids.length - 1, 1);
    }
    return ids;
  }, [recentViews]);

  const getNewViewIndex = useCallback((): number => {
    let names = allViews.map(v => v.name).concat(Array.from(viewStateRef.current.values()).map(v => v.name));
    if (creatingViews) {
      names = names.concat(creatingViews.map(v => v.name));
    }
    let index = 1;
    for (const name of names) {
      if (name.startsWith(viewNamePrefix)) {
        const i = parseInt(name.substring(viewNamePrefix.length));
        if (!isNaN(i) && index <= i) {
          index = i + 1;
        }
      }
    }
    return index;
  }, [allViews, creatingViews]);

  function getViewState(id: string): State {
    return viewStateRef.current.get(id) || { step: Step.None, name: "" };
  }

  function getInput(items: CreatingView[]): CreatingViewInput[] {
    return queryToInput<CreatingView[], CreatingViewInput[]>(items);
  }

  const createView = useCallback((): void => {
    const index = getNewViewIndex();
    const viewId = uuid();
    const view: ModuleInfo = {
      id: viewId + "_selectTemplate",
      name: viewNamePrefix + index,
      module: "View/SelectTemplate",
      props: { viewId },
      isLocal: true,
      aspectRatio: 16 / 9,
      ignoreAspectRatio: true,
      userId: info?.user?.id,
      hideHeaderControls: true
    };
    viewStateRef.current.set(viewId, { step: Step.Template, name: view.name });
    let newItems: CreatingViewInput[] = [];
    if (creatingViews) {
      newItems = newItems.concat(getInput(creatingViews));
    }
    const i = newItems.findIndex(item => item.viewId === viewId);
    if (i >= 0) {
      newItems[i] = { ...newItems[i], template: null, layout: null, widgets: null, layoutJSON: null };
    }
    else {
      newItems.push({ viewId, name: view.name, template: null, layout: null, widgets: null, selectedWidget: null, layoutJSON: null });
    }
    addRemoveView(view, undefined, undefined, newItems);
    setEditMode(true);
  }, [creatingViews, getNewViewIndex, addRemoveView]);

  const selectLayout = useCallback((item: CreatingView): void => {
    const {viewId} = item;
    viewStateRef.current.set(viewId, { step: Step.Layout, name: item.name });
    let created = false;
    try {
      if (!creatingViews) {
        return;
      }
      const prevView = views.find(v => v.id === viewId + "_selectTemplate");
      const view: ModuleInfo = {
        id: viewId + "_selectLayout",
        name: item.name,
        module: "View/SelectLayout",
        props: { viewId },
        isLocal: true,
        aspectRatio: prevView?.aspectRatio,
        designDisplayId: prevView?.designDisplayId,
        ignoreAspectRatio: true,
        userId: prevView?.userId,
        hideHeaderControls: true
      };
      const newItems = getInput(creatingViews);
      const i = newItems.findIndex(v => v.viewId === viewId);
      if (i < 0) {
        return;
      }
      newItems[i] = { ...newItems[i], layout: null, widgets: null };
      addRemoveView(view, [viewId + "_selectTemplate"], undefined, newItems, tabPanelRef?.current?.getTabIndex(viewId + "_selectTemplate"));
      created = true;
    }
    finally {
      if (!created) {
        const newItems = creatingViews ? getInput(creatingViews).filter(v => v.viewId !== viewId) : undefined;
        addRemoveView(undefined, [viewId + "_selectTemplate"], undefined, newItems);
      }
    }
  }, [views, creatingViews, addRemoveView]);

  const selectWidgets = useCallback((item: CreatingView): void => {
    const {viewId} = item;
    viewStateRef.current.set(viewId, { step: Step.Widgets, name: item.name });
    let created = false;
    try {
      if (!creatingViews || !creatingViews.some(v => v.viewId === viewId)) {
        return;
      }
      const layout = ViewLayouts.find(layout => layout.id === item.layout);
      if (!layout) {
        return;
      }
      const layoutClass = new ViewLayout(layout.layout);
      const widgetCount = layoutClass.getWidgetCount();
      const prevView = views.find(v => v.id === viewId + "_selectLayout");
      const view: ModuleInfo = {
        id: viewId + "_selectWidgets",
        name: item.name,
        module: layout.module,
        hasLayout: layout.hasLayout,
        isLocal: true,
        widgets: getEditModeWidgets(viewId, widgetCount),
        props: { alwaysUpdateWidgetContent: true },
        layout: layoutClass.layout,
        editMode: true,
        aspectRatio: prevView?.aspectRatio,
        designDisplayId: prevView?.designDisplayId,
        userId: prevView?.userId,
        hideHeaderControls: true,
        hasUnsavedChanges: true
      };
      addRemoveView(view, [viewId + "_selectLayout"], undefined, undefined, tabPanelRef?.current?.getTabIndex(viewId + "_selectLayout"));
      created = true;
    }
    finally {
      if (!created) {
        const newItems = creatingViews ? getInput(creatingViews).filter(v => v.viewId !== viewId) : undefined;
        addRemoveView(undefined, [viewId + "_selectLayout"], undefined, newItems);
      }
    }
  }, [views, creatingViews, addRemoveView]);

  const rememberEditView = useCallback((item: CreatingView | undefined, view: ModuleInfo | undefined): void => {
    editInfoRef.current = item && view ? { item, view, layout: getViewLayout(item.viewId, layout) } : undefined;
  }, [layout]);

  const saveView = useCallback((name: string, saveAs?: boolean): void => {
    if (!creatingViews || !saveViewId) {
      return;
    }

    const item = creatingViews.find(v => v.viewId === saveViewId && v.widgets && getViewState(v.viewId).step === Step.Widgets);
    if (!item) {
      console.error(`Could not find view with ID=${saveViewId}.`);
      return;
    }

    const { viewId, widgets } = item;
    let saved = false;
    const newItems = getInput(creatingViews).filter(v => v.viewId !== viewId);
    const selectId = viewId + "_selectWidgets";
    try {
      if (!widgets) {
        return;
      }

      const viewLayout = ViewLayouts.find(layout => layout.id === item.layout);
      if (!viewLayout) {
        console.error(`Could not find layout with ID=${item.layout} for view with ID=${saveViewId}.`);
        return;
      }

      let id = viewId;
      if (saveAs || allViews.some(v => v.id === viewId && (v.isSystem || (v.isShared && v.userId !== info?.user?.id)))) {
        id = uuid();
      }

      const widgetModules: ModuleInfo[] = widgets.map(({ widgetId }, index) => {
        let w = Widgets.find(w => w.id === widgetId);
        if (!w) {
          return { id: "", name: "", module: "" };
        }
        w = w.createInstance(id, index, widgets);
        return w.getModuleInfo(true);
      });

      const layoutClass = new ViewLayout(parseJSON<ViewLayoutItem[]>(item.layoutJSON));
      const prevView = views.find(v => v.id === selectId);

      const view: ModuleInfo = {
        id,
        name,
        module: viewLayout.module,
        hasLayout: viewLayout.hasLayout,
        widgets: widgetModules,
        templateId: item.template ?? undefined,
        layoutId: viewLayout.id,
        layout: layoutClass.getLayoutToSave(),
        aspectRatio: prevView?.aspectRatio,
        designDisplayId: prevView?.designDisplayId,
        userId: id === viewId && prevView?.userId ? prevView?.userId : info?.user?.id,
        isShared: prevView?.isShared && id === viewId,
        originalViewId: id !== viewId ? viewId : undefined
      };

      const curLayout = parseJSON<LayoutType>(layout) ?? {};
      let newLayout: string | undefined;
      const selectLayout = mainPanelRef?.current?.getWidgetLayout(selectId);
      if (selectLayout && selectLayout[selectId]) {
        curLayout.allViews = curLayout.allViews ?? {};
        curLayout.allViews[viewId] = copyFromSelectWidgetsLayout(selectLayout[selectId]);
        delete curLayout.allViews[selectId];
        newLayout = JSON.stringify(curLayout);
      }

      addRemoveView(view, [selectId], true, newItems, tabPanelRef?.current?.getTabIndex(selectId), true, newLayout);
      rememberEditView(undefined, undefined);
      saved = true;
      setEditMode(false);
    }
    finally {
      if (!saved) {
        addRemoveView(undefined, [selectId], undefined, newItems);
      }
    }
    navigate(`/view/${viewId}`);
  }, [views, allViews, creatingViews, saveViewId, layout, addRemoveView, rememberEditView]);

  const deleteView = useCallback((view: ModuleInfo): void => {
    if (isElectron()) {
      window.ipcRenderer.send(MessageId.CloseWindow, view.id);
    }
    addRemoveView(undefined, [view.id], true);
  }, [addRemoveView]);

  const editView = useCallback((view: ModuleInfo): void => {
    const newItems: CreatingView[] = getInput(creatingViews ?? []);
    if (!view || newItems.some(v => v.viewId === view.id)) {
      return;
    }
    setEditMode(true);
    const { id: viewId, name, module, hasLayout, widgets = [], templateId, layoutId, aspectRatio, designDisplayId, userId, isShared } = view;
    const layout = ViewLayouts.find(l => l.id === layoutId) ?? ViewLayouts.find(l => l.module === module);
    const layoutClass = new ViewLayout(view.layout);
    const item: CreatingViewInput = {
      viewId,
      name,
      template: templateId ?? null,
      layout: layout?.id ?? null,
      widgets: widgets.map((w, index) => {
        const widget = Widgets.find(wi => wi.id === w.widgetId) ?? Widgets.find(wi => wi.module === w.module);
        const { props } = w;
        const result: WidgetInfoInput =
          { widgetId: widget?.id ?? "", index, propsJSON: props ? JSON.stringify(props) : null };
        return result;
      }),
      selectedWidget: null,
      layoutJSON: JSON.stringify(layoutClass.layout)
    };
    const editView: ModuleInfo = {
      id: viewId + "_selectWidgets",
      name,
      module,
      hasLayout,
      isLocal: true,
      widgets: getEditModeWidgets(viewId, widgets.length),
      props: { alwaysUpdateWidgetContent: true },
      layout: layoutClass.layout,
      editMode: true,
      aspectRatio,
      designDisplayId,
      userId,
      isShared,
      hideHeaderControls: true
    };

    viewStateRef.current.set(viewId, { step: Step.Widgets, name });
    rememberEditView(item, editView);
    newItems.push(item);
    addRemoveView(editView, [viewId], undefined, newItems, tabPanelRef?.current?.getTabIndex(viewId), undefined, undefined, true);
  }, [creatingViews, addRemoveView, rememberEditView]);

  const renameView = useCallback((name: string): void => {
    const newAllViews = Array.from(allViews);
    const newViews = Array.from(views);
    const allIndex = newAllViews.findIndex(v => v.id === editViewId && !v.isSystem && v.name !== name);
    if (allIndex < 0) {
      return;
    }
    newAllViews[allIndex] = { ...newAllViews[allIndex], name };
    const viewIndex = newViews.findIndex(v => v.id === editViewId);
    if (viewIndex >= 0) {
      newViews[viewIndex] = { ...newViews[viewIndex], name };
    }
    setConfig({ allViews: newAllViews, views: viewIndex >= 0 ? newViews : undefined });
  }, [views, allViews, editViewId, setConfig]);

  const cancelEdit = useCallback((): void => {
    if (!cancelViewId) {
      return;
    }

    setEditMode(false);

    const id = cancelViewId;
    let newItems: CreatingViewInput[] | undefined;
    if (creatingViews && views.some(v => v.id === id && v.isLocal)) {
      newItems = getInput(creatingViews).filter(v =>
        v.viewId + "_selectTemplate" !== id && v.viewId + "_selectLayout" !== id && v.viewId + "_selectWidgets" !== id);
    }

    const viewId = id.replace("_selectTemplate", "").replace("_selectLayout", "").replace("_selectWidgets", "");
    const view = allViews.find(v => v.id === viewId);

    const curLayout = parseJSON<LayoutType>(layout) ?? {};
    let newLayout: string | undefined;
    if (curLayout.allViews && curLayout.allViews[id]) {
      delete curLayout.allViews[id];
      newLayout = JSON.stringify(curLayout);
    }

    addRemoveView(view, [id], undefined, newItems, tabPanelRef?.current?.getTabIndex(id), undefined, newLayout, true);
    rememberEditView(undefined, undefined);
    navigate(-1);
  }, [views, allViews, creatingViews, cancelViewId, layout, addRemoveView, rememberEditView]);

  const shareView = useCallback((view: ModuleInfo, name: string): void => {
    const { id, isShared, isSystem, isLocal } = view;
    if (isShared || isSystem || isLocal) {
      return;
    }
    const newAllViews = Array.from(allViews);
    const newView: ModuleInfo = { ...view, id: uuid(), name, isShared: true, userId: info?.user?.id, originalViewId: id };
    const index = newAllViews.findIndex(v => v.originalViewId === id && v.userId === info?.user?.id && v.isShared && !v.isSystem && v.name === name);
    if (index >= 0) {
      newAllViews.splice(index, 1, newView);
    }
    else {
      newAllViews.push(newView);
    }
    setConfig({ allViews: newAllViews });
  }, [allViews, setConfig]);

  const getCurrentViewId = useCallback((): string => {
    let id = "";
    if (tabPanelRef?.current) {
      id = tabPanelRef.current.currentTabId;
    }
    else {
      let vw = views;
      if (isElectron()) {
        vw = vw.filter(v => !v.inWindow);
      }
      if (vw.length > 0) {
        id = vw[0].id;
      }
    }
    return id;
  }, [views]);

  const updateSaveViewId = useCallback(() => {
    let id = "";
    if (creatingViews) {
      const currentId = getCurrentViewId();
      const view = creatingViews.find(v =>
        v.viewId + "_selectWidgets" === currentId && getViewState(v.viewId).step === Step.Widgets);
      if (view) {
        id = view.viewId;
      }
    }
    setSaveViewId(id);
  }, [views, creatingViews]);

  const updateEditViewId = useCallback(() => {
    const id = getCurrentViewId();
    setEditViewId(allViews.some(v => v.id === id && v.id !== startView.id) ? id : "");
  }, [views, allViews]);

  const updateCancelViewId = useCallback(() => {
    const id = getCurrentViewId();
    setCancelViewId(views.some(v => v.id === id && v.isLocal && isEditView(id)) ? id : "");
  }, [views]);

  const onCurrentViewChanged = useCallback((args: CurrentChangedArgs): void => {
    updateSaveViewId();
    updateEditViewId();
    updateCancelViewId();
  }, [updateSaveViewId, updateEditViewId, updateCancelViewId]);

  function updateWidgetTitles({ viewId, widgets }: CreatingView): void {
    if (!widgets || !tabPanelRef?.current) {
      return;
    }
    for (let i = 0; i < widgets.length; i++) {
      const widget = widgets[i];
      let w = Widgets.find(w => w.id === widget.widgetId);
      if (!w) {
        continue;
      }
      w = w.createInstance(viewId, i, widgets);
      const m = w.getModuleInfo();
      const widgetId = `${viewId}_widget${i}`;
      const title = tabPanelRef.current.getTitle(widgetId);
      if (title !== m.name) {
        tabPanelRef.current.setTitle(widgetId, m.name);
      }
    }
  }

  const onWidgetClose = useCallback(({ id }: WidgetCloseArgs): void => {
    let newItems: CreatingViewInput[] | undefined;
    if (creatingViews && views.some(v => v.id === id && v.isLocal)) {
      newItems = getInput(creatingViews).filter(v =>
        v.viewId + "_selectTemplate" !== id && v.viewId + "_selectLayout" !== id && v.viewId + "_selectWidgets" !== id);
    }
    if (id === cancelViewId) {
      setEditMode(false);
    }
    if (!newItems && !views.some(v => v.id === id)) {
      return;
    }
    addRemoveView(undefined, [id], undefined, newItems);
  }, [views, creatingViews, cancelViewId, addRemoveView]);

  const onModulesUpdated = useCallback(({ modules }: ModulesUpdateArgs): void => {
    const tabPanel = tabPanelRef?.current;
    if (tabPanel) {
      for (const { id } of modules) {
        if (isEditView(id)) {
          tabPanel.setEditIcon(id, true);
        }
        if (id.endsWith("_selectWidgets") && creatingViews) {
          const item = creatingViews.find(v => v.viewId + "_selectWidgets" === id);
          if (item && item.widgets) {
            for (let i = 0; i < item.widgets.length; i++) {
              if (!tabPanel.addClass(`${item.viewId}_widget${i}`, "Widget-Content_editMode") && i === 0) {
                tabPanel.addClass(id, "Widget-Content_editMode");
              }
            }
          }
        }
      }
      tabPanel.setTabsEnabled(!editMode);
      if (views.some(v => v.id === startView.id)) {
        const tabCount = isElectron() ? views.filter(v => !v.inWindow).length : views.length;
        tabPanel.setClosable(startView.id, tabCount > 1);
      }
    }

    updateSaveViewId();
    updateEditViewId();
    updateCancelViewId();

    const editInfo = editInfoRef.current;
    if (editInfo && !editInfo.layoutUpdated) {
      // TODO: correct logic, remove use of setTimeout
      setTimeout(() => {
        editInfo.layoutUpdated = true;
        const selectLayout = mainPanelRef?.current?.getWidgetLayout(editInfo.view.id);
        if (selectLayout && selectLayout[editInfo.view.id]) {
          editInfo.layout = copyFromSelectWidgetsLayout(selectLayout[editInfo.view.id]);
        }
      }, 1500);
    }
  }, [views, creatingViews, updateSaveViewId, updateEditViewId, updateCancelViewId]);

  function canChangeToDisabledTab(args: CanChangeToDisabledTabArgs): void {
    args.canChange = args.id.endsWith("_selectLayout") || args.id.endsWith("_selectWidgets") ||
      (args.prevId.endsWith("_selectWidgets") && args.id === args.prevId.replace("_selectWidgets", ""));
  }

  const viewToWindow = useCallback((view: ModuleInfo, displayId?: number): void => {
    if (view.isLocal || view.id === startView.id) {
      return;
    }

    const tabPanel = tabPanelRef?.current;
    const loader = viewLoaderRef?.current;

    const { id: viewId, name, rect } = view;
    const args: ViewWindowArgs = { viewId, name, index: tabPanel?.getTabIndex(viewId) ?? -1, displayId, rect };

    if (!tabPanel && loader && views.some(v => v.id === viewId)) {
      const id = `${Date.now()}.${Math.random()}`;
      loader.modulesUpdated.subscribe(id, () => {
        loader.modulesUpdated.unsubscribe(id);
        window.ipcRenderer.send(MessageId.ViewToWindow, args);
      });

      const newAllViews = Array.from(allViews);
      const i = newAllViews.findIndex(v => v.id === viewId);
      if (i >= 0) {
        newAllViews[i] = { ...newAllViews[i], inWindow: true, displayId };
      }
      setConfig({ views: [], allViews: newAllViews });
    }
    else {
      window.ipcRenderer.send(MessageId.ViewToWindow, args);
    }
  }, [views, allViews, setConfig]);

  const viewToMain = useCallback((view: ModuleInfo): void => {
    window.ipcRenderer.send(MessageId.CloseWindow, view.id);
    if (!tabPanelRef?.current) {
      addRemoveView({ ...view, inWindow: false });
    }
  }, [addRemoveView]);

  const setModuleProps = useCallback((viewId: string, props: Partial<ModuleInfo>): void => {
    const tabPanel = tabPanelRef?.current;
    const newViews = Array.from(views);
    const newAllViews = Array.from(allViews);
    const vi = newViews.findIndex(v => v.id === viewId && v.id !== startView.id);
    const avi = newAllViews.findIndex(v => v.id === viewId && v.id !== startView.id);

    let inWindow = false;
    if (tabPanel && vi >= 0) {
      inWindow = !!newViews[vi].inWindow;
    }
    else if (avi >= 0) {
      inWindow = !!newAllViews[avi].inWindow;
    }

    if (vi >= 0) {
      newViews[vi] = { ...newViews[vi], ...props };
    }
    if (avi >= 0) {
      newAllViews[avi] = { ...newAllViews[avi], ...props };
    }

    if (vi >= 0 || avi >= 0) {
      const localOnly = vi >= 0 && newViews[vi].isLocal;
      setConfig({ views: vi >= 0 ? newViews : undefined, allViews: avi >= 0 ? newAllViews : undefined }, localOnly);
    }

    if (tabPanel && inWindow && !props.inWindow) {
      const id = `${Date.now()}.${Math.random()}`;
      viewLoaderRef?.current?.modulesUpdated.subscribe(id, () => {
        viewLoaderRef?.current?.modulesUpdated.unsubscribe(id);
        viewSelected(viewId);
      });
    }
  }, [views, allViews, setConfig]);

  const onViewToWindowReply = useCallback((event: Electron.IpcRendererEvent, { viewId, displayId }: ViewWindowArgs): void => {
    setModuleProps(viewId, { inWindow: true, displayId });
  }, [setModuleProps]);

  const onViewToMain = useCallback((event: Electron.IpcRendererEvent, { viewId }: ViewWindowArgs): void => {
    setModuleProps(viewId, { inWindow: false });
  }, [setModuleProps]);

  const onWindowRectChanged = useCallback((event: Electron.IpcRendererEvent, { windowId, displayId, rect }: WindowRectChangedArgs): void => {
    setModuleProps(windowId, { displayId, rect });
  }, [setModuleProps]);

  const onLoadedFromDb = useCallback(({ views, allViews }: LoadFromDbArgs): void => {
    if (!tabPanelRef?.current && isElectron()) {
      for (const view of allViews.filter(v => !!v.inWindow)) {
        viewToWindow(view, view.displayId);
      }
    }
    if (tabPanelRef?.current && isElectron()) {
      for (const view of views.filter(v => !!v.inWindow)) {
        viewToWindow(view, view.displayId);
      }
    }
  }, [viewToWindow]);

  const currentViewId = getCurrentViewId();
  const currentView = views.find(v => v.id === currentViewId);

  useEffect(useCallback(() => {
    let item = creatingViews?.find(i => i.viewId + "_selectWidgets" === currentView?.id);
    if (!item || !currentView) {
      return;
    }

    const ignoreViewProps = ["selectedWidget", "__typename"];
    const ignoreWidgetProps = ["fixedWidth", "fixedHeight"];

    let hasChanges = true;
    if (editInfoRef.current) {
      const { item: infoItem, view: prevView, layout: prevLayout, layoutUpdated } = editInfoRef.current;

      const prevItem = { ...infoItem, widgets: infoItem.widgets?.map(w => {
        const props = copyObject(parseJSON(w.propsJSON) ?? null, undefined, ignoreWidgetProps);
        return { ...w, propsJSON: props ? JSON.stringify(props) : w.propsJSON };
      }) ?? null };

      item = { ...item, widgets: item.widgets?.map(w => {
        const props = copyObject(parseJSON(w.propsJSON) ?? null, undefined, ignoreWidgetProps);
        return { ...w, propsJSON: props ? JSON.stringify(props) : w.propsJSON };
      }) ?? null };

      hasChanges =
        JSON.stringify(copyObject(prevItem, undefined, ignoreViewProps)) !== JSON.stringify(copyObject(item, undefined, ignoreViewProps)) ||
        prevView.aspectRatio !== currentView.aspectRatio || prevView.designDisplayId !== currentView.designDisplayId;

      if (!hasChanges && prevLayout && layoutUpdated) {
        const selectLayout = mainPanelRef?.current?.getWidgetLayout(currentView.id);
        if (selectLayout && selectLayout[currentView.id]) {
          const layout = copyFromSelectWidgetsLayout(selectLayout[currentView.id]);
          hasChanges = !compareLayout(prevLayout, layout);
        }
      }
    }

    if (currentView.hasUnsavedChanges !== hasChanges) {
      setModuleProps(currentView.id, { hasUnsavedChanges: hasChanges });
    }
  }, [setModuleProps, creatingViews, currentView, layout]), [creatingViews, currentView, layout]);

  return {
    config,
    currentViewId,
    currentView,
    editMode,
    editViewId,
    saveViewId,
    cancelViewId,
    creatingViews,
    userId: info?.user?.id,
    displays,
    selectView,
    createView,
    editView,
    cancelEdit,
    saveView,
    renameView,
    deleteView,
    viewToWindow,
    viewToMain,
    shareView
  };
}

export function isEditView(id: string): boolean {
  return id.endsWith("_selectTemplate") || id.endsWith("_selectLayout") || id.endsWith("_selectWidgets");
}

export function isCameraSource(widgetInfo: WidgetInfo | undefined): boolean {
  const propJSON = widgetInfo?.propsJSON ? JSON.parse(widgetInfo?.propsJSON) : undefined;
  return !!propJSON?.object;
}

export function noDataSourceMessages(vm: ViewManagerResult): string[] | undefined {
  const currentView: CreatingView | undefined = vm.creatingViews ? vm.creatingViews[0] : undefined;

  if (currentView) {
    const isDeviceList = currentView.widgets?.find(widget => widget.widgetId === WidgetId.DeviceList);
    if (isDeviceList) {
      return undefined;
    }

    const cameras = currentView.widgets?.filter(widget => widget.widgetId === WidgetId.CameraLive || widget.widgetId === WidgetId.ArchiveViewer);
    if (cameras) {
      let noSourceMessages = undefined;
      let currentWidgetId: WidgetId | undefined;

      cameras.forEach(camera => {
        if (!isCameraSource(camera)) {
          currentWidgetId = camera.widgetId as WidgetId;
          return;
        }
      });
      if (currentWidgetId) {
        const deviceListWidget = Widgets.find(widget => widget.id === WidgetId.DeviceList);
        noSourceMessages = [__("add {{name}} widget", {name: deviceListWidget?.title})];
        const currentWidget = Widgets.find(widget => widget.id === currentWidgetId);
        currentWidget && noSourceMessages.push(__("assign {{name}} widget", {name: currentWidget.title}));
      }

      return noSourceMessages;
    }
  }

  return undefined;
}
