import React, { useRef, useEffect } from "react";
import ReactDOM from "react-dom";
import {useParams} from "react-router-dom";
import { useUserLazyQuery } from "@generated/graphql";
import { useStore } from "@core/store";
import { useConfig, useLayout } from "@core/store/actions";
import { RBoxPanel, RWidgetLoader } from "components/Layout";
import {isElectron} from "@solid/libs/utils";
import { MessageId } from "electron/types";
import {UUID} from "@solid/types";

const ViewWindow = () => {
  const { viewId } = useParams<{ viewId?: UUID }>();
  const { store: { session: { isLoggedIn } } } = useStore();
  const { views, allViews, layout, setConfig } = useConfig();
  const mainPanelRef = useRef<RBoxPanel>(null);
  const viewLoaderRef = useRef<RWidgetLoader>(null);
  const { allViewsLayoutRef } = useLayout({ layout, views, allViews, mainPanelRef, viewLoaderRef, setConfig, isLoaded: true });
  const [getUser, { data: userData, error: userError }] = useUserLazyQuery();

  const view = allViews.find(v => v.id === viewId);

  useEffect(() => {
    viewLoaderRef.current?.setLayoutChangeEnabled(false);
    if (isElectron()) {
      window.addEventListener("unload", event => {
        const root = document.getElementById("root");
        if (root) {
          const res = ReactDOM.unmountComponentAtNode(root);
          console.log("View window root component", res ? "unmounted" : "did not unmount");
        }
      });
    }
  }, []);

  useEffect(() => {
    if (view && isElectron()) {
      const { isShared, isSystem, userId } = view;
      if (isShared && !isSystem && userId) {
        getUser({ variables: { id: userId }});
      }
    }
  }, [view]);

  useEffect(() => {
    userError && console.error("Could not get user name:", userError);
    if (userData?.user?.name && window.ipcRenderer) {
      window.ipcRenderer.send(MessageId.SetWindowTitle, { windowId: view?.id, title: `${view?.name} (${userData.user.name})` });
    }
  }, [userData, userError]);

  return (
    <RBoxPanel id="main" rootElementId="root" ref={mainPanelRef} direction="top-to-bottom">
      { isLoggedIn && !!view && <RWidgetLoader ref={viewLoaderRef} modules={[view]} layoutRef={allViewsLayoutRef} /> }
    </RBoxPanel>
  );
};

export default ViewWindow;
