import React, {useState, useEffect, useRef} from "react";
import { useNavigate } from "react-router-dom";
import { Icon } from "semantic-ui-react";
import { ModuleInfo } from "@core/types";
import { useUserLazyQuery } from "@generated/graphql";
import { withStore, WithStoreProps } from "@core/store";
import { useConfig } from "@core/store/actions";
import { useDisplays } from "@core/actions";
import { RWidgetLoader } from "components/Layout";
import EditButton from "components/View/Manager/EditButton";
import ShareButton from "components/View/Manager/ShareButton";
import DeleteButton from "components/View/Manager/DeleteButton";
import ViewToWindowButton from "components/View/Manager/ViewToWindowButton";
import {isElectron} from "@solid/libs/utils";

import "./style.css";

type ViewHeaderProps = WithStoreProps & {
  view: ModuleInfo;
  loader: RWidgetLoader;
  hideCloseButton?: boolean;
  hideHeaderControls?: boolean;
  onClose?: (viewId: string) => void;
};

const ViewHeader = ({
  view,
  loader,
  hideCloseButton: hideClose,
  hideHeaderControls: hideHeader,
  onClose,
  store: { session: { info, isAdmin }, workspace: { creatingViews } }
}: ViewHeaderProps) => {
  const { id, name, isShared, isSystem, userId, hideCloseButton, hideHeaderControls } = view;
  const navigate = useNavigate();
  const { allViews, startView } = useConfig();
  const [getUser, { data: userData, error: userError }] = useUserLazyQuery();
  const { displays } = useDisplays();
  const [viewName, setViewName] = useState(name);
  const viewHeaderRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (isShared && !isSystem && userId) {
      getUser({ variables: { id: userId }});
    }
  }, []);

  useEffect(() => {
    userError && console.error("Could not get user name:", userError);
    if (userData?.user?.name) {
      setViewName(`${name} (${userData.user.name})`);
    }
  }, [userData, userError]);

  function editView(view: ModuleInfo): void {
    loader.editView.publish({ view });
  }

  function shareView(view: ModuleInfo, name: string): void {
    loader.shareView.publish({ view, name });
  }

  function deleteView(view: ModuleInfo): void {
    loader.deleteView.publish({ view });
  }

  function viewToWindow(view: ModuleInfo, displayId?: number): void {
    loader.viewToWindow.publish({ view, displayId });
    onClose && onClose(id);
    navigate(`/view/${startView.id}`);
  }

  const popupPosition = "bottom right";

  return (
    <div className="ViewHeader" ref={viewHeaderRef}>
      <span>{viewName}</span>

      <div className="view-controls">
        { !hideHeaderControls && !hideHeader && isAdmin && <EditButton view={view} editView={editView} popupPosition={popupPosition}/> }

        { !hideHeaderControls && !hideHeader && !isShared && isAdmin &&
          <ShareButton dialogAlignTo={viewHeaderRef} dialogAlignPosition="bottom" popupPosition={popupPosition}
            view={view} allViews={allViews} shareView={shareView} creatingViews={creatingViews} userId={info?.user?.id} showViews/> }

        { !hideHeaderControls && !hideHeader && !isSystem && (!isShared || userId === info?.user?.id) && isAdmin &&
          <DeleteButton view={view} viewName={viewName} deleteView={deleteView} popupPosition={popupPosition}/> }

        { !hideHeaderControls && !hideHeader && isElectron() &&
          <ViewToWindowButton dialogAlignTo={viewHeaderRef} dialogAlignPosition="bottom" popupPosition={popupPosition}
            view={view} displays={displays} viewToWindow={viewToWindow}/>}
      </div>
      { !hideCloseButton && !hideClose && <Icon name="close" className="ViewHeader-CloseButton" onClick={() => {
        onClose && onClose(id);
        navigate(`/view/${startView.id}`);
      }}/> }
    </div>
  );
};

export default withStore(ViewHeader);
