import React, { useState, useEffect } from "react";
import { Button, Segment, Popup } from "semantic-ui-react";
import { ReactSVG } from "react-svg";
import { withStore, WithStoreProps } from "@core/store";
import { ViewLayouts, ViewLayout } from "components/ViewLayouts";
import { CreatingView, CreatingViewInput, WidgetInfoInput } from "@generated/graphql";
import { queryToInput } from "utils";
import {__} from "@solid/libs/i18n";

import "./style.css";

type SelectLayoutProps = WithStoreProps & {
  viewId: string;
};

const SelectLayout = ({ viewId, store: { workspace: { creatingViews } }, setStore }: SelectLayoutProps) => {
  const [popupOpen, setPopupOpen] = useState(false);

  useEffect(() => {
    window.requestAnimationFrame(() => setTimeout(() => setPopupOpen(true), 100));
  }, []);

  function select(layoutIndex: number): void {
    if (!creatingViews) {
      return;
    }
    const newViews = queryToInput<CreatingView[], CreatingViewInput[]>(creatingViews);
    const index = newViews.findIndex(item => item.viewId === viewId);
    if (index < 0) {
      return;
    }
    const layout = ViewLayouts[layoutIndex];
    const layoutClass = new ViewLayout(layout.layout);
    const widgetCount = layoutClass.getWidgetCount();
    const widgets = new Array<WidgetInfoInput>(widgetCount);
    for (let i = 0; i < widgetCount; i++) {
      widgets[i] = { widgetId: "", propsJSON: null, index: i };
    }
    newViews[index] = { ...newViews[index], layout: layout.id, widgets, layoutJSON: JSON.stringify(layoutClass.layout) };
    setStore({ workspace: { creatingViews: newViews } });
  }

  return (
    <Segment className="SelectLayout-Root">
      <div className="SelectLayout-Main">
        <Popup open={popupOpen} content={__("Select base layout from the options below")} position="top center" offset={[0, 20]} trigger={
          <div>
            { ViewLayouts.map(({ id, name, icon }, index) =>
              <Button key={id} className={`SelectLayout-Button SelectLayout__${id}`} onClick={() => select(index)}><ReactSVG src={icon}/>{name}</Button>) }
          </div>}/>
      </div>
    </Segment>
  );
};

export default withStore(SelectLayout);
