import React, {useState, useEffect, useRef} from "react";
import {Icon, Popup, StrictPopupProps, Segment, List, Portal} from "semantic-ui-react";
import { ReactSVG } from "react-svg";
import { ModuleInfo } from "@core/types";
import { PropType } from "utils";
import { DisplayInfo } from "electron/screen";
import {__} from "@solid/libs/i18n";
import { useAccessability } from "@core/store/actions/accessability";

import CastIcon from "./images/cast_24px.svg";

type ViewToWindowButtonProps = {
  view: ModuleInfo;
  displays: DisplayInfo[];
  viewToWindow: (view: ModuleInfo, displayId?: number) => void;
  open?: boolean;
  onOpenClose?: (open: boolean) => void;
  disabled?: boolean;
  popupPosition?: PropType<StrictPopupProps, "position">;
  popupOffset?: PropType<StrictPopupProps, "offset">;
  dialogAlignTo: React.RefObject<HTMLDivElement>,
  dialogAlignPosition?: "right" | "bottom";
};

const ViewToWindowButton = ({
  view,
  displays,
  viewToWindow,
  open,
  onOpenClose,
  disabled,
  popupPosition = "top right",
  popupOffset,
  dialogAlignTo,
  dialogAlignPosition = "right"
}: ViewToWindowButtonProps) => {
  const { config: { limitedMonitorCount } } = useAccessability();
  const [modalOpen, setModalOpen] = useState(!!open);
  const dialogRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    setModalOpen(!!open);
  }, [open]);

  function getDefaultDisplayId(): number | undefined {
    const displayId = displays.find(d => d.id === view.displayId && d.windowCount === 0)?.id;
    return displayId ?? displays.find(d => d.isPrimary)?.id;
  }

  function openModal(open: boolean): void {
    if (onOpenClose) {
      onOpenClose(open);
    }
    setModalOpen(open);
  }

  function positionDialog(): void {
    if (!dialogAlignTo.current) {
      return;
    }

    if (!dialogRef.current) {
      return;
    }

    const dialog = dialogRef.current;
    const alignToRect = dialogAlignTo.current.getBoundingClientRect();
    const dialogStyle = window.getComputedStyle(dialog, null);

    // dialogPosition === "right"
    let left = `${alignToRect.right + 4}px`;
    let top = `${alignToRect.top}px`;

    if (dialogAlignPosition === "bottom") {
      const width = parseInt(dialogStyle.width) || 300;
      left = `${alignToRect.right - width}px`;
      top = `${alignToRect.bottom + 4}px`;
    }

    if (dialogStyle.left !== left || dialogStyle.top !== top) {
      dialog.setAttribute("style", `left: ${left}; top: ${top}`);
    }
  }

  function onClick(e: React.MouseEvent): void {
    e.stopPropagation();

    if (disabled) {
      return;
    }

    if (displays.length <= 1) {
      viewToWindow(view, getDefaultDisplayId());
    } else {
      openModal(true);
    }
  }

  return (
    <>
      <Popup position={popupPosition} offset={popupOffset} content={__("Cast to Display")} trigger={
        <div className="ViewList-RightButton" onClick={onClick}>
          {!limitedMonitorCount && <ReactSVG role="small-icon" className="ViewList-RightButtonSvg" src={CastIcon}/>}
        </div>
      }/>

      { modalOpen &&
        <Portal
          open={modalOpen}
          onMount={() => {
            if (modalOpen) {
              positionDialog();
            }
          }}
          onClose={() => openModal(false)
        }>
          <div ref={dialogRef} className="ViewToWindowPopup">
            <div className="ViewToWindowPopup-Header">
              <span>{__("Cast to Display...")}</span>
              <Icon name="close" onClick={(e: React.MouseEvent) => {
                e.stopPropagation();
                openModal(false);
              }}/>
            </div>
            <Segment className="ViewToWindowPopup-Body">
              <List className="ViewToWindowPopup-List">
                { displays.map(({ id, name, isPrimary, windowCount }) =>
                  <List.Item key={id} disabled={!isPrimary && windowCount >= 1} onClick={e => {
                    e.stopPropagation();

                    if (!disabled) {
                      viewToWindow(view, id);
                    }
                    openModal(false);
                  }}>
                    <List.Icon name="tv"/>
                    <List.Content>{name}</List.Content>
                  </List.Item>) }
              </List>
            </Segment>
          </div>
        </Portal> }
    </>
  );
};

export default ViewToWindowButton;
