import React, {useEffect, useRef, useState} from "react";
import {Icon, List, Popup, Ref, Portal} from "semantic-ui-react";
import {useNavigate} from "react-router-dom";
import { ReactSVG } from "react-svg";
import {ModuleInfo} from "@core/types";
import { useStore } from "@core/store";
import {useUserLazyQuery} from "@generated/graphql";
import {isElectron} from "@solid/libs/utils";
import ViewPreview from "components/View/ViewPreview";
import {ViewLayout} from "components/ViewLayouts";
import {getPreviewWidgets} from "components/Widgets";
import EditButton from "./EditButton";
import ShareButton from "./ShareButton";
import DeleteButton from "./DeleteButton";
import ViewToWindowButton from "./ViewToWindowButton";
import {ViewManagerResult} from "components/View";
import {__} from "@solid/libs/i18n";

type ViewButtonProps = {
  view: ModuleInfo;
  vm: ViewManagerResult;
  disabled?: boolean;
  editDisabled?: boolean;
  shareDisabled?: boolean;
  deleteDisabled?: boolean;
  viewToWindowDisabled?: boolean;
};

const ViewButton = ({ view, vm, editDisabled, shareDisabled, deleteDisabled, viewToWindowDisabled, ...props }: ViewButtonProps) => {
  const {
    config: { allViews },
    displays,
    selectView,
    editView,
    deleteView,
    viewToMain,
    viewToWindow,
    shareView,
    userId: currentUserId,
    creatingViews,
    editMode
  } = vm;

  const disabled = props.disabled || editMode;
  const navigate = useNavigate();
  const { store: { session: { isAdmin } } } = useStore();
  const [getUser, { data: userData, error: userError }] = useUserLazyQuery();
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [shareViewOpen, setShareViewOpen] = useState(false);
  const [viewToWindowOpen, setViewToWindowOpen] = useState(false);
  const [userName, setUserName] = useState("");
  const [previewLeft, setPreviewLeft] = useState(0);
  const [previewTop, setPreviewTop] = useState(0);
  // const [previewWidth, setPreviewWidth] = useState(200);
  // const [previewHeight, setPreviewHeight] = useState(200);
  const rootRef = useRef(document.getElementById("root"));
  const leftPanelRef = useRef(document.getElementById("main_left"));
  const viewItemRef = useRef<HTMLDivElement>(null);
  const viewPreviewRef = useRef<HTMLDivElement>(null);
  const [isPreviewOpen, openPreview] = useState(false);
  const [/*isMouseIn*/, setMouseIn] = useState(false);
  const timeoutRef = useRef<NodeJS.Timeout | undefined>(undefined);

  const previewWidth = 600;
  const previewHeight = 338;

  const inWindow = isElectron() && (!!view.inWindow || allViews.some(v => v.id === view.id && v.inWindow));

  const { id: viewId, name, module, hasLayout, isShared, isSystem, userId, icon, svgIcon } = view;
  const layoutClass = new ViewLayout(view.layout);
  const viewPreview: ModuleInfo = {
    id: viewId + "_preview",
    name,
    module,
    hasLayout,
    isLocal: true,
    widgets: getPreviewWidgets(view),
    layout: layoutClass.layout,
    isSystem,
    isShared,
    userId
  };

  useEffect(() => {
    if (isShared && !isSystem && userId) {
      getUser({ variables: { id: userId }});
    }
    return () => {
      clearTimeout(timeoutRef.current);
      setMouseIn(false);
    };
  }, []);

  useEffect(() => {
    userError && console.error("Could not get user name:", userError);
    if (userData?.user?.name) {
      setUserName(userData.user.name);
    }
  }, [userData, userError]);

  function select(): void {
    if (!deleteConfirmOpen && !shareViewOpen) {
      hidePopup();
      selectView(view);

      navigate(`/view/${view.id}`);
    }
  }

  function hidePopup(): void {
    setMouseIn(false);
    openPreview(false);
  }

  function showPopup(): void {
    setMouseIn(true);

    timeoutRef.current = setTimeout(() => {
      setMouseIn((isMouseIn) => {
        if (isMouseIn && (!disabled && !deleteConfirmOpen && !shareViewOpen && !viewToWindowOpen)) {
          openPreview(true);
        }
        return isMouseIn;
      });
    }, 300);
  }

  function positionPreview(): void {
    const root = rootRef.current;
    const leftPanel = leftPanelRef.current;
    const item = viewItemRef.current;
    const preview = viewPreviewRef.current;
    if (!root || !leftPanel || !item || !preview) {
      return;
    }

    const leftWidth = leftPanel.getBoundingClientRect().width;
    const viewItem = item.getBoundingClientRect();
    const left = leftWidth;
    let topPosition = viewItem.top;

    if (viewItem.top + preview.offsetHeight > root.clientHeight) {
      topPosition = root.clientHeight - preview.offsetHeight;
    }

    setPreviewLeft(left);
    setPreviewTop(topPosition);
  }

  const display = displays.find(d => d.id === view.displayId);
  const viewName = view.name + (userName ? " (" + userName + ")" : "");

  return (
    <>
      <Ref innerRef={viewItemRef}>
        <List.Item className={`ViewList-ViewButton ${view.id}`} onClick={select} disabled={disabled} onMouseLeave={hidePopup} onMouseEnter={showPopup}>
          {svgIcon ?
            <ReactSVG role="small-icon" className="ViewList-ViewButtonSvg" src={svgIcon}/> :
            <List.Icon name={icon || "binoculars"}/>}

          <List.Content className="name">
            {viewName}
          </List.Content>

          <List.Content className="view-controls">
            { inWindow && <><List.Icon name="tv"/>{ !!display && <small>{display.index + 1}</small> }</> }

            { inWindow &&
              <Popup position="top right" content={__("Return View to the Main Window")} trigger={
                <Icon name="sign-in" className="ViewList-RightButton" onClick={(e: React.MouseEvent) => {
                  e.stopPropagation();

                  if (disabled) {
                    return;
                  }

                  viewToMain(view);
                }}/>
              }/> }

            { !editDisabled && isAdmin && <EditButton view={view} editView={editView} disabled={disabled}/> }

            { !shareDisabled && !isShared && isAdmin &&
              <ShareButton dialogAlignTo={viewItemRef} disabled={disabled} open={shareViewOpen}
                onOpenClose={open => {
                  setShareViewOpen(open);
                  open && hidePopup();
                }}
                creatingViews={creatingViews} view={view} allViews={allViews} shareView={shareView} userId={currentUserId}
              /> }

            { !deleteDisabled && !isSystem && (!isShared || userId === currentUserId) && isAdmin &&
              <DeleteButton view={view} viewName={viewName} deleteView={deleteView} disabled={disabled} confirmOpen={deleteConfirmOpen}
                onConfirmOpenClose={open => {
                  setDeleteConfirmOpen(open);
                  open && hidePopup();
                }}
              /> }

            { !viewToWindowDisabled && isElectron() &&
              <ViewToWindowButton dialogAlignTo={viewItemRef} disabled={disabled} view={view} displays={displays} viewToWindow={viewToWindow}
                onOpenClose={isOpen => {
                  setViewToWindowOpen(isOpen);
                  isOpen && hidePopup();
                }}
              /> }
          </List.Content>
        </List.Item>
      </Ref>
      {allViews.some(v => v.id === view.id) && isPreviewOpen &&
      <Portal
        open={isPreviewOpen}
        onMount={() => positionPreview()}
      >
        <Ref innerRef={viewPreviewRef}>
          <div className="ViewPreview-Popup" style={{ left: previewLeft + "px", top: previewTop + "px" }}>
            <ViewPreview view={viewPreview} width={previewWidth} height={previewHeight}/>
          </div>
        </Ref>
      </Portal>}
    </>
  );
};

export default ViewButton;
