import React, {useState, useEffect, useRef} from "react";
import {Icon, Popup, Portal, Form, Segment, StrictPopupProps, List} from "semantic-ui-react";
import classNames from "classnames";
import { ModuleInfo } from "@core/types";
import { UUID } from "@solid/types";
import { CreatingView } from "@generated/graphql";
import { validateViewName } from "./shared";
import { PropType } from "utils";
import {__} from "@solid/libs/i18n";

import "./style.css";

type ShareButtonProps = {
  view: ModuleInfo;
  allViews: ModuleInfo[];
  shareView: (view: ModuleInfo, name: string) => void;
  disabled?: boolean;
  open?: boolean;
  onOpenClose?: (open: boolean) => void;
  creatingViews?: CreatingView[] | null;
  userId?: UUID;
  popupPosition?: PropType<StrictPopupProps, "position">;
  popupOffset?: PropType<StrictPopupProps, "offset">;
  dialogAlignTo: React.RefObject<HTMLDivElement>,
  dialogAlignPosition?: "right" | "bottom";
  showViews?: boolean;
};

const ShareButton = ({
  view,
  allViews,
  shareView,
  disabled,
  open,
  onOpenClose,
  creatingViews,
  userId,
  popupPosition = "top right",
  popupOffset,
  dialogAlignTo,
  dialogAlignPosition = "right",
  showViews = false
}: ShareButtonProps) => {
  const [modalOpen, setModalOpen] = useState(!!open);
  const [viewName, setViewName] = useState(view.name);
  const [viewNameError, setViewNameError] = useState("");
  const dialogRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    setModalOpen(!!open);
  }, [open]);

  function openModal(open: boolean): void {
    onOpenClose && onOpenClose(open);
    setModalOpen(open);
  }

  function positionDialog(): void {
    if (!dialogAlignTo.current) {
      return;
    }

    if (!dialogRef.current) {
      return;
    }

    const dialog = dialogRef.current;
    const alignToRect = dialogAlignTo.current.getBoundingClientRect();
    const dialogStyle = window.getComputedStyle(dialog, null);

    // dialogPosition === "right"
    let left = `${alignToRect.right + 4}px`;
    let top = `${alignToRect.top}px`;

    if (dialogAlignPosition === "bottom") {
      const width = parseInt(dialogStyle.width) || 300;
      left = `${alignToRect.right - width}px`;
      top = `${alignToRect.bottom + 4}px`;
    }

    if (dialogStyle.left !== left || dialogStyle.top !== top) {
      dialog.setAttribute("style", `left: ${left}; top: ${top}`);
    }
  }

  function getViewName(): string {
    const baseName = view.name;
    let name = baseName;
    let index = 1;
    while (validateViewName(view.id, name, allViews, creatingViews, userId)) {
      name = baseName + " " + index++;
    }
    return name;
  }

  return (
    <>
      <Popup content={__("Copy View to Shared")} position={popupPosition} offset={popupOffset} trigger={
        <Icon name="share alternate" className="ViewList-RightButton" onClick={(e: React.MouseEvent) => {
          e.stopPropagation();

          if (disabled) {
            return;
          }

          openModal(true);
        }}/>
      }/>

      { modalOpen &&
        <Portal
          open={modalOpen}
          onMount={() => {
            if (modalOpen) {
              positionDialog();
              setViewName(getViewName());
            }
          }}
          onClose={() => openModal(false)
        }>
          <div ref={dialogRef} className="ShareButtonPopup">
            <div className="ShareButtonPopup-Header">
              <span>{__("Share View As...")}</span>
              <Icon name="close" onClick={() => openModal(false)}/>
            </div>

            <Segment className="ShareButtonPopup-Body">
              <Form>
                <Form.Group>
                  <Form.Field control={Form.Input} className="ShareButtonPopup-ViewName"
                    value={viewName} autoFocus
                    onInput={(e: React.FormEvent<HTMLInputElement>) => {
                      const name = e.currentTarget.value;
                      setViewName(name);
                      setViewNameError(validateViewName(view.id, name.trim(), allViews, creatingViews, userId, false, true));
                    }}
                    onKeyDown={(e: React.KeyboardEvent) => {
                      if (e.keyCode === 13 && !viewNameError) {
                        openModal(false);
                        shareView(view, viewName.trim());
                      }
                    }}
                    error={viewNameError ? { content: viewNameError, pointing: "below" } : undefined}
                    width={16}
                  />
                </Form.Group>
                <Form.Group className={classNames({ "ShareButtonPopup-ButtonGroup": !showViews })}>
                  <Form.Field control={Form.Button} className="ShareButtonPopup-Button" content={__("Share")} width={8}
                    disabled={!!viewNameError} onClick={() => {
                      openModal(false);
                      shareView(view, viewName.trim());
                    }}
                  />
                  <Form.Field control={Form.Button} className="ShareButtonPopup-Button" content={__("Close")} width={8}
                    onClick={() => openModal(false)}
                  />
                </Form.Group>
              </Form>

              { showViews &&
                <div className="ShareButtonPopup-List">
                  <List>
                    { allViews.filter(v => v.isShared && v.userId && v.userId === userId).map(view =>
                      <List.Item key={view.id}><List.Content>{view.name}</List.Content></List.Item>) }
                  </List>
                  <div className="ShareButtonPopup-ListHeader">{__("Other Shared Views:")}</div>
                  <List>
                    { allViews.filter(v => v.isShared && (!v.userId || v.userId !== userId)).map(view =>
                      <List.Item key={view.id} className="ShareButtonPopup-ListItemOther"><List.Content>{view.name}</List.Content></List.Item>
                    ) }
                  </List>
                </div> }
            </Segment>
          </div>
        </Portal> }
    </>
  );
};

export default ShareButton;
