import React, { useState, useEffect, useCallback } from "react";
import { Button, Form, Header, Icon, List, Modal } from "semantic-ui-react";
import { ModuleInfo } from "@core/types";
import { noDataSourceMessages, ViewManagerResult } from "components/View";
import { Widgets } from "components/Widgets";
import { validateViewName } from "./shared";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

type EditControlsProps = {
  vm: ViewManagerResult;
};

const EditControls = ({ vm }: EditControlsProps) => {
  const {
    config: { views, allViews },
    saveViewId,
    cancelViewId,
    creatingViews,
    cancelEdit,
    saveView,
    userId
  } = vm;

  const [view, setView] = useState<ModuleInfo | undefined>();
  const [viewName, setViewName] = useState("");
  const [viewNameError, setViewNameError] = useState("");
  const [noSourceMessages, setNoSourceMessages] = useState<string[]>();
  const [isModalVisible, setIsModalVisible] = useState<boolean>(false);
  const [isOnSaveClick, setIsOnSaveClick] = useState<boolean>(false);

  useEffect(useCallback(() => {
    const view = views.find(v => v.id === saveViewId + "_selectWidgets");
    setView(view);
    setViewName(getViewName(view, allViews));
  }, [vm]), [saveViewId]);

  useEffect(() => {
    setNoSourceMessages(noDataSourceMessages(vm));
  }, [vm]);

  function getViewName(view: ModuleInfo | undefined, allViews: ModuleInfo[]): string {
    let name = "";
    if (view) {
      name = view.name;
      if (allViews.some(v => v.id + "_selectWidgets" === view.id && (v.isSystem || (v.isShared && v.userId !== userId)))) {
        let index = 1;
        for (const v of allViews.filter(v => v.name.toUpperCase().startsWith(view.name.toUpperCase()))) {
          const i = parseInt(v.name.substring(view.name.length).trim());
          if (!isNaN(i) && i >= index) {
            index = i + 1;
          }
        }
        name = view.name + " " + index;
      }
    }
    return name;
  }

  const onInput = useCallback((e: React.FormEvent<HTMLInputElement>) => {
    const name = e.currentTarget.value;
    setViewName(name);
    setViewNameError(validateViewName(saveViewId, name.trim(), allViews, creatingViews, userId));
  }, [vm]);

  const save = useCallback((saveAs?: boolean) => {
    const nameError = validateViewName(saveViewId, viewName.trim(), allViews, creatingViews, userId, saveAs);
    setViewNameError(nameError);
    if (nameError) {
      return;
    }

    const view = creatingViews?.find(v => v.viewId === saveViewId);
    if (!view || !view.widgets) {
      console.error("Save error: Invalid creating view.");
      return;
    }
    for (const wi of view.widgets) {
      const widget = Widgets.find(w => w.id === wi.widgetId);
      if (!widget || !widget.options.getViewValidationError) {
        continue;
      }
      const viewError = widget.options.getViewValidationError(view.widgets);
      if (viewError) {
        Log.error(viewError);
        return;
      }
    }

    saveView(viewName.trim(), saveAs);
  }, [vm, viewName]);

  const saveClick = useCallback(() => save(), [save]);
  const saveAsClick = useCallback(() => save(true), [save]);

  const onSaveClick = useCallback(() => {
    if (noSourceMessages) {
      setIsModalVisible(true);
      setIsOnSaveClick(true);
      return;
    }
    saveClick();
  }, [save, noSourceMessages]);

  const onSaveAsClick = useCallback(() => {
    if (noSourceMessages) {
      setIsModalVisible(true);
      setIsOnSaveClick(false);
      return;
    }
    saveAsClick();
  }, [save, noSourceMessages]);

  return (
    <div className="EditControls">
      <Form>
        <Form.Group>
          <Form.Field control={Form.Input} className="EditControls-ViewName" value={viewName} autoFocus
            onInput={onInput} error={viewNameError ? { content: viewNameError, pointing: "below" } : undefined}
            width={16} disabled={!saveViewId}/>
        </Form.Group>
        <Form.Group className="EditControls-ButtonGroup">
          <Form.Field control={Form.Button} className="EditControls-Button" content={__("Save")} width={5}
            disabled={!saveViewId || !!viewNameError || !view || view.isSystem || (view.isShared && view.userId !== userId)}
            onClick={onSaveClick}/>
          <Form.Field control={Form.Button} className="EditControls-Button" content={__("Save as")} width={5}
            disabled={!saveViewId || !!viewNameError} onClick={onSaveAsClick}/>
          <Form.Field control={Form.Button} className="EditControls-Button" content={__("Revert")} width={5}
            disabled={!cancelViewId} onClick={() => cancelEdit()}/>
        </Form.Group>
      </Form>
      {isModalVisible && noSourceMessages &&
      <Modal open onClose={() => setIsModalVisible(false)}>
        <Header>{__("Save")}</Header>
        <Modal.Content>
          <List>
            <span>{`${__("Some widgets has no data source:")}`}</span><br/>
            {noSourceMessages.map((message, index) =>
              <span key={`${index}_${message}`} className="NoSourceMessages-List">
                <List.Item>
                  {`\u2013 ${message}`}
                </List.Item>
                <List.Item>
                  {index < noSourceMessages.length - 1 && __("or")}
                </List.Item>
              </span>)}
          </List>
        </Modal.Content>
        <Modal.Actions>
          <Button onClick={() => setIsModalVisible(false)}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
          <Button color="orange" onClick={isOnSaveClick ? saveClick : saveAsClick}>
            <Icon name="check"/>{__("Save")}
          </Button>
        </Modal.Actions>
      </Modal>}
    </div>
  );
};

export default EditControls;
