import React, { useState, useEffect } from "react";
import { Icon, Modal, Header, Popup, StrictPopupProps, Button } from "semantic-ui-react";
import { ModuleInfo } from "@core/types";
import { PropType } from "utils";
import {__} from "@solid/libs/i18n";

import "./style.css";

type DeleteButtonProps = {
  view: ModuleInfo;
  viewName: string;
  deleteView: (view: ModuleInfo) => void;
  confirmOpen?: boolean;
  onConfirmOpenClose?: (open: boolean) => void;
  disabled?: boolean;
  popupPosition?: PropType<StrictPopupProps, "position">;
  popupOffset?: PropType<StrictPopupProps, "offset">;
};

const DeleteButton = ({ view, viewName, deleteView, confirmOpen, onConfirmOpenClose, disabled, popupPosition = "top right", popupOffset }: DeleteButtonProps) => {
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(!!confirmOpen);

  useEffect(() => {
    setDeleteConfirmOpen(!!confirmOpen);
  }, [confirmOpen]);

  function openConfirm(open: boolean): void {
    setDeleteConfirmOpen(open);
    onConfirmOpenClose && onConfirmOpenClose(open);
  }

  return (
    <>
      <Popup position={popupPosition} offset={popupOffset} content={__("Delete View")} trigger={
        <Icon name="trash alternate" className="ViewList-RightButton" onClick={(e: React.MouseEvent) => {
          e.stopPropagation();

          if (disabled)
          {
            return;
          }

          openConfirm(true);
        }}/>
      }/>

      {deleteConfirmOpen &&
        <Modal open={deleteConfirmOpen} onClose={() => openConfirm(false)}>
          <Header>{__("Delete View")}</Header>
          <Modal.Content>{__("Are you sure you want to delete view '{{name}}'?", {name: viewName})}</Modal.Content>
          <Modal.Actions>
            <Button
              negative
              onClick={() => {
                deleteView(view);
                openConfirm(false);
              }}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => openConfirm(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>}
    </>
  );
};

export default DeleteButton;
