import React, { useState, useEffect } from "react";
import classNames from "classnames";
import { ReactSVG } from "react-svg";
import { WidgetResizeEvent, WidgetResizeArgs } from "components/Layout";

import ColPlusAfter from "./images/table-column-plus-after.svg";
import ColPlusBefore from "./images/table-column-plus-before.svg";
import RowPlusAfter from "./images/table-row-plus-after.svg";
import RowPlusBefore from "./images/table-row-plus-before.svg";

import "./style.css";

export type PanelSide = "top" | "bottom" | "left" | "right";

type AddRowColPanelProps = {
  viewId: string;
  side: PanelSide;
  contentResize: WidgetResizeEvent;
  onClick: (side: PanelSide) => void;
};

const AddRowColPanel = ({ viewId, side, contentResize, onClick } : AddRowColPanelProps) => {
  const [disabled, setDisabled] = useState(false);

  const widgetId = `${viewId.replace("_selectWidgets", "")}_widget0`;

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    contentResize.subscribe(id, onContentResize);

    return function cleanup() {
      contentResize.unsubscribe(id);
    };
  });

  function updateDisabled(): void {
    const view = document.getElementById(viewId);
    if (!view) {
      return;
    }

    const rect = view.getBoundingClientRect();

    let minWidgetWidth = 0;
    let minWidgetHeight = 0;
    const widget = document.getElementById(widgetId);
    if (widget) {
      const style = window.getComputedStyle(widget, null);
      minWidgetWidth = parseInt(style.minWidth) || 0;
      minWidgetHeight = parseInt(style.minHeight) || 0;
    }

    const style = window.getComputedStyle(view, null);
    const minViewWidth = parseInt(style.minWidth) || 0;
    const minViewHeight = parseInt(style.minHeight) || 0;

    if (side === "top" || side === "bottom") {
      if (minViewHeight && rect.height) {
        setDisabled(minViewHeight + minWidgetHeight >= rect.height);
      }
    }
    else if (minViewWidth && rect.width) {
      setDisabled(minViewWidth + minWidgetWidth >= rect.width);
    }
  }

  function onContentResize({ id }: WidgetResizeArgs): void {
    if (id === viewId) {
      window.requestAnimationFrame(() => updateDisabled());
    }
  }

  function getSrc(): string {
    switch (side) {
      case "left": return ColPlusBefore;
      case "right": return ColPlusAfter;
      case "top": return RowPlusBefore;
      case "bottom": return RowPlusAfter;
      default: return "";
    }
  }

  return (
    <div className={classNames("AddRowColPanel", { "AddRowColPanel_disabled": disabled })}
      onClick={() => { if (!disabled) onClick(side); }}>
      <ReactSVG role="middle-icon" src={getSrc()}/>
    </div>
  );
};

export default AddRowColPanel;
