import React, { useEffect, useState, useCallback } from "react";
import { Segment, Icon } from "semantic-ui-react";
import { ReactSVG } from "react-svg";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { WidgetProps, Widgets } from "components/Widgets";
import { DragObject, DragObjectType, isDragObject } from "@core/types";
import { CreatingView, WidgetInfo } from "@generated/graphql";
import { withStore, WithStoreProps } from "@core/store";
import { parseJSON } from "utils";
import {__} from "@solid/libs/i18n";
import { UUID } from "@solid/types";

import "./style.css";
import "../style.css";

type VideoTouringCellStubProps = WithStoreProps & WidgetProps & {
  object?: DragObject;
  objects?: DragObject[];
  delay?: string
};

const VideoTouringCellStub = ({
  viewId,
  index,
  setCellProps,
  widgetId,
  store: { workspace: { creatingViews } },
  setStore,
  ...props
}: VideoTouringCellStubProps) => {
  const [object, setObject] = useState<DragObject | undefined>();
  const widget = Widgets.find(w => w.id === widgetId);
  const [objList, setObjList] = useState<DragObject[]>([]);

  const findWidget = useCallback((): [CreatingView[] | undefined, WidgetInfo[] | undefined, number | undefined, number | undefined] => {
    if (!creatingViews || index === undefined) {
      return [undefined, undefined, undefined, undefined];
    }

    const views = Array.from(creatingViews);
    const viewIndex = views.findIndex(v => v.viewId + "_selectWidgets" === viewId);
    if (viewIndex < 0 || viewIndex >= views.length) {
      return [undefined, undefined, undefined, undefined];
    }

    const view = views[viewIndex];
    if (!view || !view.widgets || index >= view.widgets.length) {
      return [undefined, undefined, undefined, undefined];
    }

    const widgets = Array.from(view.widgets);
    return [views, widgets, viewIndex, index];
  }, [creatingViews, viewId, index]);

  const setObjectToStore = useCallback((object: DragObject | DragObject[]) => {
    const [views, widgets, viewIndex, widgetIndex] = findWidget();
    if (!views || !widgets || viewIndex === undefined || widgetIndex === undefined) {
      return;
    }

    const props = parseJSON(widgets[widgetIndex].propsJSON) ?? {};
    let objects: DragObject[] = [];
    if (isDragObject(object)) {
      objects = objList.concat(object);
    } else {
      objects = object;
    }
    props["objects"] = objects;
    widgets[widgetIndex] = { ...widgets[widgetIndex], propsJSON: JSON.stringify(props) };
    views[viewIndex] = { ...views[viewIndex], widgets };
    setStore({ workspace: { creatingViews: views } });
  }, [findWidget, objList]);

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: DragObjectType.Camera,

    drop: useCallback((item, monitor) => {
      if (item.type === DragObjectType.Camera && !objList.some(cam => cam.obj["data-id"] === item.obj["data-id"])) {
        setObject(item);
        setObjectToStore(item);
      }
      return undefined;
    }, [setObjectToStore]),

    collect: monitor => ({
      dragOver: monitor.isOver()
    }),
  });

  useEffect(() => {
    if (props.objects && props.object && !objList.some(cam => cam.obj["data-id"] === props.object?.obj["data-id"])) {
      const objects = objList.concat(props.object);
      setObjList(objects);
    }
  }, [object]);

  useEffect(() => {
    if (props.objects) {
      setObjList(props.objects);
    }
  }, [props.objects]);

  function deleteFromList(id: UUID): void {
    const newList = objList.filter(camera => camera.obj["data-id"] !== id);
    setObjList(newList);
    setObjectToStore(newList);
  }

  function onRemove(e: MouseEvent, camera: DragObject) {
    e.stopPropagation();
    deleteFromList(camera.obj["data-id"]);
  }

  return (
    <div ref={dropRef} className={classNames({ "VideoTouringCellStub-Root_dragOver": dragOver })}>
      { objList.length === 0 ?
        <Segment className="VideoTouringCellStub-Empty">
          { !!widget && <ReactSVG role="icon" src={widget.icon} /> }
          <p>{widget?.title}</p>
          <p>{__("Drag and drop camera to activate the touring")}</p>
        </Segment> :
        <Segment className="VideoTouringCellStub">
          <div className="ui segment CameraList-Root">
            {objList.length > 0 && objList.map((camera, index) =>
              <div className="CameraListItem-Root" key={`${camera.obj["data-id"]}_editList_${index}`}>
                <div className="CameraListItem-Content">
                  <div className="CameraListItem-Name">{camera.obj["name"]}</div>
                  <Icon name="close" className="CameraListItem-Remove" onClick={(e: MouseEvent) => onRemove(e, camera)}/>
                </div>
              </div>
            )
      }
          </div>
        </Segment> }
    </div>
  );
};

export default withStore(VideoTouringCellStub);
