import React, { useCallback, useEffect, useState } from "react";
import { withStore, WithStoreProps } from "@core/store";
import { DeviceFunctionalAspectType, useDeviceListByAspectTypesQuery } from "@generated/graphql";
import { WidgetProps } from "components/Widgets";
import VideoTouringList from "./VideoTouringList";
import DelayTouring from "./DelayTouring";
import CameraCell from "components/CellContainer/CameraCell";
import WithQueryStatus from "components/WithQueryStatus";
import { Icon, Popup, Segment } from "semantic-ui-react";
import { DragObject, DragObjectType, isObject } from "@core/types";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { UUID } from "@solid/types";
import {__} from "@solid/libs/i18n";

import "./style.css";

type VideoTouringProps = WithStoreProps & WidgetProps & {
  editMode: boolean,
  objects?: CameraObj[],
  delay?: string
};

export type CameraObj = {
  type: DragObjectType;
  obj: {
    id: UUID,
    name: string,
  };
  cameraDisabled?: boolean
};

const VideoTouring = (props: VideoTouringProps) => {
  const {editMode, setCellProps, cellProps, objects, delay} = props;
  const delayDuration = !isNaN(Number(delay)) ? Number(delay) : 5;
  const { loading, error, data } = useDeviceListByAspectTypesQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const [currentList, setCurrentList] = useState<CameraObj[]>([]);
  const [videoCamera, setVideoCamera] = useState<CameraObj | undefined>();
  const [currentIndex, setCurrentIndex] = useState<number | undefined>();
  const [duration, setDuration] = useState<number>(delayDuration);
  const [listVisible, setListVisible] = useState<boolean>(true);

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: DragObjectType.Camera,

    drop: useCallback((item, monitor) => {
      const object: DragObject = item;
      if (isObject(object.obj)) {
        const obj = object.obj;
        const device = data?.devicesByAspectTypes.find(device => isObject(obj) && obj["data-id"] === device.id);
        if (device) {
          const newCamera = {
            type: DragObjectType.Camera,
            obj: {
              id: device.id,
              name: device.name,
            },
            cameraDisabled: !device.enabled
          };
          setVideoCamera(newCamera);
        }
      }
      return undefined;
    }, [data]),

    collect: monitor => ({
      dragOver: monitor.isOver()
    }),

    canDrop: useCallback((item: DragObject, monitor) => {
      return editMode && (isObject(item.obj) && item.type === DragObjectType.Camera && !currentList.some(camera => camera.obj["data-id"] === item.obj["data-id"]));
    }, [currentList, videoCamera])
  });

  useEffect(() => {
    if (videoCamera && !currentList.some(camera => camera.obj.id === videoCamera.obj.id)) {
      const newList = currentList.concat(videoCamera);
      setCurrentList(newList);

      if (typeof currentIndex === "undefined") {
        setCurrentIndex(0);
      }
    }
  }, [videoCamera]);

  useEffect(() => {
    const intervalId = setInterval(goNextVideo, duration * 1000);
    return () => clearInterval(intervalId);
  }, [currentIndex, currentList, duration]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }

    const title =
      <div className="CameraList-Title">
        <div className="CameraList-TitleName">{cellProps?.title}</div>

        <Popup
          trigger={
            <Icon
              name="th list"
              className="CameraList-TitleIcon"
              onClick={() => setListVisible(!listVisible)}
            />}
          content={listVisible ? __("Hide list of cameras") : __("Show list of cameras")}
          position="bottom right"
        />
      </div>;

    setCellProps({ title });
  }, [listVisible]);

  useEffect(() => {
    if (data && objects) {
      const cameraList: CameraObj[] = [];
      objects.forEach(cam => {
        const dev = data.devicesByAspectTypes.find(dev => dev.id === cam.obj["data-id"]);
        if (dev) {
          cameraList.push({
            type: cam.type,
            obj: {
              id: cam.obj["data-id"],
              name: dev.name
            },
            cameraDisabled: !dev.enabled
          });
        }
      });
      setCurrentList(cameraList);
      if (cameraList.length > 0) {
        setCurrentIndex(0);
      }
    }
  }, [objects, data]);

  function deleteFromList(id: UUID, index: number): void {
    const newList = currentList.filter(camera => camera.obj.id !== id);
    setCurrentList(newList);
    if (newList.length === 0) {
      setCurrentIndex(undefined);
    } else if (index === currentIndex && index >= newList.length - 1) {
      setCurrentIndex(0);
    } else if (index === currentIndex && index < newList.length - 1) {
      setCurrentIndex(index);
    } else if (currentIndex && index < currentIndex && index < newList.length - 1) {
      setCurrentIndex(currentIndex - 1);
    }
  }

  const goNextVideo = useCallback(() => {
    if (typeof currentIndex === "undefined") {
      return;
    }
    if (currentIndex < currentList.length - 1) {
      setCurrentIndex(currentIndex + 1);
    }
    if (currentIndex !== 0 && currentIndex === currentList.length - 1) {
      setCurrentIndex(0);
    }
  }, [currentIndex, currentList]);

  function getObj(camera: CameraObj): DragObject {
    return {
      type: camera.type,
      obj: {
        "data-id": camera.obj.id,
        name: camera.obj.name
      },
      cameraDisabled: camera.cameraDisabled
    };
  }

  return (
    <WithQueryStatus loading={loading} error={error}>
      <div className={classNames("VideoTouring-Wrapper", {"VideoTouring-Wrapper_dragOver": dragOver})} ref={dropRef}>
        <Segment className="VideoTouring-Player">
          {currentList.length > 0 && typeof currentIndex === "number" ?
            <CameraCell object={getObj(currentList[currentIndex])} stepBackControl={false}/> :
            <div className="VideoTouring-Message">
              <div>
                {__("Drag and drop camera to activate the touring")}
              </div>
            </div>
          }
        </Segment>
        {listVisible &&
          <Segment className="VideoTouring-Control">
            <VideoTouringList
              cameraList={currentList}
              deleteFromList={deleteFromList}
              editMode={editMode}
              playingIndex={currentIndex}
              setCurrentIndex={setCurrentIndex}/>
            <DelayTouring changeInterval={setDuration} interval={duration}/>
          </Segment>}
      </div>
    </WithQueryStatus>
  );
};

export default withStore(VideoTouring);
