import React, { useMemo, useState } from "react";
import { Icon, Loader, Popup } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { statusCancelsRequests, statusPauseRequest } from "@core/actions/videoDelivery";
import { useStore } from "@core/store";
import { AodRequest, AodRequestInput, AodRequestStatus, AodRequestType, useAodCancelMutation, useAodPauseMutation, useAodResumeMutation, useAodSubmitMutation } from "@generated/graphql";
import { Log, __ } from "@solid/libs";
import VDMConfirm from "components/VideoDeliveryManager/VDMConfirm";

import "./style.css";


type VDMCellActionsProps =  {
  rowData: AodRequest;
  updateRequests?: (requests: AodRequest[]) => void;
  onDownloadsOpen?: (rowData: AodRequest) => void;
};

const VDMCellActions = ({ rowData, onDownloadsOpen, updateRequests }: VDMCellActionsProps) => {
  const [cancelRequest, { loading: cancelLoading }] = useAodCancelMutation();
  const [resumeRequest, { loading: resumeLoading }] = useAodResumeMutation();
  const [pauseRequest, { loading: pauseLoading }] = useAodPauseMutation();
  const [resubmitRequest, { loading: resubmitLoading }] = useAodSubmitMutation();
  const [warningOpen, setWarningOpen] = useState<boolean>(false);
  const [cancelOpen, setCancelOpen] = useState<boolean>(false);
  const { store: { session: { isAdmin, info } } } = useStore();
  const { status, files, note, userId } = rowData;


  const isManaged = useMemo(() => {
    if (isAdmin) return true;
    if (info) {
      const { user: {id} } = info;
      return id === userId;
    }

    return false;
  }, [info, userId, isAdmin]);

  async function cancel(rowData: AodRequest) {
    try {
      const result = await cancelRequest({ variables: { requestIds: [rowData.id] } });
      if (result.data?.aodCancel) {
        const updatedRequest: AodRequest = { ...rowData, ...{ status: AodRequestStatus.Canceled } };
        updateRequests && updateRequests([updatedRequest]);
      }
    }
    catch (e: any) {
      Log.error(e);
    }
  }

  async function resume(rowData: AodRequest) {
    try {
      const result = await resumeRequest({
        variables:
        { requestId: rowData.id }
      });
      const { data } = result;
      if (data?.aodResume) {
        const updatedRequest: AodRequest = { ...rowData, ...{ status: AodRequestStatus.Queued } };
        updateRequests && updateRequests([updatedRequest]);
      }
    }
    catch (e: any) {
      Log.error(e);
    }
  }

  async function pause(rowData: AodRequest) {
    try {
      const result = await pauseRequest({
        variables:
          { requestId: rowData.id }
      });
      const { data } = result;
      if (data?.aodPause) {
        const updatedRequest: AodRequest = { ...rowData, ...{ status: AodRequestStatus.Paused } };
        updateRequests && updateRequests([updatedRequest]);
      }
    }
    catch (e: any) {
      Log.error(e);
    }
  }

  async function resubmit(rowData: AodRequest) {
    const input: AodRequestInput = {
      obj: rowData.deviceId,
      startTime: rowData.startTime,
      endTime: rowData.endTime,
      isPrivate: typeof rowData.type === "undefined" || rowData.type === AodRequestType.Auto ? true : false
    };
    try {
      const result = await resubmitRequest({ variables: { input } });
      const { data } = result;
      if (data?.aodSubmit) {
        updateRequests && updateRequests([data.aodSubmit[0]]);
      }
    }
    catch (e: any) {
      Log.error(e);
    }
  }

  const formatWarning = (note: string) => {
    return <pre className="VDMCellContent-WarningMessage">{note}</pre>;
  };

  return (
    <>
      {(cancelLoading || resumeLoading || pauseLoading || resubmitLoading) ?
        <div className="VDMCellContent-ActionLoading">
          <Loader active inline inverted size="small" />
        </div>
        :
        <div className="VDMCellContent-Action">
          {note &&
            <Popup
              on="click"
              hideOnScroll
              position="top center"
              open={warningOpen}
              onClose={() => setWarningOpen(false)}
              content={formatWarning(note)}
              trigger={
                <Popup
                  content={formatWarning(note)}
                  position="top center"
                  disabled={warningOpen}
                  hideOnScroll
                  trigger={
                    <Icon
                      name="exclamation"
                      className="VDMCellContent-ActionIcon"
                      onClick={(e: React.MouseEvent) => {
                        e.preventDefault();
                        setWarningOpen(!warningOpen);
                      }} />
                  }
                />
              }
            />}

          {status === AodRequestStatus.Failed && isManaged &&
            <Popup
              content={__("Resubmit request")}
              position="top center"
              hideOnScroll
              trigger={
                <Icon
                  name="repeat"
                  className="VDMCellContent-ActionIcon"
                  onClick={(e: React.MouseEvent) => {
                    e.preventDefault();
                    resubmit(rowData);
                  }} />
              }
              />}

          {status === AodRequestStatus.Completed && files !== null &&
          <Popup
            content={__("Download")}
            position="top center"
            hideOnScroll
            trigger={
              <FontAwesomeIcon
                icon="cloud-download-alt"
                className="VDMCellContent-ActionIcon"
                onClick={(e: React.MouseEvent) => {
                  e.preventDefault();
                  onDownloadsOpen && onDownloadsOpen(rowData);
                }} />
            }
          />}

          {statusPauseRequest.indexOf(status) !== -1 && isManaged &&
          <Popup
            content={__("Pause request")}
            position="top center"
            hideOnScroll
            trigger={
              <Icon
                name="pause"
                className="VDMCellContent-ActionIcon"
                onClick={(e: React.MouseEvent) => {
                  e.preventDefault();
                  pause(rowData);
                }} />
            }
          />}

          {status === AodRequestStatus.Paused && isManaged &&
          <Popup
            content={__("Resume request")}
            position="top center"
            hideOnScroll
            trigger={
              <Icon
                name="play"
                className="VDMCellContent-ActionIcon"
                onClick={(e: React.MouseEvent) => {
                  e.preventDefault();
                  resume(rowData);
                }} />
            }
          />}

          {statusCancelsRequests.indexOf(status) !== -1 && isManaged &&
          <Popup
            on="click"
            hideOnScroll
            position="top center"
            open={cancelOpen}
            onClose={(e: React.SyntheticEvent) => {
              e.preventDefault();
              setCancelOpen(false);
            }}
            content={
              <VDMConfirm
                title={__("Remove request?")}
                confirmButtonText={__("Confirm")}
                onCancel={() => setCancelOpen(false)}
                onConfirm={() => {
                  setCancelOpen(false);
                  cancel(rowData);
                }}
              />
            }
            trigger={
              <Popup
                content={__("Remove request")}
                position="top center"
                disabled={cancelOpen}
                hideOnScroll
                trigger={
                  <Icon
                    name="trash"
                    className="VDMCellContent-ActionIcon"
                    onClick={(e: React.MouseEvent) => {
                      e.preventDefault();
                      setCancelOpen(!cancelOpen);
                    }} />
                } />
            }
          />}
        </div>}
    </>
  );
};

export default VDMCellActions;
