import React, { useState, useEffect } from "react";
import axios from "axios";
import { Item, Segment } from "semantic-ui-react";
import { useSystemParametersQuery } from "@generated/graphql";
import useIsMounted from "@core/useIsMounted";

import "./style.css";

type PlaylistItem  = {
  etag: string,
  snippet: {
    title: string,
    description: string,
    resourceId: {
      "kind": string,
      "videoId": string,
    }
  }
};

const lengthLimit = 20;

const TrainingVideos = () => {
  const { data, loading: systemParamLoading } = useSystemParametersQuery();
  const [loading, setLoading] = useState<boolean>(true);
  const [videoList, setVideoList] = useState<PlaylistItem[]>([]);
  const isMounted = useIsMounted();

  useEffect(() => {
    if (!data?.systemParameters) return;

    const GOOGLE_YOUTUBE_API_KEY = data.systemParameters.googleYouTubeApiKey;
    const GOOGLE_YOUTUBE_PLAYLIST_ID = data.systemParameters.googleYouTubePlaylistId;
    if (!GOOGLE_YOUTUBE_API_KEY || !GOOGLE_YOUTUBE_PLAYLIST_ID) return;

    const request = `https://www.googleapis.com/youtube/v3/playlistItems?key=${GOOGLE_YOUTUBE_API_KEY}&playlistId=${GOOGLE_YOUTUBE_PLAYLIST_ID}&part=snippet,id&order=date&maxResults=${lengthLimit}`;
    axios.get(request)
      .then(({ data }) => { return data.items; })
      .then(items => {
        if (items.length > 0) {
          if (!isMounted()) {
            return;
          }

          setLoading(false);
          setVideoList(items);
        }
      });

  }, [data]);

  return (
    <Segment className="TrainingVideos" loading={systemParamLoading}>
      <Item.Group className="TrainingVideos-Content">
        {!loading &&
          videoList.map(video =>
            video.snippet.resourceId.videoId &&
            <Item key={video.snippet.resourceId.videoId} className="TrainingVideos-Item">
              <div className="TrainingVideos-Item-Video">
                <iframe src={`https://www.youtube.com/embed/${video.snippet.resourceId.videoId}`} frameBorder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowFullScreen />
              </div>
              <Item.Content>
                <Item.Header as='a'>{video.snippet.title}</Item.Header>
                <Item.Description>
                  <p>{video.snippet.description}</p>
                </Item.Description>
              </Item.Content>
            </Item>
          )
        }
      </Item.Group>
    </Segment>
  );
};

export default TrainingVideos;


