import React, { useState, useRef, useEffect, useCallback, useMemo } from "react";
import { Button, Icon, Modal, Header, Form, Dropdown, Input, DropdownItemProps, Message } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import rome from "@bevacqua/rome";
import moment from "moment";
import { formatDateTime } from "@core/utils";
import { useStore } from "@core/store";
import {__} from "@solid/libs/i18n";

import "./style.css";

export type ExportDevice = {
  id: string;
  name?: string;
};

enum DurationUnit {
  Minute = "Minute",
  Hour = "Hour"
}

enum CalendarType {
  StartFrom = "startFrom",
  EndTo = "endTo"
}

type VideoExportDialogProps = {
  title: string;
  okButtonText: string;
  devices: ExportDevice[];
  selectedDevice?: ExportDevice;
  open: boolean;
  minTime: number;
  onGetTime: () => number;
  onOk: (deviceId: string, start: number, end: number) => void;
  onClose: () => void;
  onTimeChanged?: (time: number) => void;
};

const defaultDuration = 1; //min
const minutesOptionValues: number[] = [0, 5, 10, 15, 30, 45, 60];

const VideoExportDialog = ({
  title,
  okButtonText,
  devices,
  selectedDevice,
  open,
  minTime,
  onGetTime,
  onOk,
  onClose,
  onTimeChanged }: VideoExportDialogProps) => {
  const { store: { session: { info } } } = useStore();
  const [deviceId, setDeviceId] = useState("");
  const [startFrom, setStartFrom] = useState<number>(Date.now());
  const [endTo, setEndTo] = useState<number>(Date.now());
  const calendarStartPlaceholderRef = useRef<HTMLDivElement>(null);
  const calendarStartElemRef = useRef<HTMLDivElement>();
  const calendarStartRef = useRef<any>();
  const calendarEndPlaceholderRef = useRef<HTMLDivElement>(null);
  const calendarEndElemRef = useRef<HTMLDivElement>();
  const calendarEndRef = useRef<any>();
  const durationRef = useRef<number>(defaultDuration * 1000 * 60);
  const [isCalendarStartOpen, setIsCalendarStartOpen] = useState(false);
  const [isCalendarEndOpen, setIsCalendarEndOpen] = useState(false);
  const [hours, setHours] = useState<number>(0);
  const [minutes, setMinutes] = useState<number>(1);
  const [durationError, setDurationError] = useState("");
  const maxDuration = useMemo(() => {
    if (info) {
      const { realm: { maxDownloadLength } } = info;
      if (maxDownloadLength) {
        return maxDownloadLength * 60 * 1000;
      }
    }
    return undefined;
  }, [info]);

  const durationHourOptions: DropdownItemProps[] = useMemo(() => {
    if (!maxDuration || maxDuration <= 3600 * 1000) {
      return [];
    }

    const maxDurationInHour = maxDuration / 1000 / 3600;
    const hourOptions: DropdownItemProps[] = [{ key: "hour_0", value: 0, text: 0 }];

    for (let i = 1; i <= maxDurationInHour; i++) {
      const optionItem: DropdownItemProps = { key: `hour_${i}`, value: i, text: i };
      hourOptions.push(optionItem);
    }

    return hourOptions;
  }, [maxDuration]);

  const durationMinuteOptions: DropdownItemProps[] = useMemo(() => {
    if (!maxDuration || maxDuration <= 60 * 1000) {
      return [];
    }

    const maxDurationInMin = maxDuration / 1000 / 60 > 60 ? 60 : maxDuration / 1000 / 60;
    const minOptions: DropdownItemProps[] = [];

    for (let i = 0; i <= minutesOptionValues.length; i++) {
      if (minutesOptionValues[i] <= maxDurationInMin) {
        const optionItem: DropdownItemProps = { key: `min_${minutesOptionValues[i]}`, value: minutesOptionValues[i], text: minutesOptionValues[i] };
        minOptions.push(optionItem);
      }
    }

    return minOptions;
  }, [maxDuration]);

  useEffect(() => {
    if (selectedDevice) {
      setDeviceId(selectedDevice.id);
    }
  }, [selectedDevice]);

  useEffect(() => {
    setDeviceId(id => {
      if (devices.length === 0) {
        return "";
      }
      if (!id || !devices.some(dev => dev.id === id)) {
        return devices[0].id;
      }
      return id;
    });
  }, [devices]);

  useEffect(() => {
    const start = moment(startFrom);
    if (start.isValid()) {
      calendarStartRef.current?.setValue(start.toDate());
    }

    validateDurationValue();
  }, [startFrom]);

  useEffect(() => {
    const end = moment(endTo);
    if (end.isValid()) {
      calendarEndRef.current?.setValue(end.toDate());
    }

    validateDurationValue();
  }, [endTo]);

  useEffect(() => {
    durationRef.current = getDurationInMs(minutes, DurationUnit.Minute) + getDurationInMs(hours, DurationUnit.Hour);
    setEndTo(startFrom + durationRef.current);
  }, [startFrom, minutes, hours]);

  const createCalendar = () => {
    const startPlaceholder = calendarStartPlaceholderRef.current;
    const endPlaceHolder = calendarEndPlaceholderRef.current;
    if (!startPlaceholder || !endPlaceHolder) {
      return;
    }
    destroyCalendar();

    const elemStart = document.createElement("div");
    elemStart.className = "VideoExportDialog-Calendar";
    const elemEnd = document.createElement("div");
    elemEnd.className = "VideoExportDialog-Calendar";

    startPlaceholder.appendChild(elemStart);
    endPlaceHolder.appendChild(elemEnd);

    calendarStartElemRef.current = elemStart;
    calendarEndElemRef.current = elemEnd;

    calendarStartRef.current = rome(elemStart, {
      min: moment(minTime),
      max: moment(Date.now())
    });
    calendarEndRef.current = rome(elemEnd, {
      min: moment(minTime),
      max: moment(Date.now())
    });

    if (calendarStartRef.current) {
      calendarStartRef.current.hide();

      calendarStartRef.current.on("data", (timeStr: string) => {
        const value = moment(timeStr);
        const time = moment(value).valueOf();

        setEndTo(time + durationRef.current);
        hideCalendar(CalendarType.StartFrom);
        setStartFrom(time);
      });
    }
    if (calendarEndRef.current) {
      calendarEndRef.current.hide();

      calendarEndRef.current.on("data", (timeStr: string) => {
        const value = moment(timeStr);
        const end = moment(value);
        const start = calendarStartRef.current?.getMoment();

        hideCalendar(CalendarType.EndTo);

        if (start && start.isValid() && end && end.isValid()) {
          const duration = moment.duration(end.diff(start)).abs();

          setHours(duration.hours());
          setMinutes(duration.minutes());
        }
      });
    }
  };

  const destroyCalendar = () => {
    hideCalendar();
    if (calendarStartRef.current) {
      calendarStartRef.current.destroy();
      calendarStartRef.current = undefined;
    }
    if (calendarStartElemRef.current) {
      calendarStartElemRef.current.remove();
      calendarStartElemRef.current = undefined;
    }
    if (calendarEndRef.current) {
      calendarEndRef.current.destroy();
      calendarEndRef.current = undefined;
    }
    if (calendarEndElemRef.current) {
      calendarEndElemRef.current.remove();
      calendarEndElemRef.current = undefined;
    }
  };

  const showCalendar = (calendarType: CalendarType) => {
    switch (calendarType) {
      case CalendarType.StartFrom:
        calendarStartRef.current?.show();
        setIsCalendarStartOpen(true);
        break;
      case CalendarType.EndTo:
        calendarEndRef.current?.show();
        setIsCalendarEndOpen(true);
        break;
      default:
        break;
    }
  };

  const hideCalendar = (calendarType?: CalendarType) => {
    if (!calendarType) {
      calendarStartRef.current?.hide();
      setIsCalendarStartOpen(false);
      calendarEndRef.current?.hide();
      setIsCalendarEndOpen(false);
    }

    switch (calendarType) {
      case CalendarType.StartFrom:
        calendarStartRef.current?.hide();
        setIsCalendarStartOpen(false);
        break;
      case CalendarType.EndTo:
        calendarEndRef.current?.hide();
        setIsCalendarEndOpen(false);
        break;
      default:
        break;
    }
  };

  const onDialogMount = useCallback(() => {
    createCalendar();

    const time = onGetTime();
    let roundedTime: number = time - time % (60 * 1000);

    const date = new Date(time);
    if (date.getUTCSeconds() > 50) {
      roundedTime = time + ((60 - date.getUTCSeconds()) * 1000);
    }

    setStartFrom(roundedTime);
    setEndTo(roundedTime + defaultDuration * 1000 * 60);
  }, [onGetTime, createCalendar]);

  const onDialogClose = () => {
    destroyCalendar();
    onClose();
  };

  const onDurationHandleChange = useCallback((value: number, unit: DurationUnit) => {
    switch (unit) {
      case DurationUnit.Minute:
        if (value >= 60) {
          const hours = Math.floor(value / 60);
          setHours(prevHours => prevHours + hours);
          setMinutes(value - (hours * 60));
          break;
        }
        setMinutes(value);
        break;
      case DurationUnit.Hour:
        setHours(value);
        break;
    }
  }, [startFrom]);

  const validateDurationValue = useCallback((): boolean => {
    if (!maxDuration || maxDuration === 0) {
      setDurationError(__("You are not allowed to download videos"));
      return false;
    }

    const duration = durationRef.current;
    if (duration === 0) {
      setDurationError(__("Invalid value"));
      return false;
    }
    if (duration > maxDuration || (endTo - startFrom) > maxDuration) {
      setDurationError(__("Max duration value is {{value}} minutes", { value: maxDuration / 1000 / 60 }));
      return false;
    }

    setDurationError("");
    return true;
  }, [startFrom, maxDuration, endTo]);

  const onOkClick = useCallback(() => {
    const m = moment(startFrom);
    if (!m.isValid()) {
      return;
    }

    const duration = durationRef.current;
    const start = m.valueOf();
    const end = start + duration;
    if (end > Date.now()) {
      return;
    }
    onClose();
    onTimeChanged && onTimeChanged(startFrom);
    onOk(deviceId, start, end);
  }, [deviceId, startFrom]);

  const getDevices = (devices: ExportDevice[]): ExportDevice[] => {
    const result: ExportDevice[] = [];
    for (const device of devices) {
      if (!result.some(dev => dev.id === device.id)) {
        result.push(device);
      }
    }
    return result;
  };

  function getDurationInMs(value: string | number, unit: DurationUnit): number {
    let duration: number = formatInputValue(value);

    switch (unit) {
      case DurationUnit.Minute:
        duration = duration * 60 * 1000;
        break;
      case DurationUnit.Hour:
        duration = duration * 3600 * 1000;
    }

    return duration;
  }

  function formatInputValue(value?: boolean | string | number | true | (string | number | boolean)[]): number {
    const num: number = isNaN(Number(value)) ? 0 : Number(value);
    return Math.abs(num);
  }

  return (
    <Modal className="VideoExportDialog" open={open} onMount={onDialogMount} onClose={onDialogClose}>
      <Header content={title}/>
      <Modal.Content>
        { !!durationError && <Message error>{durationError}</Message> }
        <Form className="VideoExportDialog-Form" onSubmit={(e) => { e.preventDefault(); }}>
          <Form.Group inline>
            <Form.Field inline className="VideoExportDialog-StartFrom" >
              <label>{__("Start from")}</label>
              <div className="VideoExportDialog-CalendarWrapper">
                <Input
                  readOnly
                  autoFocus
                  labelPosition="right"
                  placeholder={__("Start from")}
                  value={formatDateTime(new Date(startFrom))}
                  label={
                    <>
                      { !isCalendarStartOpen ?
                        <Button icon onClick={() => showCalendar(CalendarType.StartFrom)}>
                          <FontAwesomeIcon icon="calendar-alt"/>
                        </Button> :
                        <Button icon onClick={() => hideCalendar(CalendarType.StartFrom)}>
                          <FontAwesomeIcon icon="times"/>
                        </Button> }
                    </>
                  }
                />
                <div ref={calendarStartPlaceholderRef} className="VideoExportDialog-CalendarPlaceholder"/>
              </div>
            </Form.Field>
            <Form.Field inline className="VideoExportDialog-EndTo">
              <label>{__("End to")}</label>
              <div className="VideoExportDialog-CalendarWrapper">
                <Input
                  readOnly
                  labelPosition="right"
                  placeholder={__("End to")}
                  value={formatDateTime(new Date(endTo))}
                  label={
                    <>
                      { !isCalendarEndOpen ?
                        <Button icon onClick={() => showCalendar(CalendarType.EndTo)}>
                          <FontAwesomeIcon icon="calendar-alt"/>
                        </Button> :
                        <Button icon onClick={() => hideCalendar(CalendarType.EndTo)}>
                          <FontAwesomeIcon icon="times"/>
                        </Button> }
                    </>
                  }
                />
                <div ref={calendarEndPlaceholderRef} className="VideoExportDialog-CalendarPlaceholder"/>
              </div>
            </Form.Field>
          </Form.Group>

          <Form.Group inline>
            <Form.Field
              inline
              className="VideoExportDialog-Duration"
              error={!!durationError}
            >
              <label>{__("Hour")}</label>
              <Input
                type="number"
                placeholder={__("hour")}
                labelPosition="right"
                value={hours}
                onInput={(e: React.FormEvent<HTMLInputElement>) => onDurationHandleChange(formatInputValue(e.currentTarget.value), DurationUnit.Hour)}
                label={
                  <Dropdown
                    scrolling
                    trigger={<div/>}
                    openOnFocus={false}
                    options={durationHourOptions}
                    value={hours}
                    onChange={(e, data) => onDurationHandleChange(formatInputValue(data.value), DurationUnit.Hour)}
                  />
                }
              />
            </Form.Field>
            <Form.Field
              inline
              className="VideoExportDialog-Duration"
              error={!!durationError}
            >
              <label>{__("Minute")}</label>
              <Input
                type="number"
                placeholder={__("minute")}
                labelPosition="right"
                value={minutes}
                onInput={(e: React.FormEvent<HTMLInputElement>) => onDurationHandleChange(formatInputValue(e.currentTarget.value), DurationUnit.Minute)}
                label={
                  <Dropdown
                    scrolling
                    trigger={<div/>}
                    openOnFocus={false}
                    options={durationMinuteOptions}
                    value={minutes}
                    onChange={(e, data) => onDurationHandleChange(formatInputValue(data.value), DurationUnit.Minute)}
                  />
                }
              />
            </Form.Field>
          </Form.Group>
          <Form.Field>
            <label>{__("Device")}</label>
          </Form.Field>
          <>
            { getDevices(devices).map(({ id, name }) => (
              <Form.Radio key={id} name="device" label={name} value={id} checked={id === deviceId} onChange={() => setDeviceId(id)}/>
            ))}
          </>
        </Form>
      </Modal.Content>
      <Modal.Actions>
        <Button positive onClick={onOkClick} disabled={!!durationError}>
          <Icon name="checkmark"/>{okButtonText}
        </Button>
        <Button negative onClick={onDialogClose}><Icon name="cancel"/>{__("Cancel")}</Button>
      </Modal.Actions>
    </Modal>
  );
};

export default VideoExportDialog;
