import React, {useCallback, useEffect, useMemo, useRef, useState} from "react";
import {Button, Dropdown, Icon, Popup} from "semantic-ui-react";
import {FontAwesomeIcon} from "@fortawesome/react-fontawesome";
import moment from "moment";
import rome from "@bevacqua/rome";
import { WidgetId } from "@generated/graphql";
import {WidgetProps} from "components/Widgets";
import { PlayerCommand } from "@core/playerTypes";
import {DeviceWithState, TimelineControlsPosition} from "components/Timeline";
import StepBackForward from "@core/actions/StepBackForward";
import { UUID } from "@solid/types";
import VideoExportDialog from "components/Timeline/VideoExportDialog";
import {__} from "@solid/libs/i18n";
import {MSEMediaPlayerConnection} from "@solid/types/player";
import { isElectron } from "@solid/libs/utils";
import { useAccessability } from "@core/store/actions/accessability";
import { playerCommand } from "../timelineUtils";
import { setTimelineStore } from "../timelineStore";

import "./style.css";

export type TimelineControlsProps = WidgetProps & {
  position: TimelineControlsPosition;
  devices: DeviceWithState[];
  allDevices: DeviceWithState[];
  archiveStartTime: number;
  step: StepBackForward;
  selectedDevice?: DeviceWithState;
  onTimeChanged: (time: number) => void;
  onGetTime: () => number;
  onSetSpeed: (speed: number) => void;
};

const speedUp = 10;
const advancedSpeedList: number[] = [-30, -10, -5, -2, -1, 0.125, 0.25, 0.5, 1, 2, 5, 10, 30];
const jumpStep = 30; // sec

const tooSmallSpeed = -1000;
// const tooLargeSpeed = 1000;

const TimelineControls = ({
  position,
  devices,
  allDevices,
  archiveStartTime,
  step,
  selectedDevice,
  onTimeChanged,
  onGetTime,
  onSetSpeed,
  widgetEvent
}: TimelineControlsProps) => {
  const { config: { limitedSpeedList } } = useAccessability({ widgetId: WidgetId.Timeline });
  const calendarPlaceholderRef = useRef<HTMLDivElement>(null);
  const calendarElemRef = useRef<HTMLDivElement>();
  const calendarRef = useRef<any>();
  const [isCalendarOpen, setIsCalendarOpen] = useState(false);
  const [isAODDialogOpen, setIsAODDialogOpen] = useState(false);
  const [connectionType, setConnectionType] = useState<MSEMediaPlayerConnection>(MSEMediaPlayerConnection.AUTO);
  const [timelineSpeed, setTimelineSpeed] = useState<number>(1);
  const pressState = useRef<{
    leftMouseButtonDown: boolean,
    leftMouseButtonIsHeldDown: boolean,
    isPlayBeforeLeftMouseButtonIsHeldDown: boolean,
    speedBeforeLeftMouseButtonIsHeldDown: number
  }>({leftMouseButtonDown: false, leftMouseButtonIsHeldDown: false, isPlayBeforeLeftMouseButtonIsHeldDown: false, speedBeforeLeftMouseButtonIsHeldDown: 1});

  const speedList: number[] = useMemo(() => {
    let speedList = advancedSpeedList;
    if (limitedSpeedList) {
      speedList = [-5, -2, -1, 0.125, 0.25, 0.5, 1, 2, 5];
    }

    return speedList;
  }, [limitedSpeedList]);

  useEffect(() => {
    return function cleanup() {
      hideCalendar();
    };
  }, []);

  function showCalendar(): void {
    const placeholder = calendarPlaceholderRef.current;
    if (!placeholder) {
      return;
    }
    hideCalendar();

    const elem = document.createElement("div");
    elem.className = "TimelineControls-Calendar";
    document.body.appendChild(elem);
    calendarElemRef.current = elem;

    calendarRef.current = rome(elem, {
      min: moment(archiveStartTime),
      max: moment(Date.now())
    });

    if (calendarRef.current) {
      positionCalendar();
      calendarRef.current.on("show", () => positionCalendar());

      calendarRef.current.setValue(new Date(onGetTime()));

      calendarRef.current.on("data", (timeStr: string) => {
        const value = moment(timeStr);
        const time = moment(value).valueOf();
        hideCalendar();
        onTimeChanged(time);
      });
    }

    setIsCalendarOpen(true);
  }

  function hideCalendar(): void {
    if (calendarRef.current) {
      calendarRef.current.hide();
      calendarRef.current.destroy();
      calendarRef.current = undefined;
    }
    if (calendarElemRef.current) {
      calendarElemRef.current.remove();
      calendarElemRef.current = undefined;
    }
    setIsCalendarOpen(false);
  }

  function positionCalendar(): void {
    const placeholder = calendarPlaceholderRef.current;
    const elem = calendarElemRef.current;
    const root = elem?.querySelector(".rd-container");
    if (!placeholder || !elem || !root) {
      return;
    }
    const placeRect = placeholder.getBoundingClientRect();
    const rootRect = root.getBoundingClientRect();
    elem.style.left = (placeRect.left - rootRect.width) + "px";
    elem.style.top = (placeRect.top - rootRect.height) + "px";
  }

  const addDownloadJob = (deviceId: UUID, start: number, end: number) => {
    if (!widgetEvent) {
      return;
    }
    const duration = Math.round((end - start) / 1000);
    playerCommand(widgetEvent, devices, PlayerCommand.GetAOD, { deviceId, start: Math.round(start), duration });
  };

  const preferredConnectionType = () => {
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    let preferredConnectionType: MSEMediaPlayerConnection = connectionType;

    if (connectionType === MSEMediaPlayerConnection.AUTO) {
      preferredConnectionType = MSEMediaPlayerConnection.CLOUD;
    } else
    if (connectionType === MSEMediaPlayerConnection.CLOUD) {
      preferredConnectionType = MSEMediaPlayerConnection.LOCAL;
    } else
    if (connectionType === MSEMediaPlayerConnection.LOCAL) {
      preferredConnectionType = MSEMediaPlayerConnection.AUTO;
    }

    setTimelineStore({ connectionType: preferredConnectionType });
    setConnectionType(preferredConnectionType);
    playerCommand(widgetEvent, devices, PlayerCommand.PreferredConnectionType, {preferredConnectionType});
  };

  const setSpeed = (speed: number) => {
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    playerCommand(widgetEvent, devices, PlayerCommand.SetSpeed, { speed });
    onSetSpeed(speed);
  };

  // @ts-ignore
  const stepBackForward = async (forward: boolean) => {
    const time = onGetTime();
    const newTime = forward ? await step.stepForward(time) : await step.stepBackward(time);
    if (time !== newTime) {
      onTimeChanged(newTime);
    }
  };


  const disabled = devices.length === 0;
  const paused = devices.every(dev => dev.stopped);
  const speed = devices.reduce((acc, dev) => Math.max(acc, dev.speed ?? 1), tooSmallSpeed);

  const onSpeedUp = (speed: number) => {
    pressState.current.leftMouseButtonDown = true;

    setTimeout(() => {
      if (!widgetEvent) {
        console.error("!widgetEvent");
        return;
      }

      if (!pressState.current.leftMouseButtonDown) {
        return;
      }

      pressState.current.isPlayBeforeLeftMouseButtonIsHeldDown = true;
      pressState.current.speedBeforeLeftMouseButtonIsHeldDown = timelineSpeed;
      pressState.current.leftMouseButtonIsHeldDown = true;

      setTimelineSpeed(speed);
      playerCommand(widgetEvent, devices, PlayerCommand.SetSpeed, {speed});
    }, 1000);
  };

  const onBackwardMouseDown = useCallback(() => {
    console.log("onBackwardMouseDown", pressState.current, timelineSpeed);
    onSpeedUp(-speedUp);
  }, [devices]);

  const onSpeedRestore = useCallback(() => {
    console.log("onBackwardMouseUp", pressState.current, timelineSpeed);
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    pressState.current.leftMouseButtonDown = false;

    if (pressState.current.leftMouseButtonIsHeldDown) {
      if (!pressState.current.isPlayBeforeLeftMouseButtonIsHeldDown) {
        playerCommand(widgetEvent, devices, PlayerCommand.Pause);
      }
      const speed = pressState.current.speedBeforeLeftMouseButtonIsHeldDown;
      setTimelineSpeed(speed);
      playerCommand(widgetEvent, devices, PlayerCommand.SetSpeed, {speed});
    }
  }, [devices]);

  const onBackwardClick = useCallback(() => {
    console.log("onBackwardClick", pressState.current);

    if (pressState.current.leftMouseButtonIsHeldDown) {
      pressState.current.leftMouseButtonIsHeldDown = false;
      return;
    }

    const newSpeed = -Math.abs(timelineSpeed);
    if (newSpeed !== speed) {
      setSpeed(newSpeed);
      setTimelineSpeed(newSpeed);
    }
  }, [devices]);

  const onForwardMouseDown = useCallback(() => {
    console.log("onForwardMouseDown", pressState.current, timelineSpeed);

    onSpeedUp(speedUp);
  }, [devices]);

  const onForwardClick = useCallback(() => {
    if (pressState.current.leftMouseButtonIsHeldDown) {
      pressState.current.leftMouseButtonIsHeldDown = false;
      return;
    }

    const newSpeed = Math.abs(timelineSpeed);
    if (newSpeed !== speed) {
      setSpeed(newSpeed);
      setTimelineSpeed(newSpeed);
    }
  }, [devices, timelineSpeed]);

  const onBackwardStepClick = useCallback(() => {
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    playerCommand(widgetEvent, devices, PlayerCommand.SetTime, {time: new Date(onGetTime() - jumpStep * 1000), byUser: true});
  }, [devices]);

  const onForwardStepClick = useCallback(() => {
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    playerCommand(widgetEvent, devices, PlayerCommand.SetTime, {time: new Date(onGetTime() + jumpStep * 1000), byUser: true});
  }, [devices]);

  const onPlayPauseClick = useCallback(() => {
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    setTimelineStore({ isPlay: paused });

    if (paused) {
      playerCommand(widgetEvent, devices, PlayerCommand.Play, { time: new Date(onGetTime()), speed: timelineSpeed });
    } else {
      playerCommand(widgetEvent, devices, PlayerCommand.Pause, { byUser: true });
    }
  }, [paused, devices, widgetEvent, timelineSpeed]);

  const onSpeedChange = useCallback((e, data) => {
    if (!widgetEvent) {
      console.error("!widgetEvent");
      return;
    }

    const speed = data.value;
    setTimelineSpeed(speed);
    playerCommand(widgetEvent, devices, PlayerCommand.SetSpeed, {speed});
  }, [devices, widgetEvent]);

  return (
    <div className="TimelineControls">
      <div className="TimelineControls-PlayButtons">
        {isElectron() &&
        <>
          <Popup
            trigger={
              <div className="TimelineControls-Container" onMouseDown={onBackwardMouseDown} onMouseUp={onSpeedRestore}>
                <Button icon className="TimelineControls-backward medium" disabled={disabled || paused} onClick={onBackwardClick}
                >
                  <Icon name="backward"/>
                </Button>
              </div>
            }
            content={__("Backward")}
          />
          <Popup
            trigger={
              <div className="TimelineControls-Container">
                <Button icon className="TimelineControls-jump-backward medium" disabled={disabled || paused} onClick={onBackwardStepClick}>
                  <Icon name="arrow left"/>
                </Button>
              </div>
            }
            content={__("Jump {{number}}sec Backward", {number: jumpStep})}
          />
        </>}

        <Popup
          trigger={
            <div className="TimelineControls-Container">
              <Button icon className={`TimelineControls-${paused ? "play" : "pause"} large`} disabled={disabled} onClick={onPlayPauseClick}>
                <Icon name={paused ? "play" : "pause"}/>
              </Button>
            </div>
            }
          content={paused ? __("Play") : __("Pause")}
          />

        {isElectron() &&
        <>
          <Popup
            trigger={
              <div className="TimelineControls-Container">
                <Dropdown
                  className="TimelineControls-speedList"
                  value={timelineSpeed}
                  scrolling
                  onChange={onSpeedChange}
                  options={speedList.map((speed) => { return {key: speed, text: `${Math.abs(speed) > 1 ? Math.ceil(speed) : speed}x`, value: speed}; })}
                />
              </div>
            }
            content={__("Speed")}
          />
          <Popup
            trigger={
              <div className="TimelineControls-Container">
                <Button icon className="TimelineControls-jump-forward medium" disabled={disabled || paused} onClick={onForwardStepClick}>
                  <Icon name="arrow right"/>
                </Button>
              </div>
            }
            content={__("Jump {{number}}sec Forward", {number: jumpStep})}
          />
          <Popup
            trigger={
              <div className="TimelineControls-Container" onMouseDown={onForwardMouseDown} onMouseUp={onSpeedRestore}>
                <Button icon className="TimelineControls-forward medium" disabled={disabled || paused} onClick={onForwardClick}>
                  <Icon name="forward"/>
                </Button>
              </div>
            }
            content={__("Forward")}
          />
        </>}
      </div>

      <div className="TimelineControls-ActionButtons">
        <div>
          <Popup
            trigger={
              <Button icon onClick={() => preferredConnectionType()}>
                {connectionType === MSEMediaPlayerConnection.AUTO
                  ? "A"
                  : <FontAwesomeIcon icon={connectionType === MSEMediaPlayerConnection.CLOUD ? "cloud" : "home"}/>}
              </Button>
            }
            content={__("Connection type")}
          />

          <Popup
            trigger={
              <Button icon onClick={() => showCalendar()}>
                <FontAwesomeIcon icon="calendar-alt"/>
              </Button>
            }
            content={__("Calendar")}
          />

          {isCalendarOpen && <Popup
            trigger={
              <Button icon onClick={() => hideCalendar()}>
                <FontAwesomeIcon icon="times"/>
              </Button>
            }
            position="left center"
            content={__("Close Calendar")}
          />}

          <div ref={calendarPlaceholderRef} className="TimelineControls-CalendarPlaceholder" />

          <Popup
            trigger={
              <Button icon disabled={allDevices.length === 0} onClick={() => setIsAODDialogOpen(true)}>
                <FontAwesomeIcon icon="cloud-download-alt"/>
              </Button>
            }
            content={__("Add Download Job")}
          />
        </div>
      </div>

      <VideoExportDialog
        title={__("Add Download Job")}
        okButtonText={__("Add")}
        devices={allDevices}
        selectedDevice={selectedDevice}
        open={isAODDialogOpen}
        minTime={archiveStartTime}
        onGetTime={onGetTime}
        onOk={addDownloadJob}
        onClose={() => setIsAODDialogOpen(false)}
      />
    </div>
  );
};

export default TimelineControls;
