import React, {useCallback, useEffect, useRef, useState} from "react";
import {useApolloClient} from "@apollo/client";
import {Segment} from "semantic-ui-react";
import {ModuleInfo, PlayerProps} from "@core/types";
import {CommonWidgetEvent, CommonWidgetEventArgs, EventClickEventArgs, WidgetEventArgs, WidgetProps} from "components/Widgets";
import TimelineControl from "@solid/timeline/components/TimeLine";
import {TimeLineError, TimeLineEventInfo, TimeLineTrack, TimeLineType, VisTimeLine} from "@solid/timeline/vistimeline";
import TimelineControls, {TimelineControlsProps} from "./TimelineControls";
import {WidgetId, ArchiveBoundaryDocument, ArchiveBoundaryQuery, ArchiveBoundaryQueryVariables, AuditEntriesWithSnapshotsDocument, AuditEntriesWithSnapshotsQuery, AuditEntriesWithSnapshotsQueryVariables, BoundaryFind, CameraWidgetProps, DevicesByIdsDocument, DevicesByIdsQuery, DevicesByIdsQueryVariables} from "@generated/graphql";
import {CameraCellProps} from "components/CellContainer/CameraCell";
import StatusTitle from "components/StatusTitle";
import {withStore, WithStoreProps} from "@core/store";
import {PlayerCommand, PlayerCommandArgs, PlayerEvent, PlayerEventArgs} from "@core/playerTypes";
import StepBackForward, {Event} from "@core/actions/StepBackForward";
import CoverageReader from "./CoverageReader";
import EventReader, {Event as AuditEntry} from "./EventReader";
import {__} from "@solid/libs/i18n";
import {Granularity} from "@solid/types/coverage";
import diff from "microdiff";
import {UUID} from "@solid/types";
import { setTimelineStore } from "./timelineStore";
import { playerCommand } from "./timelineUtils";

import "./style.css";

export enum TimelineControlsPosition {
  Top = "Top",
  Bottom = "Bottom"
}

export enum TimelineEvent {
  GotoTime = "GotoTime"
}

export type TimelineGotoTimeEventArgs = {
  time: Date;
};

type TimelineProps = WidgetProps & WithStoreProps & {
  controlsPosition?: TimelineControlsPosition;
};

type Device = {
  id: string;
  widgetIndex: number;
};

type DeviceState = {
  name?: string;
  playerTime?: number;
  timelineTime?: number;
  outOfSync?: boolean;
  stopped?: boolean;
  speed?: number;
  widgetIndex?: number;
};

export type DeviceWithState = Device & DeviceState;

type Request = {
  start: number;
  end: number;
};

type SelectedEventInterval = {
  deviceId: string;
  startTime: number;
  endTime: number;
};

const Timeline = (props: TimelineProps) => {
  const {
    controlsPosition = TimelineControlsPosition.Bottom,
    viewId,
    widgets,
    cellProps,
    setCellProps,
    widgetEvent,
    store: { workspace: { cameraWidgetProps } },
    index: widgetIndex = 0
  } = props;

  const client = useApolloClient();
  const coverageReaderRef = useRef(new CoverageReader(client));
  const eventReaderRef = useRef(new EventReader(client));
  const [loadingCount, setLoadingCount] = useState(0);
  const [timelineError, setTimelineError] = useState<Error | undefined>();

  const timelineRef = useRef<VisTimeLine>();
  const firstLoadRef = useRef(true);
  const selectedEventIntervalRef = useRef<SelectedEventInterval | undefined>();

  const getDevices = (widgets?: ModuleInfo[], cameraWidgetProps?: CameraWidgetProps[] | null): Device[] => {
    if (!widgets) {
      return [];
    }

    return widgets
      .map((w, index) => ({...w, index}))
      .filter(w => w.widgetId === WidgetId.ArchiveViewer)
      .map(w => {
        let id: string | undefined;
        const storeProps = cameraWidgetProps?.find(
          p => p.viewId === viewId && p.index === widgets.findIndex(wi => wi.id === w.id));
        if (storeProps) {
          id = storeProps.deviceId;
        } else {
          const playerProps = w.props ? w.props as CameraCellProps : undefined;
          id = playerProps?.object?.obj ? (playerProps?.object?.obj as PlayerProps)["data-id"] : undefined;
        }
        return id ? { id, widgetIndex: w.index } : undefined;
      })
      .filter(dev => !!dev)
      .map(dev => dev!);
  };

  const [devices, setDevices] = useState(getDevices(widgets, cameraWidgetProps));
  const [deviceState, setDeviceState] = useState(new Map<number, DeviceState>());

  const defaultArchiveStart = Date.now() - 24 * 60 * 60 * 1000;
  const [archiveStartTime, setArchiveStartTime] = useState(defaultArchiveStart);

  const currentTimeRef = useRef(Date.now());

  const getGroup = (): Device[] => {
    if (devices.length === 0) {
      return [];
    }
    if (devices.length === 1) {
      return devices;
    }

    let group: Device[] = [];
    const timeline = timelineRef.current;
    const index = timeline?.selectedTrack();
    if (typeof index !== "number") {
      return devices.filter((d, i) => !getDeviceState(d)?.outOfSync);
    }

    const device = devices[index];
    if (device && getDeviceState(device)?.outOfSync) {
      group = [devices[index]];
    }
    else {
      group = devices.filter((d, i) => !getDeviceState(d)?.outOfSync);
    }
    return group;
  };

  const [group, setGroup] = useState(getGroup());

  const stepRef = useRef(new StepBackForward());

  const [selectedTrack, setSelectedTrack] = useState(-1);

  const requestsRef = useRef<Request[]>([]);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    widgetEvent?.subscribe(id, onWidgetEvent);

    stepRef.current.setStreamNumbers([0, 1]);
    stepRef.current.setOnGetEvents(onGetEvents);
    stepRef.current.setOnGetBoundary(onGetBoundary);
    stepRef.current.setOnLoading(loading => setLoadingCount(count => loading ? count + 1 : count - 1));

    return function cleanup() {
      widgetEvent?.unsubscribe(id);
    };
  });

  useEffect(() => {
    const newDevices = getDevices(widgets, cameraWidgetProps);
    if (diff(newDevices, devices).length !== 0) {
      setDevices(newDevices);
      const newWidgetIndexes = newDevices.map(dev => dev.widgetIndex);

      setDeviceState(map => {
        const newMap = new Map<number, DeviceState>(map);

        for (const [devIndex, devState] of newMap.entries()) {
          if (devState.widgetIndex && !newWidgetIndexes.includes(devState.widgetIndex)) {
            newMap.delete(devIndex);
            const args: PlayerCommandArgs = { widgetIndices: [devState.widgetIndex] };
            widgetEvent?.publish({ event: PlayerCommand.BackToSync, args });
          }
        }

        const stateWidgetIndexes = Array.from(newMap.values()).map(dev => dev.widgetIndex);
        for (const dev of newDevices) {
          if (!stateWidgetIndexes.includes(dev.widgetIndex)) {
            newMap.set(dev.widgetIndex, { widgetIndex: dev.widgetIndex });
          }
        }

        return newMap;
      });

      setSelectedTrack(value => value < newDevices.length ? value : -1);
    }
  }, [widgets, cameraWidgetProps]);

  useEffect(() => {
    devices.length === 1 && syncTimelineWithLastPlayer();

    stepRef.current.clearCache();
    timelineRef.current?.updateData();
  }, [devices]);

  useEffect(() => {
    updateGroup();
  }, [devices, deviceState]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }

    if (loadingCount > 0 || timelineError) {
      setCellProps({
        title: <StatusTitle title={cellProps?.title} loading={loadingCount > 0} error={timelineError} />
      });
    } else {
      setCellProps({ title: cellProps?.title ?? "" });
    }
  }, [loadingCount, timelineError]);

  function syncTimelineWithLastPlayer() {
    const timeline = timelineRef.current;
    if (!timeline) return;

    const currentDeviceStateKey = Array.from(deviceState.keys())[0];
    const currentDevice = deviceState.get(currentDeviceStateKey);

    if (currentDevice) {
      const args: PlayerCommandArgs = { objs: [devices[0].id], widgetIndices: [devices[0].widgetIndex] };
      widgetEvent?.publish({ event: PlayerCommand.BackToSync, args });

      if (currentDevice.playerTime) {
        const playerTime = currentDevice.playerTime;
        const args: PlayerCommandArgs = { objs: [devices[0].id], widgetIndices: [devices[0].widgetIndex], time: new Date(playerTime) };
        widgetEvent?.publish({ event: PlayerCommand.SetTime, args });
        timeline.setTime(playerTime);

        const newDeviceState = { ...currentDevice, outOfSync: false, timelineTime: playerTime };
        updateDeviceState(currentDeviceStateKey, newDeviceState);
      }
    }
  }

  const updateDeviceState = (index: number, state: DeviceState): void => {
    if (index < 0) {
      return;
    }

    setDeviceState(map => {
      const newMap = new Map<number, DeviceState>(map);
      const prevState = newMap.get(index);
      if (!prevState) {
        newMap.set(index, state);
      } else {
        if (state.playerTime
            && Object.keys(state).length === 1
            && Math.abs((prevState.playerTime ?? 0) - state.playerTime) < 1000
        ) {
          return map;
        }

        const newState = {...prevState, ...state};
        if (diff(prevState, newState).length === 0) {
          return map;
        }

        newMap.set(index, newState);
      }

      return newMap;
    });
  };

  const onTimelineLoad = async (timeline: VisTimeLine): Promise<void> => {
    timelineRef.current = timeline;
    await timelineRef.current.setTime(Date.now() - 10 * 60 * 1000);
    timeline.initTrackThumb();
  };

  const onTimelineGetData = useCallback(async (beginTime: number, endTime: number, setData: (list: TimeLineTrack[]) => void): Promise<void> => {
    if (!timelineRef.current) {
      return;
    }

    if (devices.length === 0) {
      setData([]);
      return;
    }

    const requests = requestsRef.current;
    if (requests.some(r => r.start <= beginTime && r.end >= endTime)) {
      return;
    }

    if (requests.length > 0) {
      beginTime = Math.min(beginTime, requests.reduce((acc, req) => Math.min(acc, req.start), requests[0].start));
      endTime = Math.max(endTime, requests.reduce((acc, req) => Math.max(acc, req.end), requests[0].end));
    }

    const request: Request = { start: beginTime, end: endTime };

    setLoadingCount(count => count + 1);
    setTimelineError(undefined);
    try {
      requests.push(request);

      const maxMinutes = Math.round(timelineRef.current.getWidthInMilliseconds() / (60 * 1000))  * 0.2; // 20% of timeline width
      const beginDataTime = beginTime - maxMinutes * 60 * 1000;
      const endDataTime = endTime + maxMinutes * 60 * 1000;

      let tracks: TimeLineTrack[] = [];
      let minArchiveStart = defaultArchiveStart;

      if (firstLoadRef.current) {
        for (let i = 0; i < devices.length; i++) {
          const device = devices[i];

          const args: CommonWidgetEventArgs = { widgetIndex: device.widgetIndex, name: "" };
          const eventArgs = { event: CommonWidgetEvent.GetName, args };
          widgetEvent?.publish(eventArgs);

          const track: TimeLineTrack = { obj: device.id, name: args.name ?? "", rows: [
            { list: {}, granularity: Granularity.chunk },
            { list: {}, granularity: Granularity.chunk },
            { list: {}, granularity: Granularity.chunk, isEvent: true }
          ]};
          tracks.push(track);

          updateDeviceState(device.widgetIndex, args);
        }

        if (tracks.every(t => !!t.name)) {
          setData(tracks);
        }
      }

      tracks = [];
      const { data: devData } = await client.query<DevicesByIdsQuery, DevicesByIdsQueryVariables>({
        query: DevicesByIdsDocument,
        variables: { ids: devices.map(dev => dev.id) }
      });

      const existingDevices = devices.filter(device => devData.devicesByIds.some(dev => dev.id === device.id));
      if (existingDevices.length < devices.length) {
        onTimelineGetData(beginTime, endTime, setData);
        setDevices(existingDevices);
        return;
      }

      for (let i = 0; i < devices.length; i++) {
        const deviceId = devices[i].id;
        const device = devData.devicesByIds.find(dev => dev.id === deviceId);
        const widgetIndex = devices[i].widgetIndex;
        const deviceKey = getDeviceKeyInState(widgetIndex);

        const track: TimeLineTrack = { obj: deviceId, name: device?.name ?? "", rows: [
          { list: {}, granularity: Granularity.chunk },
          { list: {}, granularity: Granularity.chunk },
          { list: {}, granularity: Granularity.second, isEvent: true }
        ]};
        tracks.push(track);

        typeof deviceKey === "number" && updateDeviceState(deviceKey, { name: device?.name });
      }

      if (firstLoadRef.current) {
        firstLoadRef.current = false;
        setData(tracks);
      }

      const coverageData = await coverageReaderRef.current.getCoverage(new Date(beginDataTime), new Date(endDataTime), devices.map(dev => dev.id));

      for (const coverageTtl of coverageData.coverageTtl) {
        const ttlHours = coverageTtl.ttl;
        let archiveStart = ttlHours > 0 ? Date.now() - ttlHours * 60 * 60 * 1000 : defaultArchiveStart;
        archiveStart = new Date(archiveStart).setMinutes(0, 0, 0);
        minArchiveStart = Math.min(minArchiveStart, archiveStart);
      }
      setArchiveStartTime(minArchiveStart);
      stepRef.current.setArchiveStartTime(minArchiveStart);

      const deviceIndices: { [key: string]: number[] } = {};
      for (let i = 0; i < devices.length; i++) {
        if (!deviceIndices[devices[i].id]) {
          deviceIndices[devices[i].id] = [];
        }
        deviceIndices[devices[i].id].push(i);
      }

      for (const coverage of coverageData.coverage) {
        const indices = deviceIndices[coverage.deviceId];
        if (indices === undefined) {
          continue;
        }

        for (const i of indices) {
          const row = coverage.streamNumber === 0 ? tracks[i].rows[0] : tracks[i].rows[1];
          row.list[coverage.time * 1000] = {
            isChunk: true,
            percentage: 100,
            className: coverage.streamNumber === 0 ? "low-resolution-chunk" : "high-resolution-chunk"
          };
        }
      }

      for (const coverage of coverageData.partialCoverage.filter(item => item.streamNumber === 1)) {
        const indices = deviceIndices[coverage.deviceId];
        if (indices === undefined) {
          continue;
        }

        for (const i of indices) {
          const row = tracks[i].rows[1];
          row.list[coverage.time * 1000] = {
            isPartialChunk: true,
            percentage: 100,
            className: "partial-chunk"
          };
        }
      }

      for (const aod of coverageData.aod) {
        const indices = deviceIndices[aod.deviceId];
        if (indices === undefined) {
          continue;
        }

        for (const i of indices) {
          const row = aod.streamNumber === 0 ? tracks[i].rows[0] : tracks[i].rows[1];
          const startTime = alignTimeToChunk(new Date(aod.startTime));
          const endTime = new Date(aod.endTime);
          endTime.setUTCSeconds(endTime.getUTCSeconds() <= 29 ? 29 : 59, 0);
          const times = splitTime(startTime, endTime, 30);
          for (const time of times) {
            if (!row.list[time.getTime()]) {
              row.list[time.getTime()] = {
                isAOD: true,
                percentage: 100,
                className: "aod-chunk"
              };
            }
          }
        }
      }

      for (const deviceId in coverageData.offlineCoverage) {
        const indices = deviceIndices[deviceId];
        if (indices === undefined) {
          continue;
        }

        const coverage = coverageData.offlineCoverage[deviceId];
        for (const timestamp in coverage) {
          for (const i of indices) {
            const row = tracks[i].rows[1];
            row.list[timestamp] = {
              isOffline: true,
              percentage: 100,
              className: "offline-chunk"
            };
          }
        }
      }

      if (requestsRef.current.some(r => r !== request && r.start <= beginTime && r.end >= endTime)) {
        return;
      }

      const cachedEvents = eventReaderRef.current.getCachedEvents(new Date(beginTime), new Date(endTime), devices.map(d => d.id));
      setEvents(devices, tracks, cachedEvents);
      setData(tracks);

      const events = await eventReaderRef.current.getEvents(new Date(beginTime), new Date(endTime), devices.map(d => d.id));

      stepRef.current.setEvents(beginTime, endTime, events.map(entry => ({
        context: entry.context,
        triggeredAt: new Date(entry.triggeredAt).getTime(),
        deviceIds: entry.witnesses.map(w => w.id)
      })));

      if (requestsRef.current.some(r => r !== request && r.start <= beginTime && r.end >= endTime)) {
        return;
      }

      if (events.length !== cachedEvents.length) {
        setEvents(devices, tracks, events);
        setData(tracks);
      }
    }
    catch (e: any) {
      console.error("Timeline error:", e);
      setTimelineError(e);
    }
    finally {
      const index = requestsRef.current.indexOf(request);
      index >= 0 && requestsRef.current.splice(index, 1);
      setLoadingCount(count => count - 1);
    }
  }, [devices]);

  const setEvents = (devices: Device[], tracks: TimeLineTrack[], events: AuditEntry[]): void => {
    const deviceIndices: { [key: string]: number[] } = {};
    for (let i = 0; i < devices.length; i++) {
      tracks[i].rows[2].list = {};
      if (!deviceIndices[devices[i].id]) {
        deviceIndices[devices[i].id] = [];
      }
      deviceIndices[devices[i].id].push(i);
    }

    for (const event of events) {
      const witnessSet = new Set<string>(event.witnesses.map(w => w.id));
      const selection = selectedEventIntervalRef.current;
      for (const witness of witnessSet.keys()) {
        const indices = deviceIndices[witness];
        if (indices === undefined) {
          continue;
        }
        for (const i of indices) {
          const eventRow = tracks[i].rows[2];
          const time = new Date(event.triggeredAt).getTime();
          const item = eventRow.list[time];
          if (item) {
            item.eventCount = (item.eventCount ?? 1) + 1;
          } else {
            const isSelected = selection && witnessSet.has(selection.deviceId) && time >= selection.startTime && time <= selection.endTime;
            eventRow.list[time] = {
              isEvent: true,
              isSelected,
              percentage: 100,
              eventCount: 1,
              className: isSelected ? "selected-event" : "event"
            };
          }
        }
      }
    }
  };

  const onGetEventInfo = async (startTime: number, endTime: number, obj: string): Promise<TimeLineEventInfo> => {
    const { data } = await client.query<AuditEntriesWithSnapshotsQuery, AuditEntriesWithSnapshotsQueryVariables>({
      query: AuditEntriesWithSnapshotsDocument,
      variables: { filter: { witnesses: [obj], from: new Date(startTime), to: new Date(endTime) } }
    });
    if (data.auditEntries.length === 0) {
      return { time: new Date(0), html: "", snapshot: "" };
    }

    const entry = data.auditEntries[0];
    const snapshot = entry.snapshots.find(snap => snap.default)?.snapshot ?? "";
    return { time: new Date(entry.triggeredAt), html: entry.message, snapshot };
  };

  const onEventClick = useCallback((startTime: number, endTime: number, obj: string): void => {
    const devIndex = devices.findIndex(dev => dev.id === obj);
    const args: EventClickEventArgs = {
      widgetIndex,
      deviceId: obj,
      deviceName: deviceState.get(devIndex)?.name ?? "",
      startTime: new Date(startTime),
      endTime: new Date(endTime)
    };
    widgetEvent?.publish({ event: CommonWidgetEvent.EventClick, args });
    selectedEventIntervalRef.current = { deviceId: obj, startTime, endTime };
    timelineRef.current?.updateData();
  }, [devices, deviceState]);

  const pauseGroup = (): void => {
    const args: PlayerCommandArgs = {
      objs: group.map(dev => dev.id),
      widgetIndices: group.map(dev => dev.widgetIndex)
    };
    widgetEvent?.publish({ event: PlayerCommand.Pause, args });
  };

  const setTime = async (time: number, isCallback: boolean, byUser: boolean = false): Promise<void> => {
    try {
      await timelineRef.current?.setTime(time, isCallback, byUser);
    }
    catch (e) {
      if (e instanceof TimeLineError) {
        pauseGroup();
      } else {
        throw e;
      }
    }
  };

  const onWidgetEvent = useCallback((args: WidgetEventArgs): void => {
    const { obj, widgetIndex, time } = args.args as PlayerEventArgs;
    const index = devices.findIndex(d => d.id === obj && d.widgetIndex === widgetIndex);
    const deviceStateIndex = getDeviceKeyInState(widgetIndex);
    const isDeviceInState = typeof deviceStateIndex === "number";
    const outOfSync = devices.length > 1;

    switch (args.event) {
      case TimelineEvent.GotoTime:
        const { time: gotoTime } = args.args as TimelineGotoTimeEventArgs;
        setTime(gotoTime.getTime(), true);
        break;

      case PlayerEvent.PausedByUser:
        if (index >= 0) {
          isDeviceInState && updateDeviceState(deviceStateIndex, { stopped: true, outOfSync });
          const device = devices[index];
          const args: PlayerCommandArgs = { objs: [device.id], widgetIndices: [device.widgetIndex] };
          outOfSync && widgetEvent?.publish({ event: PlayerCommand.OutOfSync, args });
        }
        break;

      case PlayerEvent.Play:
        isDeviceInState && updateDeviceState(deviceStateIndex, { stopped: false });
        break;

      case PlayerEvent.Stop:
      case PlayerEvent.Pause:
        isDeviceInState && updateDeviceState(deviceStateIndex, { stopped: true });
        break;

      case PlayerEvent.Frame:
        if (index >= 0 && time) {
          if (isDeviceInState && deviceState.get(deviceStateIndex)?.stopped) {
            return;
          }

          isDeviceInState && updateDeviceState(deviceStateIndex, { playerTime: time });

          // const playerTime = time;
          const playerTime = group.reduce((accumulator, value) => {
            const state = getDeviceState(value);
            const i = devices.findIndex(d => d.id === value.id && d.widgetIndex === value.widgetIndex);
            const devTime = index === i ? time : (state?.playerTime ?? 0);
            return i >= 0 ? Math.max(accumulator, devTime) : accumulator;
          }, 0);

          // const timelineTime = time;
          const timelineTime = group.reduce((accumulator, value) => {
            const state = getDeviceState(value);
            const i = devices.findIndex(d => d.id === value.id && d.widgetIndex === value.widgetIndex);
            return i >= 0 ? Math.max(accumulator, (state?.timelineTime ?? 0)) : accumulator;
          }, 0);

          if (playerTime <= 0) {
            return;
          }

          if (timelineTime > 0 && Math.abs(timelineTime - time) > 5000) {
            return;
          }

          group.forEach(dev => {
            const widgetIndex = devices.find(d => d.id === dev.id && d.widgetIndex === dev.widgetIndex)?.widgetIndex;
            if (typeof widgetIndex === "number") {
              const deviceStateKey = getDeviceKeyInState(widgetIndex);
              typeof deviceStateKey === "number" && updateDeviceState(deviceStateKey, { timelineTime: 0 });
            }
          });

          setTime(playerTime, false);
        }
        break;

      case PlayerEvent.AODDone:
        timelineRef.current?.updateData();
        break;
    }
  }, [devices, deviceState, group]);

  const onTrackSelected = useCallback((index: number): void => {
    if (index >= 0 && index < devices.length) {
      const args: PlayerCommandArgs = { widgetIndices: [devices[index].widgetIndex] };
      widgetEvent?.publish({ event: PlayerCommand.SetSelected, args });
    }
    else {
      widgetEvent?.publish({ event: PlayerCommand.ClearSelected, args: {} });
    }
    updateGroup();
    setSelectedTrack(index);
  }, [devices, deviceState, group]);

  const backToSync = (device: Device): void => {
    const index = getDeviceKeyInState(device.widgetIndex);
    if (typeof index !== "number") {
      return;
    }
    const playing = devices.some((dev, i) => {
      const state = getDeviceState(dev);
      return !state?.outOfSync && !state?.stopped;
    });

    const time = devices.reduce((accumulator, dev, i) => {
      const state = getDeviceState(dev);
      return state && !state.outOfSync ? Math.max(accumulator, (state.timelineTime || state.playerTime) ?? 0) : accumulator;
    }, 0);

    let timelineTime = devices.reduce((accumulator, dev, i) => {
      const state = getDeviceState(dev);
      return state && !state.outOfSync ? Math.max(accumulator, state.timelineTime ?? 0) : accumulator;
    }, 0);

    if (!playing) {
      timelineTime = timelineTime || time; // Set requested play time
    }

    const tooSmallSpeed = -1000;

    const speed = devices.reduce((accumulator, dev, i) => {
      const state = getDeviceState(dev);
      return state && !state.outOfSync ? Math.max(accumulator, state.speed ?? 1) : accumulator;
    }, tooSmallSpeed);

    if (speed > tooSmallSpeed) {
      updateDeviceState(index, { outOfSync: false, timelineTime, speed });
    }
    else {
      updateDeviceState(index, { outOfSync: false, timelineTime });
    }

    const args: PlayerCommandArgs = { objs: [device.id], widgetIndices: [device.widgetIndex] };
    widgetEvent?.publish({ event: PlayerCommand.BackToSync, args });

    if (playing) {
      if (time > 0) {
        args.time = new Date(time);
        widgetEvent?.publish({ event: PlayerCommand.Play, args });
      }

      if (speed > tooSmallSpeed && speed !== (getDeviceState(device)?.speed ?? 1)) {
        args.speed = speed;
        widgetEvent?.publish({ event: PlayerCommand.SetSpeed, args });
      }
    }
  };

  const setGrayedTracks = () => {
    const timeline = timelineRef.current;
    if (!timeline || devices.length === 0) {
      return;
    }

    let grayed: number[] = [];
    const index = timeline.selectedTrack();
    const device = devices[index];
    if (device && getDeviceState(device)?.outOfSync) {
      grayed = devices
        .filter((d, i) => i !== index)
        .map(d => devices.indexOf(d));
    }
    else {
      grayed = devices
        .filter((d, i) => getDeviceState(d)?.outOfSync)
        .map(d => devices.indexOf(d));
    }
    timeline.setGrayedTracks(grayed);
  };

  const updateGroup = () => {
    const newGroup = getGroup();
    setGrayedTracks();
    setGroup(newGroup);
    const time = newGroup.reduce((accumulator, value) => {
      const state = getDeviceState(value);
      return state ? Math.max(accumulator, (state.timelineTime || state.playerTime) ?? 0) : accumulator;
    }, 0);
    if (time) {
      // console.trace();
      setTime(time, false);
    }
    stepRef.current.setDeviceIds(newGroup.map(dev => dev.id));
  };

  const onControlsTimeChanged = useCallback((time: number): void => {
    setTime(time, true, true);
  }, [group]);

  const onTimeChange = useCallback((time: number, changedByUser: boolean): void => {
    currentTimeRef.current = time;

    setTimelineStore({ time });

    if (changedByUser) {
      if (widgetEvent) {
        playerCommand(widgetEvent, devices, PlayerCommand.SetTime, {time: new Date(time), byUser: changedByUser});
      }

      for (const dev of group) {
        const widgetIndex = devices.find(d => d.id === dev.id && d.widgetIndex === dev.widgetIndex)?.widgetIndex;
        const deviceKey = typeof widgetIndex === "number" ? getDeviceKeyInState(widgetIndex) : undefined;
        typeof deviceKey === "number" &&  updateDeviceState(deviceKey, { timelineTime: time });
      }
      pauseGroup();
    }
  }, [group, devices]);

  const onGetTime = (): number => {
    return currentTimeRef.current;
  };

  const onSetSpeed = useCallback((speed: number): void => {
    for (const dev of group) {
      const widgetIndex = devices.find(d => d.id === dev.id && d.widgetIndex === dev.widgetIndex)?.widgetIndex;
      const deviceKey = typeof widgetIndex === "number" ? getDeviceKeyInState(widgetIndex) : undefined;
      typeof deviceKey === "number" &&  updateDeviceState(deviceKey, { speed });
    }
  }, [devices, group]);

  const onGetEvents = useCallback(async (start: number, end: number, limit: number): Promise<Event[]> => {
    const events = await eventReaderRef.current.getEvents(new Date(start), new Date(end), devices.map(dev => dev.id));
    return events.map(entry => ({
      context: entry.context,
      triggeredAt: new Date(entry.triggeredAt).getTime(),
      deviceIds: entry.witnesses.map(w => w.id)
    }));
  }, [devices]);

  const onGetBoundary = async (time: number, forward: boolean, deviceIds: UUID[], streamNumbers: number[]): Promise<number> => {
    const { data } = await client.query<ArchiveBoundaryQuery, ArchiveBoundaryQueryVariables>({
      query: ArchiveBoundaryDocument,
      variables: { ids: deviceIds, streamNumbers, startTime: new Date(time), find: forward ? BoundaryFind.Next : BoundaryFind.Prev }
    });
    return data.archiveBoundary.boundary ? new Date(data.archiveBoundary.boundary).getTime() : 0;
  };

  const getDeviceWithState = (device: Device): DeviceWithState => {
    const state = getDeviceState(device);
    return state ? { ...device, ...state } : device;
  };

  function alignTimeToChunk(time: Date): Date {
    const result = new Date(time);
    result.setUTCSeconds(time.getUTCSeconds() - time.getUTCSeconds() % 30, 0);
    return result;
  }

  function splitTime(startTime: Date, endTime: Date, stepInSeconds: number): Date[] {
    const list: Date[] = [];
    const start = Math.round(startTime.getTime() / 1000);
    const end = Math.round(endTime.getTime() / 1000);
    for (let timeStamp = start; timeStamp < end - stepInSeconds; timeStamp += stepInSeconds) {
      if (timeStamp % stepInSeconds !== 0) {
        continue;
      }
      list.push(new Date(timeStamp * 1000));
    }
    return list;
  }

  function getDeviceKeyInState(widgetIndex: number): number | undefined {
    const device = Array.from(deviceState.entries()).find(ds => ds[1].widgetIndex === widgetIndex);
    if (!device) {
      return undefined;
    }
    return device[0];
  }

  function getDeviceState(device: Device | undefined): DeviceState | undefined {
    const widgetIndex = devices.find(d => d.id === device?.id && d.widgetIndex === device?.widgetIndex)?.widgetIndex;
    const deviceKey = typeof widgetIndex === "number" ? getDeviceKeyInState(widgetIndex) : undefined;
    const state = typeof deviceKey === "number" ? deviceState.get(deviceKey) : undefined;
    return state;
  }

  const hasOutOfSync = Array.from(deviceState.values()).some(s => s.outOfSync);
  const selectedDevice = selectedTrack >= 0 ? getDeviceWithState(devices[selectedTrack]) : undefined;

  const controlsProps: TimelineControlsProps = {
    ...props,
    position: controlsPosition,
    devices: group.map(dev => getDeviceWithState(dev)),
    allDevices: devices.map(dev => getDeviceWithState(dev)),
    archiveStartTime,
    step: stepRef.current,
    selectedDevice,
    onTimeChanged: onControlsTimeChanged,
    onGetTime,
    onSetSpeed
  };

  useEffect(() => {
    return () => {
      setDeviceState(new Map());
      setDevices([]);
      setLoadingCount(0);
      setTimelineError(undefined);
      setSelectedTrack(-1);

      setTimelineStore({ time: 0 });
    };
  }, []);

  return (
    <div className="Timeline">
      { controlsPosition === TimelineControlsPosition.Top &&
      <Segment className="Timeline-Controls">
        <TimelineControls {...controlsProps}/>
      </Segment> }

      <Segment className="Timeline-Timeline">
        <div className="Timeline-TimelineContainer">
          <TimelineControl
            type={TimeLineType.MULTI_TRACK}
            className="Timeline-Control"
            options={{ zoomable: true }}
            hideSelection
            showTime
            onLoad={onTimelineLoad}
            onGetData={(beginTime, endTime, setData) => onTimelineGetData(beginTime, endTime, setData)}
            onGetEventInfo={onGetEventInfo}
            onEventClick={onEventClick}
            onTrackSelected={onTrackSelected}
            onTimeChange={onTimeChange}
          />
          { hasOutOfSync &&
          <div className="Timeline-SyncButtons">
            { devices.map((dev, index) => (
              <a key={dev.id + index} className={!getDeviceState(dev)?.outOfSync ? "hidden" : undefined} onClick={() => backToSync(dev)}>
                {__("SYNC")}
              </a>
            ))}
          </div> }
        </div>
      </Segment>

      { controlsPosition === TimelineControlsPosition.Bottom &&
      <Segment className="Timeline-Controls">
        <TimelineControls {...controlsProps}/>
      </Segment> }
    </div>
  );
};

export default withStore(Timeline);
