import { ApolloClient } from "@apollo/client";
import { AuditEntriesTimeDocument, AuditEntriesTimeQuery, AuditEntriesTimeQueryVariables } from "@generated/graphql";
import { PropType } from "utils";
import { Interval, findGaps, maxInterval } from "./Interval";

export type Event = PropType<AuditEntriesTimeQuery, "auditEntries">[0];

class EventRequest extends Interval {
  data: Event[] = [];
  promise?: Promise<Event[]>;

  constructor(from: Date, to: Date, public witnesses: string[]) {
    super(from, to);
  }

  intersectsWithRequest(from: Date, to: Date, witnesses: string[]): boolean {
    return super.intersectsWith(from, to) && witnesses.every(w => this.witnesses.includes(w));
  }
}

const eventLimit = 100;

class EventReader {
  private requests: EventRequest[] = [];
  private avgEventsPerMinute = 0;

  constructor(private client: ApolloClient<any>) {
  }

  async getEvents(from: Date, to: Date, witnesses: string[]): Promise<Event[]> {
    to = new Date(Math.min(Date.now(), to.getTime()));

    const promises = this.requests.filter(r => r.intersectsWithRequest(from, to, witnesses) && !!r.promise).map(r => r.promise!);
    if (promises.length > 0) {
      await Promise.all(promises);
    }

    let gaps = this.findGaps(from, to, witnesses);
    if (gaps.length === 0) {
      return this.getCachedEvents(from, to, witnesses);
    }

    if (this.avgEventsPerMinute > 0) {
      const maxGap = maxInterval(gaps);
      const minutes = (maxGap.to.getTime() - maxGap.from.getTime()) / (60 * 1000);
      const queryCount = Math.trunc(this.avgEventsPerMinute * minutes / eventLimit) + 1;
      if (gaps.length > queryCount) {
        gaps = [maxGap];
      }
    }

    for (const { from, to } of gaps) {
      const request = new EventRequest(from, to, witnesses);
      this.requests.push(request);
      await this.queryEvents(request, from, to, witnesses);
    }
    return this.getCachedEvents(from, to, witnesses);
  }

  getCachedEvents(from: Date, to: Date, witnesses: string[]): Event[] {
    const events: { [key: string]: Event } = {};
    for (const req of this.requests) {
      for (const event of req.data) {
        const time = new Date(event.triggeredAt).getTime();
        if (time >= from.getTime() && time <= to.getTime() && event.witnesses.some(w => witnesses.includes(w.id))) {
          events[event.context] = event;
        }
      }
    }
    const result: Event[] = [];
    for (const key in events) {
      result.push(events[key]);
    }
    return result;
  }

  private queryEvents(request: EventRequest, from: Date, to: Date, witnesses: string[]): Promise<Event[]> {
    const promise = new Promise<Event[]>(async (resolve, reject) => {
      try {
        let eventCount = 0;
        let end = to;
        do {
          const { data } = await this.client.query<AuditEntriesTimeQuery, AuditEntriesTimeQueryVariables>({
            query: AuditEntriesTimeDocument,
            variables: { filter: { from, to: end, witnesses, limit: eventLimit, categories: ["47", "61"] }}
          });

          const auditEntries = [...data.auditEntries];
          const events = auditEntries.sort((a, b) => new Date(a.triggeredAt).getTime() - new Date(b.triggeredAt).getTime());
          request.data = request.data.concat(events);
          eventCount = events.length;

          if (eventCount > 0) {
            end = new Date(events[0].triggeredAt);
          }
        }
        while (eventCount === eventLimit);

        const eventsPerMinute = request.data.length / ((to.getTime() - from.getTime()) / (60 * 1000));
        if (eventsPerMinute > 0) {
          this.avgEventsPerMinute = this.avgEventsPerMinute === 0 ? eventsPerMinute : (this.avgEventsPerMinute + eventsPerMinute) / 2;
        }

        resolve(request.data);
      }
      catch (e) {
        const index = this.requests.indexOf(request);
        index >= 0 && this.requests.splice(index, 1);
        reject(e);
      }
      finally {
        request.promise = undefined;
      }
    });
    request.promise = promise;
    return promise;
  }

  private findGaps(from: Date, to: Date, witnesses: string[]): Interval[] {
    const requests = this.requests.filter(r => witnesses.every(w => r.witnesses.includes(w)) && !r.promise);
    return findGaps(from, to, requests, 2000);
  }
}

export default EventReader;
