import {ApolloClient, ApolloQueryResult} from "@apollo/client";
import { ArchiveCoverageDocument, ArchiveCoverageQuery, ArchiveCoverageQueryVariables, AodRequestStatus } from "@generated/graphql";
import { PropType } from "utils";
import { Interval, findGaps, maxInterval } from "./Interval";
import {TimeLineDataList, UUID} from "@solid/types";
import { getOfflineCoverage } from "@solid/libs/avatar";
import {clone} from "@solid/libs";

type CoverageQuery = PropType<ArchiveCoverageQuery, "archiveCoverage">;
export type Coverage = PropType<CoverageQuery, "coverage">[0];
export type PartialCoverage = PropType<CoverageQuery, "partialCoverage">[0];
export type CoverageTTL = PropType<CoverageQuery, "ttl">[0];
export type Aod = PropType<ArchiveCoverageQuery, "aodGet">[0];

export type CoverageData = {
  coverage: Coverage[],
  partialCoverage: PartialCoverage[],
  coverageTtl: CoverageTTL[],
  aod: Aod[],
  offlineCoverage: { [deviceId in UUID]: TimeLineDataList }
};

const cacheTTL = 1000;

class CoverageRequest extends Interval {
  promise?: Promise<void>;
  completedAt?: Date;

  constructor(from: Date, to: Date, public deviceIds: UUID[], public data: CoverageData = { coverage: [], partialCoverage: [], coverageTtl: [], aod: [], offlineCoverage: {} }) {
    super(from, to);
  }

  intersectsWithRequest(from: Date, to: Date, deviceIds: UUID[]): boolean {
    return super.intersectsWith(from, to) && deviceIds.every(id => this.deviceIds.includes(id));
  }
}

type CoverageMap = {
  coverage: { [deviceId in UUID]: Coverage },
  partialCoverage: { [deviceId in UUID]: PartialCoverage },
  coverageTtl: { [deviceId in UUID]: CoverageTTL },
  aod: { [deviceId in UUID]: Aod },
  offlineCoverage: { [deviceId in UUID]: TimeLineDataList }
};

class CoverageReader {
  private requests: CoverageRequest[] = [];

  constructor(private client: ApolloClient<any>) {
  }

  async getCoverage(from: Date, to: Date, deviceIds: UUID[]): Promise<CoverageData> {
    to = new Date(Math.min(Date.now(), to.getTime()));

    this.requests = this.requests.filter(r => !r.completedAt || (Date.now() - r.completedAt.getTime() < cacheTTL));

    const promises = this.requests.filter(r => r.intersectsWithRequest(from, to, deviceIds) && !!r.promise).map(r => r.promise!);
    if (promises.length > 0) {
      await Promise.all(promises);
    }

    const coverage: CoverageMap = { coverage: {}, partialCoverage: {}, coverageTtl: {}, aod: {}, offlineCoverage: {} };
    this.addCachedCoverage(coverage, from, to, deviceIds);

    let gaps = this.findGaps(from, to, deviceIds);
    if (gaps.length === 0) {
      return this.getCoverageFromMap(coverage);
    }
    if (gaps.length > 2) {
      gaps = [maxInterval(gaps)];
    }

    for (const gap of gaps) {
      const request = new CoverageRequest(gap.from, gap.to, deviceIds);
      this.requests.push(request);
      await this.queryCoverage(request);
      this.addRequestCoverage(coverage, request, from, to);
    }

    return this.getCoverageFromMap(coverage);
  }

  private addCachedCoverage(map: CoverageMap, from: Date, to: Date, deviceIds: UUID[]): void {
    const requests = this.requests.filter(r => deviceIds.every(id => r.deviceIds.includes(id)));
    for (const request of requests) {
      this.addRequestCoverage(map, request, from, to);
    }
  }

  private addRequestCoverage(map: CoverageMap, request: CoverageRequest, from: Date, to: Date): void {
    for (const coverage of request.data.coverage) {
      const time = coverage.time * 1000;
      if (time >= from.getTime() && time <= to.getTime()) {
        map.coverage[this.getCoverageKey(coverage)] = coverage;
      }
    }

    for (const coverage of request.data.partialCoverage) {
      const time = coverage.time * 1000;
      if (time >= from.getTime() && time <= to.getTime()) {
        map.partialCoverage[this.getCoverageKey(coverage)] = coverage;
      }
    }

    for (const coverageTtl of request.data.coverageTtl) {
      map.coverageTtl[this.getCoverageTtlKey(coverageTtl)] = coverageTtl;
    }

    for (const aod of request.data.aod) {
      const interval = new Interval(new Date(aod.startTime), new Date(aod.endTime));
      if (interval.intersectsWith(from, to)) {
        map.aod[this.getAodKey(aod)] = aod;
      }
    }

    map.offlineCoverage = request.data.offlineCoverage;
  }

  private getCoverageFromMap(map: CoverageMap): CoverageData {
    const data: CoverageData = {
      coverage: Object.keys(map.coverage).map(key => map.coverage[key]),
      partialCoverage: Object.keys(map.partialCoverage).map(key => map.partialCoverage[key]),
      coverageTtl: Object.keys(map.coverageTtl).map(key => map.coverageTtl[key]),
      aod: Object.keys(map.aod).map(key => map.aod[key]),
      offlineCoverage: clone(map.offlineCoverage)
    };
    return data;
  }

  private queryCoverage(request: CoverageRequest): Promise<void> {
    const promise = new Promise<void>(async (resolve, reject) => {
      try {
        const promiseList: Promise<ApolloQueryResult<ArchiveCoverageQuery> | TimeLineDataList>[] = [
          this.client.query<ArchiveCoverageQuery, ArchiveCoverageQueryVariables>({
            query: ArchiveCoverageDocument,
            variables: { ids: request.deviceIds, startTime: request.from, endTime: request.to },
            fetchPolicy: "network-only"
          })
        ];
        for (const deviceId of request.deviceIds) {
          promiseList.push(getOfflineCoverage(deviceId, request.from.getTime(), request.to.getTime(), process.env.DATA_DIR ?? ""));
        }
        const result = await Promise.all(promiseList);

        const {data} = result[0] as ApolloQueryResult<ArchiveCoverageQuery>;

        const offlineCoverage = {};
        for (let i = 0; i < request.deviceIds.length; i++) {
          const deviceId = request.deviceIds[i];
          offlineCoverage[deviceId] = result[i + 1];
        }

        request.data = {
          coverage: data.archiveCoverage.coverage,
          partialCoverage: data.archiveCoverage.partialCoverage,
          coverageTtl: data.archiveCoverage.ttl,
          aod: data.aodGet.filter(aod => ![AodRequestStatus.Canceled, AodRequestStatus.Failed].includes(aod.status)),
          offlineCoverage
        };
        resolve();
      }
      catch (e) {
        const index = this.requests.indexOf(request);
        index >= 0 && this.requests.splice(index, 1);
        reject(e);
      }
      finally {
        request.promise = undefined;
        request.completedAt = new Date();
      }
    });
    request.promise = promise;
    return promise;
  }

  private findGaps(from: Date, to: Date, deviceIds: UUID[]): Interval[] {
    const requests = this.requests.filter(request => deviceIds.every(id => request.deviceIds.includes(id)) && !request.promise);
    return findGaps(from, to, requests, 2000);
  }

  private getCoverageKey(coverage: Coverage | PartialCoverage): string {
    return `${coverage.deviceId}_${coverage.time}_${coverage.streamNumber}`;
  }

  private getCoverageTtlKey(coverageTtl: CoverageTTL): string {
    return `${coverageTtl.deviceId}_${coverageTtl.ttl}`;
  }

  private getAodKey(aod: Aod): string {
    return `${aod.deviceId}_${aod.startTime}_${aod.endTime}_${aod.streamNumber}`;
  }
}

export default CoverageReader;
