import React, { useState, useEffect, useMemo } from "react";
import { List, Header } from "semantic-ui-react";
import { Link, useNavigate } from "react-router-dom";
import { useStore } from "@core/store";
import { SystemStatus as SystemStatusEnum, SystemStatusQuery, SystemStatusReason } from "@generated/graphql";
import { ApolloError } from "@apollo/client";
import WithQueryStatus from "components/WithQueryStatus";
import {__} from "@solid/libs/i18n";
import {isElectron} from "@solid/libs/utils";


import "./style.css";

type SystemStatusProps = {
  data: SystemStatusQuery | undefined;
  loading: boolean;
  error: ApolloError | undefined;
};

export const searchText = "Missing local disk space for";

const SystemStatus = ({ data, loading, error }: SystemStatusProps) => {
  const navigate = useNavigate();
  const { store: { session: { isLoggedIn } } } = useStore();
  const [timeDiff, setTimeDiff] = useState(0);
  const [goNext, setGoNext] = useState(false);

  const systemReasons: SystemStatusReason[] = useMemo(() => {
    return  data
      ? data.systemStatus.reasons
        .filter(reason => reason.object?.type !== "camera" && !reason.message.startsWith(searchText))
      : [];
  }, [data]);

  useEffect(() => {
    if (data) {
      let diff = 0;
      if (data.serverTime) {
        diff = Math.round(Math.abs(new Date(data.serverTime).getTime() - Date.now()) / 1000);
        if (diff <= 5) {
          diff = 0;
        }
        setTimeDiff(diff);
      }
      setGoNext(data.systemStatus && data.systemStatus.status === SystemStatusEnum.Online &&
        !data.systemStatus.description && diff === 0);
    }
  }, [data]);

  useEffect(() => {
    if (goNext) {
      next();
    }
  }, [goNext]);

  function next() {
    if (isElectron()) {
      navigate("/guilist");
    }
  }

  return (
    <div className="SystemStatus-Root">
      <WithQueryStatus loading={loading} error={error} noData={!data}>
        { isLoggedIn &&
        <div className="system-status">
          <Header className="SystemStatus-Header" size="huge" textAlign="center">{__("System Check")}</Header>
          { !!data && !!data.systemStatus &&
            <div className="SystemStatus-Info">
              <List>
                { !!data.systemStatus.description &&
                  <List.Item>
                    <List.Icon name="warning circle" color="red" size="big" />
                    <List.Content>{data.systemStatus.description}</List.Content>
                    {systemReasons &&
                    <List bulleted>
                      {systemReasons.map((reason, index) => {
                        if (reason.object) {
                          let href = "/main";
                          if (reason.object.type === "camera") {
                            href = `/view/camera/edit/${reason.object.id}`;
                          } else
                          if (reason.object.type === "avatar") {
                            href = `/view/link/edit/${reason.object.id}/properties`;
                          }
                          return <List.Item key={reason.message + index}><Link to={href}>{reason.message}</Link></List.Item>;
                        }
                        return <List.Item key={reason.message + index}>{reason.message}</List.Item>;
                      })}
                    </List>}
                  </List.Item>}
                { !!timeDiff && timeDiff > 0 &&
                  <List.Item>
                    {__("WARNING: Time shift between your client station and server was identified and it is {{time}} seconds.", {time: timeDiff})}
                    <br/>{__("Please make sure to synchronize your time as this may affect your user experience.")}
                  </List.Item> }
              </List>
            </div> }
        </div> }
      </WithQueryStatus>
    </div>
  );
};

export default SystemStatus;
