import React, { useRef, useEffect } from "react";
import { Button, Header, Icon, Modal, Segment } from "semantic-ui-react";
import { __ } from "@solid/libs";
import { AutoForm, AutoLayout, FormSchema } from "components/AutoForm";
import { useValidateEmailMutation } from "@generated/graphql";

import "./style.css";

const validationEmailSchema: FormSchema = [
  {
    name: "validationEmailFrom",
    label: __("From"),
    required: true,
  },
  {
    name: "validationEmailTo",
    label: __("To"),
    required: true
  },
];

const validationEmailValues = {
  validationEmailFrom: "",
  validationEmailTo: ""
};

type EmailValidationProps = {
  dialogOpen: boolean;
  onClose: (sended: boolean, email?: string) => void;
};

const EmailValidation = ({ dialogOpen, onClose }: EmailValidationProps) => {
  const [validateEmail, { data, loading }] = useValidateEmailMutation();
  const validationEmailFormRef = useRef<AutoForm>(null);

  const validate = () => {
    if (!validationEmailFormRef.current) {
      return;
    }

    const values = validationEmailFormRef.current.getValues();
    validateEmail({ variables: {
      addresses: { from: values["validationEmailFrom"], to: values["validationEmailTo"] }
    }});
  };

  useEffect(() => {
    if (data?.validateEmail) {
      const email = validationEmailFormRef?.current?.getValues()["validationEmailTo"];
      onClose(true, email || "");
    }
  }, [data]);

  return (
    <Modal open={dialogOpen} onClose={() => onClose(false)} className="EmailValidation-Modal">
      <Header>{__("Validate Email")}</Header>
      <Modal.Content>
        <Segment loading={loading} className="EmailValidation-Segment">
          <AutoForm
            ref={validationEmailFormRef}
            schema={validationEmailSchema}
            values={validationEmailValues}
          >
            <AutoLayout/>
          </AutoForm>
        </Segment>
      </Modal.Content>
      <Modal.Actions>
        <Button onClick={() => onClose(false)}>
          <Icon name="cancel"/>{__("Cancel")}
        </Button>
        <Button positive onClick={() => validate()}>
          <Icon name="check"/>{__("Validate")}
        </Button>
      </Modal.Actions>
    </Modal>
  );
};

export default EmailValidation;
